/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

// import QtQuick 1.0 // to target S60 5th Edition or Maemo 5
import QtQuick 1.1
import com.nokia.meego 1.0
import 'UIConstants.js' as UI

import 'HelpDialog.js' as HelpHelper

Page {
    id: editorTabPage
    tools: editorTabTool

    property string currentPriority

    property bool showHelp: false
    property string helpMessage: qsTr(
                                     "<p>Select font family and edit priorities.</p>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-add\">: Add fonts to the target <i>Priority</i> in config file.</p>" +
                                     "<hr/>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-up\">: Move up selected font.</p>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-delete\">: Remove selected font from the its <i>Priority</i> in config file.  <i>Note: The fonts are NOT uninstalled</i>.</p>" +
                                     "<p><img src=\"image://theme/icon-m-toolbar-down\">: Move down selected font.</p>" +
                                     "<hr/>" +
                                     "<h2>Priorities</h2><p>" +
                                     "&nbsp;&nbsp;<b>Higher</b>: These fonts have a higher priority.  ('prepend' fonts in fontconfig)<br/>" +
                                     "&nbsp;&nbsp;<b>Normal</b>: These fonts are usally used as default.  System settings may be given priority than these.  ('prefer' fonts in fontconfig)<br/>" +
                                     "&nbsp;&nbsp;<b>Lowser</b>: These fonts have a lower priority.  ('accept' fonts in fontconfig)<br/>" +
                                     "<ul><li>Please check <a href=\"http://www.freedesktop.org/software/fontconfig/fontconfig-user.html\">fontconfig</a> for more details.</li></ul>" +
                                     "<hr/>" +
                                     "<p><b>Import System Settings</b>: Import settings from system(/etc/fonts/local.conf).</p>" +
                                     "<hr/>" +
                                     "<h2>Menu(<img src=\"image://theme/icon-m-toolbar-view-menu\">)</h2>" +
                                     "<p><b>Remove current font config</b>(Menu): Remove current user's font config file.</p>" +
                                     "<p><b>View current font config</b>(Menu): View current user's font config file.</p>"
                                     )

    property Item selectionDialog: null

    property int currentIndex: 0
    ListModel {
        id: familyModel
        ListElement { family: "sans-serif";         name: QT_TR_NOOP("Sans Serif") }
        ListElement { family: "serif";              name: QT_TR_NOOP("Serif") }
        ListElement { family: "monospace";          name: QT_TR_NOOP("Monospace") }
        ListElement { family: "Nokia Pure Text";    name: QT_TR_NOOP("System") }
    }

    PageHeader {
        id: pageHeader
        anchors.top: parent.top
        anchors.left: parent.left
        anchors.right: parent.right
        text: qsTr("Edit Fonts Config: %1").arg(confEditor.familyName)
        enabled: !showHelp
    }

    Item {
        id: contents
        anchors.top: pageHeader.bottom
        anchors.bottom: parent.bottom
        anchors.left: parent.left
        anchors.right: parent.right


        opacity: enabled ? 1.0 : 0.5

        FontsConfEditor {
            id: confEditor
            anchors.top: parent.top
            anchors.bottom: bottomButtons.top
            anchors.left: parent.left
            anchors.right: parent.right

            family: familyModel.get(currentIndex).family
            familyName: familyModel.get(currentIndex).name
            editorController: controller.editorController(family)
        }

        BottomButtons {
            id: bottomButtons
            anchors.bottom: parent.bottom
            anchors.left: parent.left
            anchors.right: parent.right
            targetTab: confEditor
        }

    }

    ToolBarLayout {
        id: editorTabTool
        visible: true
        opacity: enabled ? 1.0 : 0.5
        ToolIcon {
            platformIconId: "toolbar-back"
            onClicked: {
                if (editMenu.status !== DialogStatus.Closed)
                    editMenu.close()
                pageStack.pop()
            }
        }

        Button {
            text: confEditor.familyName

            Image {
                id: openImage
                anchors.right: parent.right
                anchors.rightMargin: UI.DEFAULT_MARGIN
                anchors.verticalCenter: parent.verticalCenter
                source: "image://theme/icon-m-common-expand"
            }
            onClicked: {
                selectCurrentFamilyDialog.open()
            }
        }

        ToolIcon {
            platformIconId: "toolbar-view-menu"
            anchors.right: (parent === undefined) ? undefined : parent.right
            onClicked: {
                if (!mainPage.enabled)
                    return
                if (editMenu.status === DialogStatus.Closed)
                    editMenu.open()
                else
                    editMenu.close()
            }
        }
    }

    Menu {
        id: editMenu
        visualParent: pageStack
        MenuLayout {
            MenuItem {
                text: qsTr("Remove current fonts config");
                enabled: !controller.isEmptyFontsConf
                onClicked: {
                    editMenu.close()
                    openRemoveConfirmDialog()
                }
            }
            MenuItem {
                text: qsTr("View current fonts config");
                enabled: controller.localFontsConfExists
                onClicked: {
                    editMenu.close()
                    var component = Qt.createComponent("FontsConfViewPage.qml")
                    if (component.status === Component.Error)
                        console.log("Error: " + component.errorString())
                    pageStack.push(component, { "text": controller.localFontsConf } )
                }
            }
            MenuItem {
                text: qsTr("Help")
                onClicked: {
                    showHelp = true
                    HelpHelper.openHelpDialog(editorTabPage, contents, helpMessage, closeHelp)
                }
            }
        }
    }

    function closeHelp()
    {
        showHelp = false
    }

    function openRemoveConfirmDialog()
    {
        queryDialog.titleText = qsTr("Remove it?")
        queryDialog.message = qsTr("Existing fonts config %1 will be removed.  Are you sure?").arg(controller.localFontsConfPath)
        queryDialog.acceptButtonText = qsTr("OK")
        queryDialog.rejectButtonText = qsTr("Cancel")
        queryDialog.accepted.connect(resetLocalFontsConf)
        queryDialog.open()
    }

    function resetLocalFontsConf()
    {
        queryDialog.accepted.disconnect(resetLocalFontsConf)
        controller.resetLocalFontsConf()
    }

    function openSelectInstallFamilyDialog(priority)
    {
        deleteSelectInstallFamilyDialog()
        var dialogComponent = Qt.createComponent("SelectInstallFamilyDialog.qml")
        if (dialogComponent.status === Component.Error)
            console.log("Error: " + dialogComponent.errorString())
        controller.syncInstallableFamilyFor(confEditor.family)
        editorTabPage.currentPriority = priority
        selectionDialog = dialogComponent.createObject(editorTabPage, { "editor": confEditor } )
        selectionDialog.selectedIndexes = []
        selectionDialog.accepted.connect(deleteSelectInstallFamilyDialog)
        selectionDialog.rejected.connect(deleteSelectInstallFamilyDialog)
        selectionDialog.open()
    }

    function deleteSelectInstallFamilyDialog()
    {
        if (selectionDialog)
            selectionDialog.destroy()
        selectionDialog = null
    }

    SelectionDialog {
        id: selectCurrentFamilyDialog
        model: familyModel
        selectedIndex: currentIndex
        titleText: qsTr("Select Family to edit")
        onAccepted: {
            currentIndex = selectedIndex
        }
    }

    states: [
        State {
            name: "showHelp"
            when: showHelp
            PropertyChanges {
                target: contents
                enabled: false
            }
            PropertyChanges {
                target: editorTabTool
                enabled: false
            }
        }
    ]

}
