/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "installedfontinfo.h"

#include "fontsconfigproperties.h"

#include <QTextStream>

InstalledFontInfo::InstalledFontInfo(const QByteArray buf, const QString &localFontPath, QObject *parent) :
    QObject(parent), mSystemFont(true), mLocale("en"), mFontProperty(0)
{
    mValid = analyze(buf, localFontPath);
}

bool InstalledFontInfo::isValid()
{
    return mValid;
}

QStringList InstalledFontInfo::family() const
{
    return mFamily;
}

QStringList InstalledFontInfo::familylang() const
{
    return mFamilylang;
}

QStringList InstalledFontInfo::style() const
{
    return mStyle;
}

QStringList InstalledFontInfo::stylelang() const
{
    return mStylelang;
}

QStringList InstalledFontInfo::fullname() const
{
    return mFullname;
}

QStringList InstalledFontInfo::fullnamelang() const
{
    return mFullnamelang;
}

QString InstalledFontInfo::foundry() const
{
    return mFoundry;
}

QString InstalledFontInfo::file() const
{
    return mFile;
}

QStringList InstalledFontInfo::lang() const
{
    return mLang;
}

bool InstalledFontInfo::outline() const
{
    return mOutline;
}

bool InstalledFontInfo::scalable() const
{
    return mScalable;
}

int InstalledFontInfo::slant() const
{
    return mSlant;
}

int InstalledFontInfo::weight() const
{
    return mWeight;
}

int InstalledFontInfo::width() const
{
    return mWidth;
}

QString InstalledFontInfo::localefamily() const
{
    return localeString(mFamily, mFamilylang, mLocale);
}

QString InstalledFontInfo::localestyle() const
{
    return localeString(mStyle, mStylelang, mLocale);
}

QString InstalledFontInfo::localefullname() const
{
    return localeString(mFullname, mFullnamelang, mLocale);
}

QString InstalledFontInfo::enfamily() const
{
    return localeString(mFamily, mFamilylang, QLatin1String("en"));
}

QString InstalledFontInfo::enstyle() const
{
    return localeString(mStyle, mStylelang, QLatin1String("en"));
}

QString InstalledFontInfo::enfullname() const
{
    return localeString(mFullname, mFullnamelang, QLatin1String("en"));
}

bool InstalledFontInfo::systemFont() const
{
    return mSystemFont;
}

QString InstalledFontInfo::localeString(const QStringList &strlist, const QStringList &langlist, const QString &lang) const
{
    int idx = langlist.indexOf(lang);
    if (idx < 0)
        idx = langlist.indexOf(QLatin1String("en"));
    if (idx < 0)
        idx = 0;
    Q_ASSERT(idx < strlist.count());
    return strlist.at(idx);
}

FontsConfigProperties *InstalledFontInfo::fontProperty() const
{
    return mFontProperty;
}

void InstalledFontInfo::setFontProperty(FontsConfigProperties *prop)
{
    mFontProperty = prop;
    if (mFontProperty->parent() == 0)
        mFontProperty->setParent(this);
}

bool InstalledFontInfo::compare(const InstalledFontInfo *info1, InstalledFontInfo *info2)
{
    QString key1 = info1->localefamily() + ":" + info1->localefullname();
    QString key2 = info2->localefamily() + ":" + info2->localefullname();
    return (key1 < key2);
}

bool InstalledFontInfo::analyze(const QByteArray &buf, const QString &localFontPath)
{
    static QRegExp headerRegexp("^Pattern has \\d+ elts \\(size \\d+\\)$");
    static QRegExp splitter(":\\s+");
    int idx = -1;

    QTextStream ts(buf, QIODevice::ReadOnly);
    QString linebuf = ts.readLine();
    if (!headerRegexp.exactMatch(linebuf))
        return false;

    while (!ts.atEnd()) {
        QString linebuf = ts.readLine().trimmed();
        if ((idx = linebuf.indexOf(splitter)) > 0){
            QString key = linebuf.left(idx);
            QString val = linebuf.mid(idx + splitter.matchedLength());
            if (key == "family") {
                mFamily = toStringList(val);
            } else if (key == "familylang") {
                mFamilylang = toStringList(val);
            } else if (key == "style") {
                mStyle = toStringList(val);
            } else if (key == "stylelang") {
                mStylelang = toStringList(val);
            } else if (key == "fullname") {
                mFullname = toStringList(val);
            } else if (key == "fullnamelang") {
                mFullnamelang = toStringList(val);
            } else if (key == "slant") {
                mSlant = toInt(val);
            } else if (key == "weight") {
                mWeight = toInt(val);
            } else if (key == "width") {
                mWidth = toInt(val);
            } else if (key == "foundry") {
                mFoundry = toString(val);
            } else if (key == "file") {
                mFile = toString(val);
                if (mFile.startsWith(localFontPath))
                    mSystemFont = false;
            } else if (key == "outline") {
                mOutline = toBool(val);
            } else if (key == "scalable") {
                mScalable = toBool(val);
            } else if (key == "lang") {
                int index = val.indexOf(QLatin1String("(s)"));
                Q_ASSERT(index >= 0);
                QString langs = val.left(index);
                mLang = langs.split(QLatin1Char('|'));
            }
        }
    }

    emit detailsChanged();

    return true;
}

QString InstalledFontInfo::toString(const QString &value)
{
    static QRegExp reg("\"([^\"]+)\"\\(s\\)");
    QString str;
    if (reg.indexIn(value) == 0)
        str = reg.cap(1);
    return str;
}

QStringList InstalledFontInfo::toStringList(const QString &value)
{
    static QRegExp reg("\"([^\"]+)\"\\(s\\)");
    QStringList list;
    int index = 0;
    while ((index = reg.indexIn(value, index)) >= 0) {
        list << reg.cap(1);
        index += reg.matchedLength();
    }
    return list;
}

int InstalledFontInfo::toInt(const QString &value)
{
    static QRegExp reg("(\\d+)\\(i\\)\\(s\\)");
    int val = 0;
    if (reg.indexIn(value) == 0)
        val = reg.cap(1).toInt();
    return val;
}

bool InstalledFontInfo::toBool(const QString &value)
{
    static QLatin1String fcTrue("FcTrue(s)");
    if (value == fcTrue)
        return true;
    return false;
}

QString InstalledFontInfo::locale() const
{
    return mLocale;
}

void InstalledFontInfo::setLocale(const QString &loc)
{
    mLocale = loc;
}

