/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontsconfigproperties.h"
#include "fontconfigdefs.h"

FontsConfigProperties::FontsConfigProperties(const QString &family, QObject *parent) :
    QObject(parent), mFamily(family), mEmbeddedBitmap(Default), mHinting(Default)
{
    Q_ASSERT(!family.isEmpty());
    connect(this, SIGNAL(configValueChanged(QString)), SLOT(checkConfigValueChanged(QString)));
}

QString FontsConfigProperties::family() const
{
    return mFamily;
}

QStringList FontsConfigProperties::preferFamilies() const
{
    return mPreferFamilies;
}

void FontsConfigProperties::addPreferFamily(const QString &family)
{
    if (mPreferFamilies.contains(family))
        return;
    mPreferFamilies << family;
    emit preferFamiliesChanged();
}

void FontsConfigProperties::removePreferFamily(const QString &family)
{
    if (!mPreferFamilies.contains(family))
        return;
    mPreferFamilies.removeOne(family);
    emit preferFamiliesChanged();
}

bool FontsConfigProperties::prefer(const QString &family) const
{
    return mPreferFamilies.contains(family);
}

FontsConfigProperties::ConfigValue FontsConfigProperties::configValue(const QString &key) const
{
    return mConfigValues.value(key, Default);
}

void FontsConfigProperties::setConfigValue(const QString &key, FontsConfigProperties::ConfigValue config)
{
    if (!mConfigValues.contains(key) || configValue(key) != config) {
        mConfigValues.insert(key, config);
        emit configValueChanged(key);
    }
}

FontsConfigProperties::ConfigValue FontsConfigProperties::embeddedBitmap() const
{
    return configValue(EMBEDDEDBITMAP_DEF);
}

void FontsConfigProperties::setEmbeddedBitmap(FontsConfigProperties::ConfigValue config)
{
    setConfigValue(EMBEDDEDBITMAP_DEF, config);
}

FontsConfigProperties::ConfigValue FontsConfigProperties::hinting() const
{
    return configValue(HINTING_DEF);
}

void FontsConfigProperties::setHinting(FontsConfigProperties::ConfigValue config)
{
    setConfigValue(HINTING_DEF, config);
}

FontsConfigProperties::ConfigValue FontsConfigProperties::antialias() const
{
    return configValue(ANTIALIAS_DEF);
}

void FontsConfigProperties::setAntialias(FontsConfigProperties::ConfigValue config)
{
    setConfigValue(ANTIALIAS_DEF, config);
}

FontsConfigProperties::ConfigValue FontsConfigProperties::autohint() const
{
    return configValue(AUTOHINT_DEF);
}

void FontsConfigProperties::setAutohint(FontsConfigProperties::ConfigValue config)
{
    setConfigValue(AUTOHINT_DEF, config);
}

void FontsConfigProperties::checkConfigValueChanged(const QString &key)
{
    if (key == EMBEDDEDBITMAP_DEF)
        emit embeddedBitmapChanged();
    else if (key == HINTING_DEF)
        emit hintingChanged();
    else if (key == ANTIALIAS_DEF)
        emit antialiasChanged();
    else if (key == AUTOHINT_DEF)
        emit autohintChanged();
}

