/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontsconfelement.h"

#include "fontconfigdefs.h"

#include <QXmlStreamReader>

FontsConfElement::FontsConfElement(const QString &type, QObject *parent) :
    QObject(parent), mType(type)
{
}

QString FontsConfElement::type() const
{
    return mType;
}

void FontsConfElement::clear()
{
    mAttributes.clear();
    mChildElements.clear();
}

bool FontsConfElement::parse(QXmlStreamReader &xml)
{
    if (xml.atEnd() || xml.error() != QXmlStreamReader::NoError)
        return false;

    bool check = true;
    if (!xml.isStartElement()) {
        check = xml.readNextStartElement();
        if (!check)
            return false;
    }

    mAttributes = xml.attributes();

    while (!xml.atEnd()) {
        QXmlStreamReader::TokenType t = xml.readNext();
        if (t == QXmlStreamReader::Characters) {
            mText = xml.text().toString();
        } else if (t == QXmlStreamReader::StartElement) {
            FontsConfElementPointer elem(new FontsConfElement(xml.name().toString(), this));
            check = elem->parse(xml);
            if (!check)
                return false;
            addChildElement(elem);
        } else if (t == QXmlStreamReader::EndElement)
            return true;
    }

    return false;
}

void FontsConfElement::save(QXmlStreamWriter &xml)
{
    if (mChildElements.count() == 0) {
        if (!mText.isEmpty())
            xml.writeTextElement(mType, mText);
        else
            xml.writeEmptyElement(mType);
        xml.writeAttributes(mAttributes);
    } else {
        xml.writeStartElement(mType);
        xml.writeAttributes(mAttributes);
        if (!mText.isEmpty())
            xml.writeCharacters(mText);
        foreach (FontsConfElementPointer elem, mChildElements) {
            elem->save(xml);
        }
        xml.writeEndElement();
    }
}

QXmlStreamAttributes FontsConfElement::attributes() const
{
    return mAttributes;
}

bool FontsConfElement::hasAttribute(const QString &key) const
{
    return mAttributes.hasAttribute(key);
}

QString FontsConfElement::value(const QString &key) const
{
    return mAttributes.value(key).toString();
}

void FontsConfElement::setAttribute(const QString &key, const QString &val)
{
    mAttributes.append(key, val);
}

int FontsConfElement::count() const
{
    return mChildElements.count();
}

FontsConfElementPointer FontsConfElement::childElementAt(int index) const
{
    return mChildElements.at(index);
}

FontsConfElementList FontsConfElement::findChildrenElements(const QString &type) const
{
    FontsConfElementList list;
    foreach (FontsConfElementPointer elem, mChildElements) {
        if (elem->type() == type)
            list << elem;
    }
    return list;
}

int FontsConfElement::indexOf(const QString &type, int from) const
{
    if (from < 0 || from >= mChildElements.count())
        return -1;

    int index = from;
    while (index < mChildElements.count()) {
        if (mChildElements.at(index)->type() == type)
            return index;
        index++;
    }
    return -1;
}

FontsConfElementPointer FontsConfElement::childElementOf(const QString &type, int from) const
{
    int index = indexOf(type, from);
    if (index < 0)
        return FontsConfElementPointer();
    return childElementAt(index);
}

void FontsConfElement::addChildElement(FontsConfElementPointer elem)
{
    if (!elem || mChildElements.contains(elem))
        return;
    elem->setParent(this);
    mChildElements.append(elem);
}

void FontsConfElement::insertChildElement(FontsConfElementPointer elem, int index)
{
    if (!elem || mChildElements.contains(elem))
        return;
    elem->setParent(this);
    mChildElements.insert(index, elem);
}

void FontsConfElement::removeAt(int index)
{
    mChildElements.removeAt(index);
}

void FontsConfElement::removeOne(FontsConfElementPointer elem)
{
    mChildElements.removeOne(elem);
}

QString FontsConfElement::text() const
{
    return mText;
}

void FontsConfElement::setText(const QString &str)
{
    mText = str;
}

QString FontsConfElement::readAttribute(QXmlStreamReader &xml, const QString &attr, const QString &defaultValue)
{
    if (xml.attributes().hasAttribute(attr))
        return xml.attributes().value(attr).toString();
    return defaultValue;
}

QVariant FontsConfElement::readValue(QXmlStreamReader &xml)
{

    bool check = false;
    if (xml.isStartElement())
        check = true;
    else
        check = xml.readNextStartElement();
    if (!check)
        return QVariant();
    QString elemName(xml.name().toString());
    if (elemName != BOOL_DEF || elemName != STRING_DEF || elemName != INT_DEF || elemName != DOUBLE_DEF) {
        return QVariant();
    }

    QVariant value;
    QXmlStreamReader::TokenType t = xml.tokenType();
    while (t != QXmlStreamReader::Characters || t != QXmlStreamReader::EndElement)
        t = xml.readNext();
    if (t != QXmlStreamReader::Characters)
        return value;
    QString str = xml.text().toString();
    if (elemName == BOOL_DEF) {
        value = (str == TRUE_DEF);
    } else if (elemName == STRING_DEF) {
        value = str;
    } else if (elemName == INT_DEF) {
        value = str.toInt(&check);
    } else if (elemName == DOUBLE_DEF) {
        value = str.toDouble(&check);
    }
    if (!check)
        return QVariant();
    return value;
}
