/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontsconfeditorcontroller.h"

#include "fontconfigdefs.h"
#include "installedfontinfo.h"

FontsConfEditorController::FontsConfEditorController(const QString &family, QObject *parent) :
    QObject(parent), mFromApp(false), mFamily(family)
{
}

int FontsConfEditorController::count() const
{
    return mPrependFamily.count() + mPreferFamily.count() + mAcceptFamily.count();
}

int FontsConfEditorController::countOfFamily(const QString &priority) const
{
    if (priority == PREPEND_DEF)
        return mPrependFamily.count();
    else if (priority == PREFER_DEF)
        return mPreferFamily.count();
    else if (priority == ACCEPT_DEF)
        return mAcceptFamily.count();
    return -1;
}

int FontsConfEditorController::indexOfFamily(const QString &family, const QString &priority) const
{
    if (priority == PREPEND_DEF)
        return mPrependFamily.indexOf(family);
    else if (priority == PREFER_DEF)
        return mPreferFamily.indexOf(family);
    else if (priority == ACCEPT_DEF)
        return mAcceptFamily.indexOf(family);
    return -1;
}

void FontsConfEditorController::syncFamilyList()
{
    emit clearFamilyList();
    emit startSection(PREPEND_DEF, tr("Higher"));
    foreach (const QString &f, mPrependFamily) {
        InstalledFontInfo *info = mFontInfoMap.value(f);
        bool systemFont = info ? info->systemFont() : true;
        emit appendFamilyToList(mFamily, f, PREPEND_DEF, systemFont);
    }
    emit startSection(PREFER_DEF, tr("Normal"));
    foreach (const QString &f, mPreferFamily) {
        InstalledFontInfo *info = mFontInfoMap.value(f);
        bool systemFont = info ? info->systemFont() : true;
        emit appendFamilyToList(mFamily, f, PREFER_DEF, systemFont);
    }
    emit startSection(ACCEPT_DEF, tr("Lower"));
    foreach (const QString &f, mAcceptFamily) {
        InstalledFontInfo *info = mFontInfoMap.value(f);
        bool systemFont = info ? info->systemFont() : true;
        emit appendFamilyToList(mFamily, f, ACCEPT_DEF, systemFont);
    }
}

void FontsConfEditorController::appendFontsInfo(const QString &family, const QString &priority, InstalledFontInfo *info)
{
    mFromApp = true;
    mFontInfoMap.insert(family, info);
    appendFamily(family, priority);
    mFromApp = false;
}

void FontsConfEditorController::appendFamily(const QString &family, const QString &priority)
{
    if (priority == PREPEND_DEF)
        mPrependFamily.append(family);
    else if (priority == PREFER_DEF)
        mPreferFamily.append(family);
    else if (priority == ACCEPT_DEF)
        mAcceptFamily.append(family);
    if (!mFromApp)
        emit appendFamilyToConfig(mFamily, family, priority);
    emit countChanged();
}

void FontsConfEditorController::removeFamily(const QString &family, const QString &priority)
{
    if (priority.isEmpty() || priority == PREPEND_DEF) {
        mPrependFamily.removeOne(family);
        if (!mFromApp)
            emit removeFamilyFromList(mFamily, family, PREPEND_DEF);
        emit countChanged();
    }
    if (priority.isEmpty() || priority == PREFER_DEF) {
        mPreferFamily.removeOne(family);
        if (!mFromApp)
            emit removeFamilyFromList(mFamily, family, PREFER_DEF);
        emit countChanged();
    }
    if (priority.isEmpty() || priority == ACCEPT_DEF) {
        mAcceptFamily.removeOne(family);
        if (!mFromApp)
            emit removeFamilyFromList(mFamily, family, ACCEPT_DEF);
        emit countChanged();
    }
    if (!mPrependFamily.contains(family) && !mPreferFamily.contains(family) && !mAcceptFamily.contains(family))
        mFontInfoMap.remove(family);
}

void FontsConfEditorController::insertFamily(const QString &family, const QString &priority, int index)
{
    if (priority == PREPEND_DEF)
        mPrependFamily.insert(index, family);
    else if (priority == PREFER_DEF)
        mPreferFamily.insert(index, family);
    else if (priority == ACCEPT_DEF)
        mAcceptFamily.insert(index, family);
    if (!mFromApp)
        emit insertFamilyToConfig(mFamily, family, priority, index);
    emit countChanged();
}

void FontsConfEditorController::moveFamilyUp(const QString &family, const QString &priority)
{
    int index = indexOfFamily(family, priority);
    if (index > 0) {
        removeFamily(family, priority);
        insertFamily(family, priority, index - 1);
    }
}

void FontsConfEditorController::moveFamilyDown(const QString &family, const QString &priority)
{
    int index = indexOfFamily(family, priority);
    if (index - 1 < countOfFamily(priority)) {
        removeFamily(family, priority);
        insertFamily(family, priority, index + 1);
    }
}

void FontsConfEditorController::clear()
{
    mFontInfoMap.clear();
    mPrependFamily.clear();
    mPreferFamily.clear();
    mAcceptFamily.clear();
    emit clearFamilyList();
}
