/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontsconf.h"

#include "fontconfigdefs.h"
#include "fontsconfelement.h"

#include <QFile>
#include <QTextStream>
#include <QXmlStreamReader>

enum FCState {
    Initialized, Ready, InvalidConfig, UnknownState
};

static inline QString boolString(bool val)
{
    return val ? TRUE_DEF : FALSE_DEF;
}

FontsConf::FontsConf(QObject *parent) :
    QObject(parent), mModified(false), mValid(true), mHasUnknownConfig(false)
{
    initFontsConf();
}

bool FontsConf::isEmpty() const
{
    return !mElements || (mElements->count() == 0);
}

bool FontsConf::modified() const
{
    return mModified;
}

void FontsConf::setModified(bool value)
{
    mModified = value;
}

bool FontsConf::hasUnknownConfig() const
{
    return mHasUnknownConfig;
}

bool FontsConf::isValid() const
{
    return mValid;
}

void FontsConf::initFontsConf()
{
    if (mElements)
        mModified = true;
    FontsConfElementPointer rootElem(new FontsConfElement(FONTCONFIG_DEF));
    mElements = rootElem;
}

bool FontsConf::parse(const QByteArray &buf)
{
    if (buf.isEmpty())
        return true;

    QXmlStreamReader xml(buf);

    FCState state = Initialized;

    bool hasUnknownConfig = false;

    FontsConfElementPointer elem;
    while (!xml.atEnd()) {
        QXmlStreamReader::TokenType token = xml.readNext();
        if (token == QXmlStreamReader::DTD) {
            if (xml.dtdName() != FONTCONFIG_DEF || xml.dtdSystemId() != "fonts.dtd") {
                state = InvalidConfig;
            }
        } else if (token == QXmlStreamReader::StartElement) {
            QString elemName(xml.name().toString());
            switch (state) {
            case Initialized:
                if (elemName == FONTCONFIG_DEF) {
                    state = Ready;
                    elem = FontsConfElementPointer(new FontsConfElement(elemName));
                    bool check = elem->parse(xml);
                    if (!check)
                        state = UnknownState;
                } else
                    state = InvalidConfig;
                break;

            default:
                ;
            }
        } else if (token == QXmlStreamReader::EndElement) {
            QString elemName(xml.name().toString());
        }

        if (state == InvalidConfig)
            qWarning("Invalid Config!!!");

        if (state == InvalidConfig)
            break;
        else if (state == UnknownState)
            hasUnknownConfig = true;
    }

    if (xml.error() != QXmlStreamReader::NoError)
        state = InvalidConfig;

    if (state == Ready) {
        mElements = elem;
        mModified = false;
        mHasUnknownConfig = hasUnknownConfig;
    }

    return !(state == InvalidConfig);
}

QStringList FontsConf::preferFamily() const
{
    if (!mElements)
        return QStringList();

    QStringList familyList;

    int index = 0;
    while ((index = mElements->indexOf(ALIAS_DEF, index)) >= 0) {
        FontsConfElementPointer aliasElem = mElements->childElementAt(index++);
        FontsConfElementPointer familyElem = aliasElem->childElementOf(FAMILY_DEF);
        if (!familyElem || familyElem->text().isEmpty())
            continue;
        familyList << familyElem->text();
    }
    //    familyList.removeDuplicates();
    return familyList;
}

QStringList FontsConf::aliasFamilyFor(const QString &family, const QString &mode) const
{
    int index = 0;
    while ((index = mElements->indexOf(ALIAS_DEF, index)) >= 0) {
        FontsConfElementPointer aliasElem = mElements->childElementAt(index++);
        FontsConfElementPointer familyElem = aliasElem->childElementOf(FAMILY_DEF);
        if (!familyElem)
            continue;
        if (familyElem->text() != family)
            continue;
        FontsConfElementPointer targetElem = aliasElem->childElementOf(mode);
        if (!targetElem)
            return QStringList();
        QStringList familyList = textElementList(targetElem, FAMILY_DEF);
        return familyList;
    }

    return QStringList();
}

void FontsConf::appendAliasFamilyFor(const QString &family, const QString &mode, const QString &value)
{
    int index = 0;
    while ((index = mElements->indexOf(ALIAS_DEF, index)) >= 0) {
        FontsConfElementPointer aliasElem = mElements->childElementAt(index++);
        bool check = appendFamilyToAlias(aliasElem, family, mode, value);
        if (check)
            return;
    }

    FontsConfElementPointer aliasElem = aliasElementFor(family);
    mElements->addChildElement(aliasElem);
    bool check = appendFamilyToAlias(aliasElem, family, mode, value);
    Q_ASSERT(check);
}

void FontsConf::insertAliasFamilyFor(const QString &family, const QString &mode, const QString &value, int idx)
{
    int index = 0;
    while ((index = mElements->indexOf(ALIAS_DEF, index)) >= 0) {
        FontsConfElementPointer aliasElem = mElements->childElementAt(index++);
        bool check = insertFamilyToAlias(aliasElem, family, mode, value, idx);
        if (check)
            return;
    }

    FontsConfElementPointer aliasElem = aliasElementFor(family);
    mElements->addChildElement(aliasElem);
    bool check = appendFamilyToAlias(aliasElem, family, mode, value);
    Q_ASSERT(check);
}

void FontsConf::removeAliasFamilyFor(const QString &family, const QString &mode, const QString &value)
{
    int index = 0;
    while ((index = mElements->indexOf(ALIAS_DEF, index)) >= 0) {
        FontsConfElementPointer aliasElem = mElements->childElementAt(index++);

        FontsConfElementPointer familyElem = aliasElem->childElementOf(FAMILY_DEF);
        if (!familyElem || familyElem->text() != family)
            continue;

        FontsConfElementPointer targetElem = aliasElem->childElementOf(mode);
        if (!targetElem || targetElem->count() == 0)
            continue;

        int findex = 0;
        while ((findex = targetElem->indexOf(FAMILY_DEF, findex)) >= 0) {
            if (targetElem->childElementAt(findex)->text() == value) {
                targetElem->removeAt(findex);
                mModified = true;
                continue;
            }
            findex++;
        }

        if (targetElem->count() == 0) {
            aliasElem->removeOne(targetElem);
            if (aliasElem->count() == 1) {
                mElements->removeOne(aliasElem);
            }
        }
    }
}

QStringList FontsConf::preferFamilyFor(const QString &family) const
{
    return aliasFamilyFor(family, PREFER_DEF);
}

void FontsConf::appendPreferFamilyFor(const QString &family, const QString &value)
{
    appendAliasFamilyFor(family, PREFER_DEF, value);
}

void FontsConf::insertPreferFamilyFor(const QString &family, const QString &value, int index)
{
    insertAliasFamilyFor(family, PREFER_DEF, value, index);
}

void FontsConf::removePreferFamilyFor(const QString &family, const QString &value)
{
    removeAliasFamilyFor(family, PREFER_DEF, value);
}

QStringList FontsConf::acceptFamilyFor(const QString &family) const
{
    return aliasFamilyFor(family, ACCEPT_DEF);
}

void FontsConf::appendAcceptFamilyFor(const QString &family, const QString &value)
{
    appendAliasFamilyFor(family, ACCEPT_DEF, value);
}

void FontsConf::insertAcceptFamilyFor(const QString &family, const QString &value, int index)
{
    insertAliasFamilyFor(family, ACCEPT_DEF, value, index);
}

void FontsConf::removeAcceptFamilyFor(const QString &family, const QString &value)
{
    removeAliasFamilyFor(family, ACCEPT_DEF, value);
}

QStringList FontsConf::matchFamilyFor(const QString &config, bool val) const
{
    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer matchElem = mElements->childElementAt(index++);

        bool check = isMatchElementFor(matchElem, config, QString(), boolString(val));
        if (!check)
            continue;

        FontsConfElementPointer testElem = matchElem->childElementOf(TEST_DEF);
        if (!testElem)
            continue;

        QStringList familyList = textElementList(testElem, STRING_DEF);
        return familyList;
    }

    return QStringList();
}

QString FontsConf::matchEditValueFor(const QString &config, const QString &family) const
{
    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer matchElem = mElements->childElementAt(index++);

        bool check = isMatchElementFor(matchElem, config, family);
        if (!check)
            continue;

        FontsConfElementPointer editElem = matchElem->childElementOf(EDIT_DEF);
        if (!editElem)
            continue;

        FontsConfElementPointer boolElem = editElem->childElementOf(BOOL_DEF);
        if (!boolElem)
            continue;

        return boolElem->text();
    }

    return QString();
}

void FontsConf::setMatchEditValueFor(const QString &config, const QString &family, bool val)
{
    FontsConfElementPointer matchElem;
    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer elem = mElements->childElementAt(index++);
        bool check = isMatchElementFor(elem, config, QString(), boolString(val));
        if (check) {
            matchElem = elem;
            break;
        }
    }

    if (!matchElem) {
        matchElem = matchElementFor(config, val);
        mElements->addChildElement(matchElem);
        mModified = true;
    }

    FontsConfElementPointer testElem = matchElem->childElementOf(TEST_DEF);
    QStringList familyList = textElementList(testElem, STRING_DEF);
    if (familyList.contains(family))
        return;

    FontsConfElementPointer stringElem(new FontsConfElement(STRING_DEF));
    stringElem->setText(family);
    testElem->addChildElement(stringElem);
    mModified = true;
}

void FontsConf::unsetMatchEditValueFor(const QString &config, const QString &family)
{
    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer matchElem = mElements->childElementAt(index++);

        bool check = isMatchElementFor(matchElem, config, family);
        if (!check)
            continue;

        FontsConfElementPointer testElem = matchElem->childElementOf(TEST_DEF);
        int sindex = 0;
        while ((sindex = testElem->indexOf(STRING_DEF, sindex)) >= 0) {
            FontsConfElementPointer stringElem = testElem->childElementAt(sindex++);
            if (stringElem->text() == family) {
                testElem->removeAt(--sindex);
                mModified = true;
            }
        }

        if (testElem->count() == 0) {
            mElements->removeAt(--index);
        }

    }
}

QStringList FontsConf::patternFamilyFor(const QString &family, const QString &mode) const
{
    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer matchElem = mElements->childElementAt(index++);
        bool check = isPatternElementFor(matchElem, family, mode);
        if (!check)
            continue;
        FontsConfElementPointer editElem = getEditPatternElementFor(matchElem, family, mode);
        return textElementList(editElem, STRING_DEF);
    }
    return QStringList();
}

void FontsConf::appendPatternFamilyFor(const QString &family, const QString &mode, const QString &value)
{
    insertPatternFamilyFor(family, mode, value, -1);
}

void FontsConf::insertPatternFamilyFor(const QString &family, const QString &mode, const QString &value, int idx)
{
    FontsConfElementPointer matchElem;

    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer elem = mElements->childElementAt(index++);
        bool check = isPatternElementFor(elem, family, mode);
        if (check) {
            matchElem = elem;
            break;
        }
    }

    if (!matchElem) {
        int index = 0;
        while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
            FontsConfElementPointer elem = mElements->childElementAt(index++);
            bool check = isPatternElementFor(elem, family);
            if (check) {
                matchElem = elem;
                break;
            }
        }
    }

    if (!matchElem) {
        matchElem = patternElementFor(family);
        mElements->addChildElement(matchElem);
        mModified = true;
    }

    FontsConfElementPointer editElem = getEditPatternElementFor(matchElem, family, mode);
    if (!editElem) {
        editElem = FontsConfElementPointer(new FontsConfElement(EDIT_DEF));
        editElem->setAttribute(NAME_DEF, FAMILY_DEF);
        editElem->setAttribute(MODE_DEF, mode);
        editElem->setAttribute(BINDING_DEF, STRONG_DEF);
        matchElem->addChildElement(editElem);
        mModified = true;
    }

    QStringList familyList = textElementList(editElem, STRING_DEF);
    if (familyList.contains(value))
        return;

    FontsConfElementPointer stringElem(new FontsConfElement(STRING_DEF));
    stringElem->setText(value);
    if (idx < 0)
        editElem->addChildElement(stringElem);
    else
        editElem->insertChildElement(stringElem, idx);
    mModified = true;
}

void FontsConf::removePatternFamilyFor(const QString &family, const QString &mode, const QString &value)
{
    FontsConfElementPointer matchElem;

    int index = 0;
    while ((index = mElements->indexOf(MATCH_DEF, index)) >= 0) {
        FontsConfElementPointer elem = mElements->childElementAt(index++);
        bool check = isPatternElementFor(elem, family, mode);
        if (check) {
            matchElem = elem;
            break;
        }
    }

    if (!matchElem)
        return;

    FontsConfElementPointer editElem = getEditPatternElementFor(matchElem, family, mode);
    if (!editElem)
        return;

    int sindex = 0;
    while ((sindex = editElem->indexOf(STRING_DEF, sindex)) >= 0) {
        FontsConfElementPointer stringElem = editElem->childElementAt(sindex++);
        if (stringElem->text() == value) {
            editElem->removeAt(--sindex);
            mModified = true;
        }
    }

    if (editElem->count() == 0) {
        mModified = true;
        matchElem->removeOne(editElem);
        FontsConfElementList editElemList = matchElem->findChildrenElements(EDIT_DEF);
        if (editElemList.isEmpty())
            mElements->removeOne(matchElem);
    }

}

QStringList FontsConf::prependFamilyFor(const QString &family) const
{
    return patternFamilyFor(family, PREPEND_DEF);
}

void FontsConf::appendPrependFamilyFor(const QString &family, const QString &value)
{
    appendPatternFamilyFor(family, PREPEND_DEF, value);
}

void FontsConf::insertPrependFamilyFor(const QString &family, const QString &value, int index)
{
    insertPatternFamilyFor(family, PREPEND_DEF, value, index);
}

void FontsConf::removePrependFamilyFor(const QString &family, const QString &value)
{
    removePatternFamilyFor(family, PREPEND_DEF, value);
}

QStringList FontsConf::appendFamilyFor(const QString &family) const
{
    return patternFamilyFor(family, APPEND_DEF);
}

void FontsConf::appendAppendFamilyFor(const QString &family, const QString &value)
{
    appendPatternFamilyFor(family, APPEND_DEF, value);
}

void FontsConf::insertAppendFamilyFor(const QString &family, const QString &value, int index)
{
    insertPatternFamilyFor(family, APPEND_DEF, value, index);
}

void FontsConf::removeAppendFamilyFor(const QString &family, const QString &value)
{
    removePatternFamilyFor(family, APPEND_DEF, value);
}

void FontsConf::copy(FontsConf *src)
{
    mModified = true;
    mValid = src->mValid;
    mHasUnknownConfig = src->mHasUnknownConfig;
    mElements = src->mElements;
}

QStringList FontsConf::genericFamilies()
{
    static QStringList families;
    if (families.isEmpty())
        families << SANSSERIF_DEF << SERIF_DEF << MONOSPACE_DEF << SYSTEM_DEF;
    return families;
}

QStringList FontsConf::configKeys()
{
    static QStringList keys;
    if (keys.isEmpty())
        keys << EMBEDDEDBITMAP_DEF << HINTING_DEF << ANTIALIAS_DEF << AUTOHINT_DEF;
    return keys;
}

void FontsConf::load(const QString &path)
{
    QFile fp(path);
    if (!fp.exists())
        return;

    fp.open(QIODevice::ReadOnly);
    if (!fp.isOpen() || fp.error() != QFile::NoError)
        return;

    QByteArray buf = fp.readAll();

    fp.close();

    mValid = parse(buf);
}

void FontsConf::save(const QString &path)
{
#if 0
    if (!mElements && fp.exists()) {
        fp.remove();
        return;
    }
#endif

    QString buf;
    QXmlStreamWriter xml(&buf);

    xml.setAutoFormatting(true);
    xml.setAutoFormattingIndent(8);

    xml.writeStartDocument();

    xml.writeDTD("<!DOCTYPE fontconfig SYSTEM 'fonts.dtd'>");

    if (mElements)
        mElements->save(xml);

    xml.writeEndDocument();

    QFile fp(path);
    fp.open(QIODevice::WriteOnly);
    if (!fp.isOpen())
        return;

    QTextStream ts(&fp);
    ts.setCodec("UTF-8");
    ts << buf;

    mModified = false;

    fp.close();
}

bool FontsConf::appendFamilyToAlias(FontsConfElementPointer aliasElem, const QString &family, const QString &mode, const QString &value)
{
    return insertFamilyToAlias(aliasElem, family, mode, value, -1);
}

bool FontsConf::insertFamilyToAlias(FontsConfElementPointer aliasElem, const QString &family, const QString &mode, const QString &value, int index)
{
    if (aliasElem->type() != ALIAS_DEF)
        return false;

    FontsConfElementPointer familyElem = aliasElem->childElementOf(FAMILY_DEF);
    if (!familyElem || familyElem->text() != family)
        return false;

    FontsConfElementPointer targetElem = aliasElem->childElementOf(mode);
    if (!targetElem) {
        targetElem = FontsConfElementPointer(new FontsConfElement(mode));
        aliasElem->addChildElement(targetElem);
        mModified = true;
    }

    int findex = 0;
    while ((findex = targetElem->indexOf(FAMILY_DEF, findex)) >= 0) {
        if (targetElem->childElementAt(findex)->text() == value)
            return true;
        findex++;
    }

    FontsConfElementPointer valueElem(new FontsConfElement(FAMILY_DEF));
    valueElem->setText(value);
    if (index < 0)
        targetElem->addChildElement(valueElem);
    else
        targetElem->insertChildElement(valueElem, index);
    mModified = true;

    return true;
}

bool FontsConf::isMatchElementFor(FontsConfElementPointer matchElem, const QString &config, const QString &family, const QString &val) const
{
    if (!matchElem->hasAttribute(TARGET_DEF) || matchElem->value(TARGET_DEF) != FONT_DEF)
        return false;

    FontsConfElementList testElemList = matchElem->findChildrenElements(TEST_DEF);
    if (testElemList.count() != 1)
        return false;

    FontsConfElementPointer testElem = testElemList.at(0);
    if (!testElem->hasAttribute(NAME_DEF) || testElem->value(NAME_DEF) != FAMILY_DEF)
        return false;
    if ((testElem->hasAttribute(QUAL_DEF) && testElem->value(QUAL_DEF) != ANY_DEF) ||
            (testElem->hasAttribute(COMPARE_DEF) && testElem->value(COMPARE_DEF) != EQ_DEF) ||
            (testElem->hasAttribute(TARGET_DEF) && testElem->value(TARGET_DEF) != DEFAULT_DEF))
        return false;

    if (!family.isEmpty()) {
        if (testElem->count() == 0 && testElem->text() != family)
            return false;
        QStringList familyList = textElementList(testElem, STRING_DEF);
        if (!familyList.contains(family))
            return false;
    }

    FontsConfElementList editElemList = matchElem->findChildrenElements(EDIT_DEF);
    if (editElemList.count() != 1)
        return false;

    FontsConfElementPointer editElem = editElemList.at(0);
    if ((!editElem->hasAttribute(NAME_DEF) || editElem->value(NAME_DEF) != config) ||
            (editElem->hasAttribute(MODE_DEF) && editElem->value(MODE_DEF) != ASSIGN_DEF))
        return false;

    FontsConfElementList boolElemList = editElem->findChildrenElements(BOOL_DEF);
    if (boolElemList.count() != 1) {
        return false;
    }

    if (!val.isEmpty()) {
        FontsConfElementPointer boolElem = boolElemList.at(0);
        if (boolElem->text() != val)
            return false;
    }

    return true;
}

bool FontsConf::isPatternElementFor(FontsConfElementPointer matchElem, const QString &family, const QString &mode) const
{
    if (matchElem->hasAttribute(TARGET_DEF) && matchElem->value(TARGET_DEF) != PATTERN_DEF)
        return false;

    FontsConfElementList testElemList = matchElem->findChildrenElements(TEST_DEF);
    if (testElemList.count() != 1)
        return false;

    FontsConfElementPointer testElem = testElemList.at(0);
    if (testElem->hasAttribute(NAME_DEF) && testElem->value(NAME_DEF) != FAMILY_DEF)
        return false;
    if ((testElem->hasAttribute(COMPARE_DEF) && testElem->value(COMPARE_DEF) != EQ_DEF) ||
            (testElem->hasAttribute(TARGET_DEF) && testElem->value(TARGET_DEF) != DEFAULT_DEF))
        return false;

    if (testElem->count() == 0) {
        if (testElem->text().isEmpty() || testElem->text() != family)
            return false;
    } else {
        QStringList familyList = textElementList(testElem, STRING_DEF);
        if (familyList.count() != 1 || !familyList.contains(family))
            return false;
    }

    if (mode.isEmpty())
        return true;

    int eindex = 0;
    while ((eindex = matchElem->indexOf(EDIT_DEF, eindex)) >= 0) {
        FontsConfElementPointer editElem = matchElem->childElementAt(eindex++);

        if ((editElem->hasAttribute(NAME_DEF) && editElem->value(NAME_DEF) == FAMILY_DEF) &&
                (editElem->hasAttribute(MODE_DEF) && editElem->value(MODE_DEF) == mode))
            return true;

    }

    return false;
}

FontsConfElementPointer FontsConf::getEditPatternElementFor(FontsConfElementPointer matchElem, const QString &family, const QString &mode) const
{
    Q_UNUSED(family);
    int eindex = 0;
    while ((eindex = matchElem->indexOf(EDIT_DEF, eindex)) >= 0) {
        FontsConfElementPointer editElem = matchElem->childElementAt(eindex++);

        if ((editElem->hasAttribute(NAME_DEF) && editElem->value(NAME_DEF) == FAMILY_DEF) &&
                (editElem->hasAttribute(MODE_DEF) && editElem->value(MODE_DEF) == mode))
            return editElem;
    }
    return FontsConfElementPointer();
}

bool FontsConf::containsTextElement(FontsConfElementPointer elem, const QString &tag, const QString &text) const
{
    int index = 0;
    while ((index = elem->indexOf(tag, index)) >= 0) {
        FontsConfElementPointer e = elem->childElementAt(index++);
        if (e->text() == text)
            return true;
    }
    return false;
}

QStringList FontsConf::textElementList(FontsConfElementPointer elem, const QString &tag) const
{
    if (!elem)
        return QStringList();
    QStringList list;
    int index = 0;
    while ((index = elem->indexOf(tag, index)) >= 0) {
        FontsConfElementPointer e = elem->childElementAt(index++);
        if (!e->text().isEmpty())
            list << e->text();
    }
    return list;
}

FontsConfElementPointer FontsConf::aliasElementFor(const QString &family) const
{
    FontsConfElementPointer aliasElem(new FontsConfElement(ALIAS_DEF));
    FontsConfElementPointer familyElem(new FontsConfElement(FAMILY_DEF));
    familyElem->setText(family);
    aliasElem->addChildElement(familyElem);
    return aliasElem;
}

FontsConfElementPointer FontsConf::matchElementFor(const QString &config, bool val) const
{
    FontsConfElementPointer matchElem = FontsConfElementPointer(new FontsConfElement(MATCH_DEF));
    matchElem->setAttribute(TARGET_DEF, FONT_DEF);

    FontsConfElementPointer testElem(new FontsConfElement(TEST_DEF));
    testElem->setAttribute(QUAL_DEF, ANY_DEF);
    testElem->setAttribute(NAME_DEF, FAMILY_DEF);
    matchElem->addChildElement(testElem);

    FontsConfElementPointer editElem(new FontsConfElement(EDIT_DEF));
    editElem->setAttribute(NAME_DEF, config);
    editElem->setAttribute(MODE_DEF, ASSIGN_DEF);
    matchElem->addChildElement(editElem);

    FontsConfElementPointer boolElem(new FontsConfElement(BOOL_DEF));
    boolElem->setText(boolString(val));
    editElem->addChildElement(boolElem);

    return matchElem;
}

FontsConfElementPointer FontsConf::patternElementFor(const QString &family) const
{
    FontsConfElementPointer matchElem = FontsConfElementPointer(new FontsConfElement(MATCH_DEF));
    matchElem->setAttribute(TARGET_DEF, PATTERN_DEF);

    FontsConfElementPointer testElem(new FontsConfElement(TEST_DEF));
    testElem->setAttribute(QUAL_DEF, ANY_DEF);
    testElem->setAttribute(NAME_DEF, FAMILY_DEF);
    matchElem->addChildElement(testElem);

    FontsConfElementPointer stringElem(new FontsConfElement(STRING_DEF));
    stringElem->setText(family);
    testElem->addChildElement(stringElem);

    return matchElem;
}
