/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontinfo.h"

#include "fontconfigmanager.h"
#include "fontsconfigproperties.h"

#include <QFile>
#include <QFileInfo>
#include <QFontDatabase>

FontInfo::FontInfo(const QString &path, FontConfigManager *fontConfig, QObject *parent) :
    QObject(parent), mFontId(-1)
{
    setFontPath(path, fontConfig);
}

FontInfo::~FontInfo()
{
    if (mFontId >= 0) {
        QFontDatabase::removeApplicationFont(mFontId);
    }
}

QString FontInfo::fileName() const
{
    QFileInfo info(mFontPath);
    return info.fileName();
}

QString FontInfo::fontPath() const
{
    return mFontPath;
}

void FontInfo::setFontPath(const QString &fontPath, FontConfigManager *fontConfig)
{
    if (mFontPath == fontPath)
        return;
    if (!fontPath.isEmpty() && !QFile::exists(fontPath))
        return;
    mFontPath = fontPath;
    qDeleteAll(mFontProperties);
    mFontProperties.clear();
    if (fontPath.isEmpty()) {
        mFontId = -1;
        mFontFamilies.clear();
    } else {
        mFontId = QFontDatabase::addApplicationFont(mFontPath);
        mFontFamilies = QFontDatabase::applicationFontFamilies(mFontId);
        foreach (const QString &family, mFontFamilies) {
            FontsConfigProperties *prop = 0;
            if (fontConfig) {
                prop = fontConfig->fontProperty(family);
                prop->setParent(this);
            } else
                prop = new FontsConfigProperties(family, this);
            mFontProperties << prop;
        }
    }
    emit fontPathChanged();

}

QStringList FontInfo::families() const
{
    return mFontFamilies;
}

FontsConfigProperties *FontInfo::fontProperty(const QString &family) const
{
    foreach (FontsConfigProperties *prop, mFontProperties) {
        if (prop->family() == family)
            return prop;
    }
    return 0;
}
