/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#ifndef FONTCONFIGMANAGER_H
#define FONTCONFIGMANAGER_H

#include <QObject>
#include <QList>
#include <QStringList>
#include <QMap>
#include <QFont>

#include "fontsconfigproperties.h"

class QUrl;

class InstalledFontInfo;
class FontsConf;

class FontConfigManager : public QObject
{
    Q_OBJECT
public:
    explicit FontConfigManager(QObject *parent = 0);

    QString currentLanguage() const;
    void setCurrentLanguage(const QString &lang);

    QString localFontPath() const;
    void setLocalFontPath(const QString &path);

    QString localFontsConfPath() const;
    void setLocalFontsConfPath(const QString &path);

    bool localFontsConfExists() const;

    bool hasUnknownConfig() const;
    bool fontsConfModified() const;
    bool isEmptyLocalFontsConf() const;

    QStringList fontCount(const QString &fontpath) const;

    InstalledFontInfo *fontInfo(const QString &family, const QString &fullname = QString()) const;
    InstalledFontInfo *fontInfo(int index) const;
    int count() const;

    void appendFontProperty(FontsConfigProperties *prop, const QStringList &familyList = QStringList());
    void appendFontProperty(InstalledFontInfo *fontInfo);
    FontsConfigProperties *fontProperty(const QString &family, const QStringList &familyList = QStringList()) const;
    FontsConfigProperties *fontProperty(InstalledFontInfo *fontInfo) const;

    QString localFontsConf() const;

    QStringList preferFamilyFor(const QString &family) const;
    QStringList acceptFamilyFor(const QString &family) const;
    QStringList prependFamilyFor(const QString &family) const;
    QStringList appendFamilyFor(const QString &family) const;

    void addPreferFamily(const QString &family, const QString &value);
    void addAcceptFamily(const QString &family, const QString &value);
    void addPrependFamily(const QString &family, const QString &value);
    void addAppendFamily(const QString &family, const QString &value);

    void insertPreferFamily(const QString &family, const QString &value, int index);
    void insertAcceptFamily(const QString &family, const QString &value, int index);
    void insertPrependFamily(const QString &family, const QString &value, int index);
    void insertAppendFamily(const QString &family, const QString &value, int index);

    void removePreferFamily(const QString &family, const QString &value);
    void removeAcceptFamily(const QString &family, const QString &value);
    void removePrependFamily(const QString &family, const QString &value);
    void removeAppendFamily(const QString &family, const QString &value);

    bool isConflict(const QString &family, const QStringList &lang) const;
    QStringList systemPreferFamilyFor(const QString &family) const;

    void importSystemSettings(const QString &family);

    void createRecommendedSettings();

    QStringList installableFamily(const QString &family, bool localOnly = false);

    QString localeFamily(const QString &family) const;

private:
    bool maybeSansSerifFont(InstalledFontInfo *info) const;
    bool maybeSerifFont(InstalledFontInfo *info) const;
    bool maybeMonospaceFont(InstalledFontInfo *info) const;
    QFont font4info(InstalledFontInfo *info, int pointSize) const;

signals:
    void fcCacheFinished();

    void localFontsConfExistsChanged();
    void localFontsConfPathChanged();

    void startUpdateFontsConfig();
    void endUpdateFontsConfig();
    void fontListUpdated();
    void fontsConfUpdated();

    void warning(const QString &message);

public slots:
    void runFcCache();
    void readFcList();
    void readFontsConf();
    void saveFontsConf();
    void backupFontsConf(const QString &filepath);
    void restoreFontsConf(const QString &filepath);

    void resetFontsConf();

private:
    QString mLang;
    QString mLocalFontPath;
    QString mLocalFontsConfPath;
    QList<InstalledFontInfo *> mFcListInfo;
    QMap<QString, QString> mEnLocaleFontMap;

    FontsConf *mLocalFontsConf;
    FontsConf *mSystemLocalConf;

    QList<QStringList> mConflictFonts;
};

#endif // FONTCONFIGMANAGER_H
