/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "fontconfigmanager.h"

#include "fontconfigdefs.h"
#include "installedfontinfo.h"
#include "fontsconfigproperties.h"

#include "fontsconf.h"

#include <QProcess>
#include <QBuffer>
#include <QFile>
#include <QXmlStreamReader>
#include <QVariant>
#include <QFontInfo>
#include <QFontMetrics>
#include <QFontDatabase>
#include <QUrl>
#include <QTextStream>

#undef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS

#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
#include <QDebug>
#endif

#define FCBIN_PATH "/usr/bin/"
#define FCCACHE_COMMAND "fc-cache"
#define FCCACHE_OPTION "-v"
#define FCLIST_COMMAND "fc-list"
#define FCLIST_OPTION "-v"

FontConfigManager::FontConfigManager(QObject *parent) :
    QObject(parent), mLang("en"), mLocalFontsConf(0)
{
    connect(this, SIGNAL(localFontsConfPathChanged()), SIGNAL(localFontsConfExistsChanged()));
    //    connect(this, SIGNAL(localFontsConfPathChanged()), SIGNAL(fontsConfUpdated()));

    mLocalFontsConf = new FontsConf(this);
    mSystemLocalConf = new FontsConf(this);

    QStringList cjkConflicts;
    cjkConflicts << "ja" << "ko" << "zh-cn" << "zh-hk" << "zh-mo" << "zh-sg" << "zh-tw";
    mConflictFonts << cjkConflicts;

    mSystemLocalConf->load("/etc/fonts/local.conf");

}

QString FontConfigManager::currentLanguage() const
{
    return mLang;
}

void FontConfigManager::setCurrentLanguage(const QString &lang)
{
    mLang = lang;
}

QString FontConfigManager::localFontPath() const
{
    return mLocalFontPath;
}

void FontConfigManager::setLocalFontPath(const QString &path)
{
    mLocalFontPath = path;
}

QString FontConfigManager::localFontsConfPath() const
{
    return mLocalFontsConfPath;
}

void FontConfigManager::setLocalFontsConfPath(const QString &path)
{
    if (mLocalFontsConfPath != path) {
        mLocalFontsConfPath = path;
        emit localFontsConfPathChanged();
    }
}

bool FontConfigManager::localFontsConfExists() const
{
    return QFile::exists(mLocalFontsConfPath);
}

bool FontConfigManager::hasUnknownConfig() const
{
    if (!mLocalFontsConf)
        return false;
    return mLocalFontsConf->hasUnknownConfig();
}

bool FontConfigManager::fontsConfModified() const
{
    if (!mLocalFontsConf)
        return false;
    return mLocalFontsConf->modified();
}

bool FontConfigManager::isEmptyLocalFontsConf() const
{
    return !mLocalFontsConf || mLocalFontsConf->isEmpty();
}

QStringList FontConfigManager::fontCount(const QString &fontpath) const
{
    QStringList fontList;
    foreach (InstalledFontInfo *info, mFcListInfo) {
        if (info->file() == fontpath)
            fontList << info->localefamily();
    }
    return fontList;
}

InstalledFontInfo *FontConfigManager::fontInfo(const QString &family, const QString &fullname) const
{
    InstalledFontInfo *info = 0;
    foreach (info, mFcListInfo) {
        if (info->family().contains(family)) {
            if (fullname.isEmpty())
                return info;
            if (info->fullname().contains(fullname))
                return info;
        }
    }
    return 0;
}

InstalledFontInfo *FontConfigManager::fontInfo(int index) const
{
    if (index >= 0 && index < mFcListInfo.count())
        return mFcListInfo.at(index);
    return 0;
}

int FontConfigManager::count() const
{
    return mFcListInfo.count();
}

void FontConfigManager::appendFontProperty(FontsConfigProperties *prop, const QStringList &familyList)
{
    if (!mLocalFontsConf)
        return;

    QStringList preferedFamilies(mLocalFontsConf->preferFamily());
    preferedFamilies << prop->preferFamilies();
    preferedFamilies.removeDuplicates();
    foreach (const QString &family, preferedFamilies) {
        if (prop->preferFamilies().contains(family)) {
            if (familyList.isEmpty())
                mLocalFontsConf->appendPreferFamilyFor(family, prop->family());
            else
                foreach (const QString &f, familyList)
                    mLocalFontsConf->appendPreferFamilyFor(family, f);
        } else {
            if (familyList.isEmpty())
                mLocalFontsConf->removePreferFamilyFor(family, prop->family());
            else
                foreach (const QString &f, familyList)
                    mLocalFontsConf->removePreferFamilyFor(family, f);
        }
    }

    QStringList keys = FontsConf::configKeys();
    foreach (const QString &key, keys) {
        if (prop->configValue(key) == FontsConfigProperties::Default) {
            if (familyList.isEmpty())
                mLocalFontsConf->unsetMatchEditValueFor(key, prop->family());
            else
                foreach (const QString &f, familyList)
                    mLocalFontsConf->unsetMatchEditValueFor(key, f);
        } else {
            if (familyList.isEmpty())
                mLocalFontsConf->setMatchEditValueFor(key, prop->family(), (prop->configValue(key) == FontsConfigProperties::True));
            else
                foreach (const QString &f, familyList)
                    mLocalFontsConf->setMatchEditValueFor(key, f, (prop->configValue(key) == FontsConfigProperties::True));
        }
    }

    emit fontsConfUpdated();
}

void FontConfigManager::appendFontProperty(InstalledFontInfo *fontInfo)
{
    QStringList familyList;
    familyList << fontInfo->enfamily();
    appendFontProperty(fontInfo->fontProperty(), familyList);
}

FontsConfigProperties *FontConfigManager::fontProperty(const QString &family, const QStringList &familyList) const
{
    FontsConfigProperties *prop = new FontsConfigProperties(family);
    QStringList families;
    QStringList preferedFamilies = mLocalFontsConf->preferFamily();
    foreach (const QString &key, preferedFamilies) {
        families = mLocalFontsConf->preferFamilyFor(key);
        if (families.contains(family))
            prop->addPreferFamily(key);
        foreach (const QString &f, familyList) {
            if (families.contains(f))
                prop->addPreferFamily(key);
        }
    }

    QStringList keys = FontsConf::configKeys();
    foreach (const QString &key, keys) {
        QString val = mLocalFontsConf->matchEditValueFor(key, prop->family());
        if (!val.isEmpty()) {
            prop->setConfigValue(key, (QVariant(val).toBool() ? FontsConfigProperties::True : FontsConfigProperties::False));
            continue;
        }
        foreach (const QString &f, familyList) {
            mLocalFontsConf->matchEditValueFor(key, f);
            if (!val.isEmpty()) {
                prop->setConfigValue(key, (QVariant(val).toBool() ? FontsConfigProperties::True : FontsConfigProperties::False));
                break;
            }
        }
    }

    return prop;
}

FontsConfigProperties *FontConfigManager::fontProperty(InstalledFontInfo *fontInfo) const
{
    return fontProperty(fontInfo->enfamily(), fontInfo->family());
}

QString FontConfigManager::localFontsConf() const
{
    QFile fp(mLocalFontsConfPath);

    if (!fp.exists())
        return QString();

    if (!fp.open(QIODevice::ReadOnly))
        return QString();

    QTextStream ts(&fp);
    ts.setCodec("UTF-8");
    QString buf = ts.readAll();

    fp.close();

    return buf;
}

QStringList FontConfigManager::preferFamilyFor(const QString &family) const
{
    return mLocalFontsConf->preferFamilyFor(family);
}

QStringList FontConfigManager::acceptFamilyFor(const QString &family) const
{
    return mLocalFontsConf->acceptFamilyFor(family);
}

QStringList FontConfigManager::prependFamilyFor(const QString &family) const
{
    return mLocalFontsConf->prependFamilyFor(family);
}

QStringList FontConfigManager::appendFamilyFor(const QString &family) const
{
    return mLocalFontsConf->appendFamilyFor(family);
}

bool FontConfigManager::isConflict(const QString &family, const QStringList &lang) const
{
    InstalledFontInfo *info = fontInfo(family);
    if (!info)
        return false;

    QList<int> confListIndex;
    for (int i = 0; i < mConflictFonts.count(); i++) {
        const QStringList list(mConflictFonts.at(i));
        foreach (const QString &f, list) {
            if (lang.contains(f)) {
                if (!confListIndex.contains(i))
                    confListIndex << i;
                break;
            }
        }
    }

    if (confListIndex.isEmpty())
        return false;

    foreach (int i, confListIndex) {
        QStringList confList = mConflictFonts.at(confListIndex.at(i));
        foreach (const QString &f, confList) {
            if (info->lang().contains(f)) {
                return true;
            }
        }
    }

    return false;
}

QStringList FontConfigManager::systemPreferFamilyFor(const QString &family) const
{
    return mSystemLocalConf->preferFamilyFor(family);
}

void FontConfigManager::importSystemSettings(const QString &family)
{
    QStringList systemFamilyList = systemPreferFamilyFor(family);

    QStringList installedFontLang;
    foreach (InstalledFontInfo *info, mFcListInfo) {
        if (info->systemFont())
            continue;
        installedFontLang << info->lang();
    }
    installedFontLang.sort();
    installedFontLang.removeDuplicates();

    emit startUpdateFontsConfig();
    foreach (const QString &f, systemFamilyList) {
        InstalledFontInfo *info = fontInfo(f);
        if (!info)
            continue;

        if (isConflict(f, installedFontLang))
            mLocalFontsConf->appendAcceptFamilyFor(family, f);
        else
            mLocalFontsConf->appendPrependFamilyFor(family, f);
    }
    emit endUpdateFontsConfig();

    if (mLocalFontsConf->modified()) {
        emit fontsConfUpdated();
    }
}

void FontConfigManager::createRecommendedSettings()
{
    QList<InstalledFontInfo*> sansSerifFonts;
    QList<InstalledFontInfo*> serifFonts;
    QList<InstalledFontInfo*> monospaceFonts;
    QList<InstalledFontInfo*> monospaceSansSerifFonts;
    QList<InstalledFontInfo*> monospaceSerifFonts;
    QList<InstalledFontInfo*> unknownFonts;

    foreach (InstalledFontInfo *info, mFcListInfo) {
        if (info->systemFont())
            continue;
        if (maybeMonospaceFont(info)) {
            if (maybeSansSerifFont(info))
                monospaceSansSerifFonts << info;
            else if (maybeSerifFont(info))
                monospaceSerifFonts << info;
            else
                monospaceFonts << info;
        } else if (maybeSansSerifFont(info))
            sansSerifFonts << info;
        else if (maybeSerifFont(info))
            serifFonts << info;
        else
            unknownFonts << info;
    }

    emit startUpdateFontsConfig();
    resetFontsConf();

#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
    QStringList f;
    foreach (InstalledFontInfo *info, monospaceFonts)
        f << info->localefamily();
    qDebug() << "Mono:" << f;
    f.clear();
    foreach (InstalledFontInfo *info, monospaceSansSerifFonts)
        f << info->localefamily();
    qDebug() << "Mono(Sans Serif):" << f;
    f.clear();
    foreach (InstalledFontInfo *info, monospaceSerifFonts)
        f << info->localefamily();
    qDebug() << "Mono(Serif):" << f;
    f.clear();
    foreach (InstalledFontInfo *info, sansSerifFonts)
        f << info->localefamily();
    qDebug() << "Sans Serif:" << f;
    f.clear();
    foreach (InstalledFontInfo *info, serifFonts)
        f << info->localefamily();
    qDebug() << "Serif:" << f;
    f.clear();
    foreach (InstalledFontInfo *info, unknownFonts)
        f << info->localefamily();
    qDebug() << "Unknown:" << f;
#endif

    if (monospaceFonts.count())
        foreach (InstalledFontInfo *info, monospaceFonts) {
            addPreferFamily(MONOSPACE_DEF, info->enfamily());
        }
    else if (monospaceSansSerifFonts.count())
        foreach (InstalledFontInfo *info, monospaceSansSerifFonts) {
            addPreferFamily(MONOSPACE_DEF, info->enfamily());
        }
    else if (monospaceSerifFonts.count())
        foreach (InstalledFontInfo *info, monospaceSerifFonts) {
            addPreferFamily(MONOSPACE_DEF, info->enfamily());
        }

    if (sansSerifFonts.count())
        foreach (InstalledFontInfo *info, sansSerifFonts) {
            addPreferFamily(SANSSERIF_DEF, info->enfamily());
            addPreferFamily(SYSTEM_DEF, info->enfamily());
        }
    else if (monospaceSansSerifFonts.count())
        foreach (InstalledFontInfo *info, monospaceSansSerifFonts) {
            addPreferFamily(SANSSERIF_DEF, info->enfamily());
            addPreferFamily(SYSTEM_DEF, info->enfamily());
        }
    else if (unknownFonts.count())
        foreach (InstalledFontInfo *info, unknownFonts) {
            addPreferFamily(SANSSERIF_DEF, info->enfamily());
            addPreferFamily(SYSTEM_DEF, info->enfamily());
        }
    else if (monospaceFonts.count())
        foreach (InstalledFontInfo *info, monospaceFonts) {
            addPreferFamily(SANSSERIF_DEF, info->enfamily());
            addPreferFamily(SYSTEM_DEF, info->enfamily());
        }

    if (serifFonts.count())
        foreach (InstalledFontInfo *info, serifFonts) {
            addPreferFamily(SERIF_DEF, info->enfamily());
        }
    else if (monospaceSerifFonts.count())
        foreach (InstalledFontInfo *info, monospaceSerifFonts) {
            addPreferFamily(SANSSERIF_DEF, info->enfamily());
        }
    else if (unknownFonts.count())
        foreach (InstalledFontInfo *info, unknownFonts) {
            addPreferFamily(SERIF_DEF, info->enfamily());
        }
    else if (monospaceFonts.count())
        foreach (InstalledFontInfo *info, monospaceFonts) {
            addPreferFamily(SERIF_DEF, info->enfamily());
        }

    foreach (const QString &f, FontsConf::genericFamilies())
        importSystemSettings(f);
    emit endUpdateFontsConfig();

    emit fontsConfUpdated();
}

QStringList FontConfigManager::installableFamily(const QString &family, bool localOnly)
{
    QStringList installedFont;
    installedFont << prependFamilyFor(family);
    installedFont << appendFamilyFor(family);
    installedFont << preferFamilyFor(family);
    installedFont << acceptFamilyFor(family);

    QStringList familyList;
    foreach (InstalledFontInfo *info, mFcListInfo) {
        if (localOnly && info->systemFont())
            continue;
        bool check = false;
        foreach (const QString &f, info->family()) {
            if (installedFont.contains(f)) {
                check = true;
                break;
            }
        }
        if (!check)
            familyList << info->localefamily();
    }
    familyList.sort();
    familyList.removeDuplicates();

    return familyList;
}

QString FontConfigManager::localeFamily(const QString &family) const
{
    if (mEnLocaleFontMap.contains(family))
        return mEnLocaleFontMap.value(family);
    return family;
}

bool FontConfigManager::maybeSansSerifFont(InstalledFontInfo *info) const
{
    foreach (const QString &f, info->family()) {
        if (f.contains("gothic", Qt::CaseInsensitive) ||
                f.contains("arial", Qt::CaseInsensitive) ||
                f.contains("helvetica", Qt::CaseInsensitive) ||
                f.contains("verdana", Qt::CaseInsensitive) ||
                f.contains("sans", Qt::CaseInsensitive))
            return true;
    }
    return false;
}

bool FontConfigManager::maybeSerifFont(InstalledFontInfo *info) const
{
    foreach (const QString &f, info->family()) {
        if (f.contains("mincho", Qt::CaseInsensitive) ||
                f.contains("times", Qt::CaseInsensitive) ||
                (f.contains("serif", Qt::CaseInsensitive) && !f.contains("sans", Qt::CaseInsensitive)))
            return true;
    }
    return false;
}

#define DEFAULT_POINTSIZE 24

bool FontConfigManager::maybeMonospaceFont(InstalledFontInfo *info) const
{
    int fontId = -1;
    QFont font = font4info(info, DEFAULT_POINTSIZE);
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
    qDebug() << "0 - FontConfigManager::maybeMonospaceFont(" << info->enfamily() << info->enstyle() << point << ")" << font.family() << font.exactMatch();
#endif
    if (!font.exactMatch() || font.family() != info->enfamily()) {
        fontId = QFontDatabase::addApplicationFont(info->file());
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
        qDebug() << "\tfontId:" << fontId << info->file();
#endif
        if (fontId >= 0) {
            font = font4info(info, DEFAULT_POINTSIZE);
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
            qDebug() << "1 - FontConfigManager::maybeMonospaceFont(" << info->enfamily() << info->enstyle() << point << ")" << font.family() << font.exactMatch();
#endif
            if (!font.exactMatch() || font.family() != info->enfamily()) {
                font = QFont(info->enfamily());
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
                qDebug() << "2 - FontConfigManager::maybeMonospaceFont(" << info->enfamily() << info->enstyle() << point << ")" << font.family() << font.exactMatch();
#endif
            }
        }
    }
    bool isFixed = false;
    if (font.exactMatch()) {
        QFontInfo fi(font);
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
        qDebug() << "\tfixedPitch:" << fi.fixedPitch();
#endif
        if (fi.fixedPitch()) {
            isFixed = true;
        } else {
            QFontMetrics fm(font);
            int w = fm.width(QLatin1Char('A'));
#ifdef FONTMANAGER_DEBUG_RECOMMENDED_SETTINGS
            qDebug() << "\twidth:" << w << fm.width(QLatin1Char('i')) << fm.width(QLatin1Char('X'));
#endif
            if (fm.width(QLatin1Char('i')) == w && fm.width(QLatin1Char('X')) == w) {
                isFixed = true;
            }
        }
    }
    if (fontId >= 0)
        QFontDatabase::removeApplicationFont(fontId);
    if (isFixed)
        return true;
    foreach (const QString &f, info->family()) {
        if (f.contains("courier", Qt::CaseInsensitive) ||
                f.contains("mono", Qt::CaseInsensitive))
            return true;
    }
    return false;
}

QFont FontConfigManager::font4info(InstalledFontInfo *info, int pointSize) const
{
    QFontDatabase fdb;
    int point = pointSize;
    if (!fdb.isScalable(info->enfamily(), info->enstyle())) {
        QList<int> points = fdb.pointSizes(info->enfamily(), info->enstyle());
        if (points.count() > 0)
            point = points.first();
    }
    QFont font = fdb.font(info->enfamily(), info->enstyle(), point);
    return font;
}

void FontConfigManager::runFcCache()
{
    QProcess *proc = new QProcess(this);
    connect(proc, SIGNAL(finished(int)), SIGNAL(fcCacheFinished()));
    connect(proc, SIGNAL(finished(int)), proc, SLOT(deleteLater()));
    proc->start(FCBIN_PATH FCCACHE_COMMAND, QStringList() << FCCACHE_OPTION);
}

void FontConfigManager::readFcList()
{
    QProcess *proc = new QProcess(this);
    proc->start(FCBIN_PATH FCLIST_COMMAND, QStringList() << FCLIST_OPTION);
    if (!proc->waitForStarted())
        return;

    if (!proc->waitForFinished())
        return;

    qDeleteAll(mFcListInfo);
    mFcListInfo.clear();
    mEnLocaleFontMap.clear();

    QByteArray buf = proc->readAllStandardOutput();
    QByteArray errbuf = proc->readAllStandardError();
    if (!errbuf.isEmpty())
        qWarning("%s", errbuf.constData());
    Q_ASSERT(errbuf.isEmpty());

    static QByteArray emptyLine("\n");
    QBuffer buffer(&buf);
    QByteArray fibuf;
    buffer.open(QIODevice::ReadOnly);
    Q_ASSERT(buffer.isOpen());
    while (!buffer.atEnd()) {
        QByteArray linebuf = buffer.readLine();
        if (linebuf.isEmpty() || linebuf == emptyLine) {
            if (!fibuf.isEmpty()) {
                InstalledFontInfo *info = new InstalledFontInfo(fibuf, mLocalFontPath, this);
                info->setLocale(mLang);
                if (info->isValid()) {
                    mFcListInfo.append(info);
                    mEnLocaleFontMap.insert(info->enfamily(), info->localefamily());
                    info->setFontProperty(fontProperty(info));
                } else
                    delete info;
                fibuf.clear();
            }
        } else {
            fibuf += linebuf;
        }
    }
    Q_ASSERT(fibuf.isEmpty());
    buffer.close();

    qSort(mFcListInfo.begin(), mFcListInfo.end(), InstalledFontInfo::compare);

    emit fontListUpdated();

    delete proc;
}

void FontConfigManager::readFontsConf()
{
    mLocalFontsConf->load(mLocalFontsConfPath);
    emit fontsConfUpdated();
}

void FontConfigManager::saveFontsConf()
{
    if (!mLocalFontsConf || mLocalFontsConfPath.isEmpty())
        return;

    bool check = localFontsConfExists();
    if (isEmptyLocalFontsConf()) {
        if (check) {
            QFile::remove(mLocalFontsConfPath);
            emit localFontsConfExistsChanged();
        }
    } else {
        mLocalFontsConf->save(mLocalFontsConfPath);
        if (!check)
            emit localFontsConfExistsChanged();
    }
}

void FontConfigManager::backupFontsConf(const QString &filepath)
{
    if (filepath.isEmpty() || isEmptyLocalFontsConf())
        return;
    mLocalFontsConf->save(filepath);
}

void FontConfigManager::restoreFontsConf(const QString &filepath)
{
    if (filepath.isEmpty() || !QFile::exists(filepath))
        return;
    FontsConf *restoredConf = new FontsConf(this);
    restoredConf->load(filepath);
    if (restoredConf->isValid()) {
        mLocalFontsConf->copy(restoredConf);
        emit fontsConfUpdated();
        delete restoredConf;
    }
}

void FontConfigManager::resetFontsConf()
{
    if (!mLocalFontsConf)
        return;
    mLocalFontsConf->initFontsConf();
    emit fontsConfUpdated();
}

void FontConfigManager::addPreferFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->appendPreferFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::addAcceptFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->appendAcceptFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::addPrependFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->appendPrependFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::addAppendFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->appendAppendFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::insertPreferFamily(const QString &family, const QString &value, int index)
{
    mLocalFontsConf->insertPreferFamilyFor(family, value, index);
    emit fontsConfUpdated();
}

void FontConfigManager::insertAcceptFamily(const QString &family, const QString &value, int index)
{
    mLocalFontsConf->insertAcceptFamilyFor(family, value, index);
    emit fontsConfUpdated();
}

void FontConfigManager::insertPrependFamily(const QString &family, const QString &value, int index)
{
    mLocalFontsConf->insertPrependFamilyFor(family, value, index);
    emit fontsConfUpdated();
}

void FontConfigManager::insertAppendFamily(const QString &family, const QString &value, int index)
{
    mLocalFontsConf->insertAppendFamilyFor(family, value, index);
    emit fontsConfUpdated();
}

void FontConfigManager::removePreferFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->removePreferFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::removeAcceptFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->removeAcceptFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::removePrependFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->removePrependFamilyFor(family, value);
    emit fontsConfUpdated();
}

void FontConfigManager::removeAppendFamily(const QString &family, const QString &value)
{
    mLocalFontsConf->removeAppendFamilyFor(family, value);
    emit fontsConfUpdated();
}
