/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#ifndef APPLICATIONCONTROLLER_H
#define APPLICATIONCONTROLLER_H

#include <QObject>
#include <QStringList>
#include <QVariant>
#include <QUrl>

#include "fontinfo.h"

class FontConfigManager;
class InstalledFontInfo;
class FontsConfEditorController;

class ApplicationController : public QObject
{
    Q_OBJECT
    Q_PROPERTY(QString version READ version NOTIFY dummySignal)
    Q_PROPERTY(QString fontDir READ fontDir WRITE setFontDir NOTIFY fontDirChanged)
    Q_PROPERTY(bool fontDirExists READ fontDirExists NOTIFY fontDirExistsChanged)

    Q_PROPERTY(QString localFontsConfPath READ localFontsConfPath NOTIFY localFontsConfPathChanged)
    Q_PROPERTY(bool localFontsConfExists READ localFontsConfExists NOTIFY localFontsConfExistsChanged)
    Q_PROPERTY(bool isEmptyFontsConf READ isEmptyFontsConf NOTIFY localFontsConfFileUpdated)
    Q_PROPERTY(QString localFontsConf READ localFontsConf NOTIFY localFontsConfFileUpdated)

    Q_PROPERTY(bool showSystemFont READ showSystemFont WRITE setShowSystemFont NOTIFY showSystemFontChanged)

    Q_PROPERTY(bool working READ working NOTIFY workingChanged)

    Q_PROPERTY(QUrl backupDir READ backupDir NOTIFY backupDirChanged)

    Q_PROPERTY(QStringList installedFonts READ installedFonts NOTIFY installedFontsChanged)
public:
    explicit ApplicationController(QObject *parent = 0);
    
public slots:
    void init();

public:
    QString version() const;

    QString currentLanguage() const;

    QString fontDir() const;
    void setFontDir(const QString &dirpath);

    bool fontDirExists() const;

    bool showSystemFont() const;
    void setShowSystemFont(bool show);

    Q_INVOKABLE FontInfo *checkFontInfo(const QUrl &path);
    Q_INVOKABLE InstalledFontInfo *fontInfo(const QString &family, const QString &fullname = QString()) const;
    Q_INVOKABLE QStringList fontCount(const QString &fontpath) const;

    Q_INVOKABLE QString localeFamily(const QString &family) const;

    Q_INVOKABLE bool fontExists(FontInfo *fontinfo);

    Q_INVOKABLE FontsConfEditorController *editorController(const QString &family);
    void updateEditorController(const QString &family);

    QUrl backupDir() const;
    Q_INVOKABLE QString defaultBackupFilename() const;

    Q_INVOKABLE QString url2path(const QUrl &url) const;
    Q_INVOKABLE QString path4display(const QString &path) const;

    QStringList installedFonts() const;

public slots:
    void updateAllEditorController();

public:
    bool localFontsConfExists() const;
    QString localFontsConfPath() const;
    QString localFontsConf() const;
    bool isEmptyFontsConf() const;

    bool working() const;

public slots:
    void startUpdateLocalFontsConf();
    void endUpdateLocalFontsConf();
    void localFontsConfUpdated();

    void resetLocalFontsConf();
    void importSystemSettings(const QString &family);

    void createRecommendedSettings();

    void backupConfig(const QString &filename);
    void restoreConfig(const QString &filename);
    void restoreConfig(const QUrl &filename);

signals:
    void alertDialog(const QString &message);

    void fontDirChanged(const QString dirpath);
    void fontDirExistsChanged();

    void showSystemFontChanged();
    void installedFontsChanged();

    void installFinished(const QString &fontpath);
    void uninstallFinished(const QString &fontpath);
    void backupConfigFinished(const QString &filepath);
    void restoreConfigFinished(const QString &filepath);

    void clearInstalledFontList();
    void appendInstalledFont(const QString &family, const QString &fullname);

    void clearInstallableFamilyListFor(const QString &family);
    void appendInstallableFamily(const QString &enfamily, const QString &family, bool systemFont);

    void localFontsConfPathChanged();
    void localFontsConfExistsChanged();
    void localFontsConfChanged();
    void localFontsConfFileUpdated();

    void workingChanged();

    void backupDirChanged();

    void dummySignal();

private slots:
    void readFcListFinished();

public slots:
    void createFontDir();

    void installFont(FontInfo *fontinfo);

    void updateFontsConf(InstalledFontInfo *fontInfo);

    void uninstallFont(const QString &fontpath);
    
    void syncInstalledFonts();
    void syncInstallableFamilyFor(const QString &family);

    void saveFontsConf();

    void appendFamilyToConfig(const QString &family, const QString &value, const QString &priority);
    void insertFamilyToConfig(const QString &family, const QString &value, const QString &priority, int index);
    void removeFamilyFromConfig(const QString &family, const QString &value, const QString &priority);

private:
    QString mLang;
    QString mFontDirPath;
    bool mFontDirExists;
    bool mShowSystemFont;
    QStringList mInstalledFonts;

    int mUpdating;
    bool mWorking;
    bool mIgnoreUpdate;

    FontConfigManager *mFontConfig;
    QMap<QString, FontsConfEditorController*> mEditorController;
};

#endif // APPLICATIONCONTROLLER_H
