/****************************************************************************
**
** Copyright (C) 2012 Takumi Asaki
** All rights reserved.
** Contact: Takumi Asaki (takumi.asaki@gmail.com)
**
** This file is part of the fontmanager application.
**
** You may use this file under the terms of the BSD license as follows:
**
** "Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions are
** met:
**   * Redistributions of source code must retain the above copyright
**     notice, this list of conditions and the following disclaimer.
**   * Redistributions in binary form must reproduce the above copyright
**     notice, this list of conditions and the following disclaimer in
**     the documentation and/or other materials provided with the
**     distribution.
**   * Neither the name of Nokia Corporation and its Subsidiary(-ies) nor
**     the names of its contributors may be used to endorse or promote
**     products derived from this software without specific prior written
**     permission.
**
** THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
** "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
** LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
** A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
** OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
** SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
** LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
** DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
** THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
** (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
** OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE."
**
****************************************************************************/

#include "applicationcontroller.h"

#include "fontconfigdefs.h"
#include "fontconfigmanager.h"
#include "installedfontinfo.h"
#include "fontsconfeditorcontroller.h"
#include "fontsconf.h"

#include <QtCore>

#define VERSION_STRING "0.5"

ApplicationController::ApplicationController(QObject *parent) :
    QObject(parent), mFontDirExists(false), mShowSystemFont(false),
    mUpdating(0),
    mWorking(false), mIgnoreUpdate(false),
    mFontConfig(0)
{

}

void ApplicationController::init()
{
    mFontConfig = new FontConfigManager(this);
    setFontDir(QDir::homePath() + QLatin1String("/.fonts"));

    QLocale curLocale;
    mLang = curLocale.name();
    int idx = mLang.indexOf(QLatin1Char('_'));
    if (idx > 0)
        mLang = mLang.left(idx);
    mFontConfig->setCurrentLanguage(mLang);

    connect(mFontConfig, SIGNAL(fcCacheFinished()), mFontConfig, SLOT(readFcList()));
    connect(mFontConfig, SIGNAL(fontListUpdated()), SLOT(readFcListFinished()));
    connect(mFontConfig, SIGNAL(fontListUpdated()), SLOT(syncInstalledFonts()));

    connect(mFontConfig, SIGNAL(localFontsConfPathChanged()), SIGNAL(localFontsConfPathChanged()));
    connect(mFontConfig, SIGNAL(localFontsConfExistsChanged()), SIGNAL(localFontsConfExistsChanged()));

    connect(mFontConfig, SIGNAL(startUpdateFontsConfig()), SLOT(startUpdateLocalFontsConf()));
    connect(mFontConfig, SIGNAL(fontsConfUpdated()), SLOT(localFontsConfUpdated()));
    connect(mFontConfig, SIGNAL(endUpdateFontsConfig()), SLOT(endUpdateLocalFontsConf()));

    connect(this, SIGNAL(localFontsConfChanged()), SLOT(updateAllEditorController()));
    connect(this, SIGNAL(localFontsConfChanged()), SLOT(saveFontsConf()));

    connect(mFontConfig, SIGNAL(warning(QString)), SIGNAL(alertDialog(QString)));

    mFontConfig->setLocalFontsConfPath(QDir::homePath() + QLatin1String("/.fonts.conf"));

    mFontConfig->readFontsConf();
    mFontConfig->readFcList();

    foreach (const QString &f, FontsConf::genericFamilies()) {
        updateEditorController(f);
    }

}

QString ApplicationController::version() const
{
    return QLatin1String(VERSION_STRING);
}

QString ApplicationController::currentLanguage() const
{
    return mLang;
}

QString ApplicationController::fontDir() const
{
    return mFontDirPath;
}

void ApplicationController::setFontDir(const QString &dirpath)
{
    if (mFontDirPath != dirpath) {
        mFontDirPath = dirpath;
        emit fontDirChanged(mFontDirPath);
        QDir fontDir(dirpath);
        mFontDirExists = fontDir.exists();
        emit fontDirExistsChanged();
        if (mFontConfig)
            mFontConfig->setLocalFontPath(dirpath);
    }
}

bool ApplicationController::fontDirExists() const
{
    return mFontDirExists;
}

bool ApplicationController::showSystemFont() const
{
    return mShowSystemFont;
}

void ApplicationController::setShowSystemFont(bool show)
{
    if (mShowSystemFont != show) {
        mShowSystemFont = show;
        emit showSystemFontChanged();
    }
}

FontInfo *ApplicationController::checkFontInfo(const QUrl &path)
{
    FontInfo *fInfo = new FontInfo(path.toLocalFile(), mFontConfig, this);
    return fInfo;
}

InstalledFontInfo *ApplicationController::fontInfo(const QString &family, const QString &fullname) const
{
    return mFontConfig->fontInfo(family, fullname);
}

QStringList ApplicationController::fontCount(const QString &fontpath) const
{
    return mFontConfig->fontCount(fontpath);
}

QString ApplicationController::localeFamily(const QString &family) const
{
    return mFontConfig->localeFamily(family);
}

bool ApplicationController::fontExists(FontInfo *fontinfo)
{
    QFileInfo srcfont(fontinfo->fontPath());
    QFileInfo dstfont(mFontDirPath + QLatin1String("/") + srcfont.fileName());
    return dstfont.exists();
}

FontsConfEditorController *ApplicationController::editorController(const QString &family)
{
    FontsConfEditorController *controller = mEditorController.value(family, 0);
    if (!controller) {
        controller = new FontsConfEditorController(family, this);
        mEditorController.insert(family, controller);
        connect(controller, SIGNAL(appendFamilyToConfig(QString,QString,QString)), SLOT(appendFamilyToConfig(QString,QString,QString)));
        connect(controller, SIGNAL(insertFamilyToConfig(QString,QString,QString,int)), SLOT(insertFamilyToConfig(QString,QString,QString,int)));
        connect(controller, SIGNAL(removeFamilyFromList(QString,QString,QString)), SLOT(removeFamilyFromConfig(QString,QString,QString)));
        updateEditorController(family);
    }
    return controller;
}

void ApplicationController::updateEditorController(const QString &family)
{
    FontsConfEditorController *controller = editorController(family);

    QStringList familyList;

    controller->clear();

    familyList = mFontConfig->prependFamilyFor(family);
    foreach (const QString &f, familyList) {
        InstalledFontInfo *info = fontInfo(f);
        controller->appendFontsInfo(f, PREPEND_DEF, info);
    }

    familyList = mFontConfig->preferFamilyFor(family);
    foreach (const QString &f, familyList) {
        InstalledFontInfo *info = fontInfo(f);
        controller->appendFontsInfo(f, PREFER_DEF, info);
    }

    familyList = mFontConfig->acceptFamilyFor(family);
    foreach (const QString &f, familyList) {
        InstalledFontInfo *info = fontInfo(f);
        controller->appendFontsInfo(f, ACCEPT_DEF, info);
    }

    controller->syncFamilyList();
}

QUrl ApplicationController::backupDir() const
{
    return QUrl(QDir::homePath() + QLatin1String("/MyDocs/Documents"));
}

QString ApplicationController::defaultBackupFilename() const
{
    QString backupdir = backupDir().toString();
    QString backupfile = backupdir + QLatin1String("/fonts.conf");
    int idx = 0;
    while (QFile::exists(backupfile)) {
        backupfile = backupdir + QString::fromAscii("/fonts-%1.conf").arg(++idx);
    }
    return backupfile;
}

QString ApplicationController::url2path(const QUrl &url) const
{
    QString path = url.toLocalFile();
    return path;
}

QString ApplicationController::path4display(const QString &path) const
{
    QString str(path);
    if (str.startsWith(QDir::homePath()))
        str.replace(0, QDir::homePath().length(), QLatin1String("~"));
//    str.replace(QLatin1Char('/'), "/<wbr>");
    return str;
}

QStringList ApplicationController::installedFonts() const
{
    return mInstalledFonts;
}

void ApplicationController::updateAllEditorController()
{
    if (!mFontConfig->fontsConfModified() || mIgnoreUpdate) {
        mIgnoreUpdate = false;
        return;
    }
    foreach (const QString &f, mEditorController.keys()) {
        updateEditorController(f);
    }
}

bool ApplicationController::localFontsConfExists() const
{
    if (!mFontConfig)
        return false;
    return QFile::exists(mFontConfig->localFontsConfPath());
}

QString ApplicationController::localFontsConfPath() const
{
    if (!mFontConfig)
        return QString();
    return mFontConfig->localFontsConfPath();
}

QString ApplicationController::localFontsConf() const
{
    if (!mFontConfig)
        return QString();
    return mFontConfig->localFontsConf();
}

bool ApplicationController::isEmptyFontsConf() const
{
    if (!localFontsConfExists())
        return true;
    return !mFontConfig || mFontConfig->isEmptyLocalFontsConf();
}

bool ApplicationController::working() const
{
    return mWorking;
}


void ApplicationController::startUpdateLocalFontsConf()
{
    mUpdating++;
}

void ApplicationController::endUpdateLocalFontsConf()
{
    mUpdating--;
    Q_ASSERT(mUpdating >= 0);
}

void ApplicationController::localFontsConfUpdated()
{
    if (mUpdating == 0)
        emit localFontsConfChanged();
}

void ApplicationController::resetLocalFontsConf()
{
    if (!mFontConfig)
        return;
    mFontConfig->resetFontsConf();
}

void ApplicationController::importSystemSettings(const QString &family)
{
    mFontConfig->importSystemSettings(family);
    if (mFontConfig->fontsConfModified())
        updateEditorController(family);
}

void ApplicationController::createRecommendedSettings()
{
    mFontConfig->createRecommendedSettings();
}

void ApplicationController::backupConfig(const QString &filename)
{
    mFontConfig->backupFontsConf(filename);
    emit backupConfigFinished(filename);
}

void ApplicationController::restoreConfig(const QString &filename)
{
    mFontConfig->restoreFontsConf(filename);
    emit restoreConfigFinished(filename);
}

void ApplicationController::restoreConfig(const QUrl &filename)
{
    restoreConfig(filename.toLocalFile());
}

void ApplicationController::createFontDir()
{
    QDir fontDir(mFontDirPath);
    if (!fontDir.exists()) {
        fontDir.mkpath(mFontDirPath);
        mFontDirExists = fontDir.exists();
        emit fontDirExistsChanged();
    }
}

void ApplicationController::installFont(FontInfo *fontinfo)
{
    QFileInfo srcfont(fontinfo->fontPath());
    QFileInfo dstfont(mFontDirPath + QLatin1String("/") + srcfont.fileName());

    QFile::copy(srcfont.absoluteFilePath(), dstfont.absoluteFilePath());

    foreach (const QString &family, fontinfo->families()) {
        FontsConfigProperties *prop = fontinfo->fontProperty(family);
        mFontConfig->appendFontProperty(prop);
    }

    if (!mInstalledFonts.contains(dstfont.absoluteFilePath())) {
        mInstalledFonts.append(dstfont.absoluteFilePath());
        emit installedFontsChanged();
    }

    emit installFinished(srcfont.fileName());

    mWorking = true;
    emit workingChanged();

    mFontConfig->runFcCache();
}

void ApplicationController::updateFontsConf(InstalledFontInfo *fontInfo)
{
    mFontConfig->appendFontProperty(fontInfo);
}

void ApplicationController::uninstallFont(const QString &fontpath)
{
    bool check = QFile::remove(fontpath);

    if (mInstalledFonts.contains(fontpath)) {
        mInstalledFonts.removeOne(fontpath);
        emit installedFontsChanged();
    }

    if (check) {
        emit uninstallFinished(fontpath);

        mWorking = true;
        emit workingChanged();

        mFontConfig->runFcCache();
    } else
        emit alertDialog(tr("Could not remove Font '%1'").arg(fontpath));
}

void ApplicationController::syncInstalledFonts()
{
    if (!mFontConfig)
        return;
    emit clearInstalledFontList();
    for (int i = 0; i < mFontConfig->count(); i++) {
        InstalledFontInfo *info = mFontConfig->fontInfo(i);
        if (mShowSystemFont || !info->systemFont())
            emit appendInstalledFont(info->localefamily(), info->localefullname());
    }
}

void ApplicationController::syncInstallableFamilyFor(const QString &family)
{
    if (!mFontConfig)
        return;
    emit clearInstallableFamilyListFor(family);
    QStringList familyList = mFontConfig->installableFamily(family);
    foreach (const QString &f, familyList) {
        InstalledFontInfo *info = mFontConfig->fontInfo(f);
        emit appendInstallableFamily(info->enfamily(), info->localefamily(), info->systemFont());
    }
}

void ApplicationController::saveFontsConf()
{
    if (!mFontConfig->fontsConfModified()) {
        emit localFontsConfFileUpdated();
        return;
    }

    mFontConfig->saveFontsConf();
    emit localFontsConfFileUpdated();
}

void ApplicationController::appendFamilyToConfig(const QString &family, const QString &value, const QString &priority)
{
    FontsConfEditorController *controller = qobject_cast<FontsConfEditorController*>(sender());
    if (controller)
        mIgnoreUpdate = true;
    if (priority == PREPEND_DEF)
        mFontConfig->addPrependFamily(family, value);
    else if (priority == APPEND_DEF)
        mFontConfig->addAppendFamily(family, value);
    else if (priority == PREFER_DEF)
        mFontConfig->addPreferFamily(family, value);
    else if (priority == ACCEPT_DEF)
        mFontConfig->addAcceptFamily(family, value);
}

void ApplicationController::insertFamilyToConfig(const QString &family, const QString &value, const QString &priority, int index)
{
    FontsConfEditorController *controller = qobject_cast<FontsConfEditorController*>(sender());
    if (controller)
        mIgnoreUpdate = true;
    if (priority == PREPEND_DEF)
        mFontConfig->insertPrependFamily(family, value, index);
    else if (priority == APPEND_DEF)
        mFontConfig->insertAppendFamily(family, value, index);
    else if (priority == PREFER_DEF)
        mFontConfig->insertPreferFamily(family, value, index);
    else if (priority == ACCEPT_DEF)
        mFontConfig->insertAcceptFamily(family, value, index);
}

void ApplicationController::removeFamilyFromConfig(const QString &family, const QString &value, const QString &priority)
{
    FontsConfEditorController *controller = qobject_cast<FontsConfEditorController*>(sender());
    if (controller)
        mIgnoreUpdate = true;
    if (priority == PREPEND_DEF)
        mFontConfig->removePrependFamily(family, value);
    else if (priority == APPEND_DEF)
        mFontConfig->removeAppendFamily(family, value);
    else if (priority == PREFER_DEF)
        mFontConfig->removePreferFamily(family, value);
    else if (priority == ACCEPT_DEF)
        mFontConfig->removeAcceptFamily(family, value);
}

void ApplicationController::readFcListFinished()
{
    mWorking = false;
    emit workingChanged();
}
