/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.transformer;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.cyberneko.html.parsers.DOMParser;
import org.seasar.extension.unit.S2TestCase;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.InputSource;

public class FessXpathTransformerTest extends S2TestCase {
    public void test_pruneNode() throws Exception {
        String data = "<html><body><br/><script>hoge</script><noscript>fuga</noscript></body></html>";
        Document document = getDocument(data);

        FessXpathTransformer transformer = new FessXpathTransformer();

        Node pruneNode = transformer.pruneNode(document.cloneNode(true));
        assertEquals(getXmlString(document), getXmlString(pruneNode));
    }

    public void test_pruneNode_removeNoScript() throws Exception {
        String data = "<html><body><br/><script>hoge</script><noscript>fuga</noscript></body></html>";
        Document document = getDocument(data);

        FessXpathTransformer transformer = new FessXpathTransformer();
        transformer.prunedTagList.add("noscript");

        Node pruneNode = transformer.pruneNode(document.cloneNode(true));
        String docString = getXmlString(document);
        String pnString = getXmlString(pruneNode);
        assertTrue(docString.contains("<SCRIPT>"));
        assertTrue(docString.contains("hoge"));
        assertTrue(docString.contains("<NOSCRIPT>"));
        assertTrue(docString.contains("fuga"));
        assertTrue(pnString.contains("<SCRIPT>"));
        assertTrue(pnString.contains("hoge"));
        assertFalse(pnString.contains("<NOSCRIPT>"));
        assertFalse(pnString.contains("fuga"));
    }

    public void test_pruneNode_removeScriptAndNoscript() throws Exception {
        String data = "<html><body><br/><script>hoge</script><noscript>fuga</noscript></body></html>";
        Document document = getDocument(data);

        FessXpathTransformer transformer = new FessXpathTransformer();
        transformer.prunedTagList.add("script");
        transformer.prunedTagList.add("noscript");

        Node pruneNode = transformer.pruneNode(document.cloneNode(true));
        String docString = getXmlString(document);
        String pnString = getXmlString(pruneNode);
        assertTrue(docString.contains("<SCRIPT>"));
        assertTrue(docString.contains("hoge"));
        assertTrue(docString.contains("<NOSCRIPT>"));
        assertTrue(docString.contains("fuga"));
        assertFalse(pnString.contains("<SCRIPT>"));
        assertFalse(pnString.contains("hoge"));
        assertFalse(pnString.contains("<NOSCRIPT>"));
        assertFalse(pnString.contains("fuga"));
    }

    private Document getDocument(String data) throws Exception {
        DOMParser parser = new DOMParser();
        ByteArrayInputStream is = new ByteArrayInputStream(data
                .getBytes("UTF-8"));
        parser.parse(new InputSource(is));
        return parser.getDocument();
    }

    private String getXmlString(Node node) throws Exception {
        TransformerFactory tf = TransformerFactory.newInstance();
        Transformer transformer = tf.newTransformer();
        transformer.setOutputProperty(OutputKeys.ENCODING, "UTF-8");
        transformer.setOutputProperty(OutputKeys.INDENT, "no");
        //        transformer.setOutputProperty(OutputKeys.METHOD, "xml");

        StringWriter writer = new StringWriter();
        StreamResult result = new StreamResult(writer);

        DOMSource source = new DOMSource(node);
        transformer.transform(source, result);

        return writer.toString();

    }
}
