/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.service;

import java.io.Serializable;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.FacetInfo;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.entity.SearchQuery.SortField;
import jp.sf.fess.helper.QueryHelper;
import jp.sf.fess.solr.FessSolrQueryException;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.solr.SolrServerManager;
import jp.sf.fess.util.QueryResponseList;

import org.apache.solr.client.solrj.SolrQuery;
import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.seasar.framework.util.StringUtil;

public class SearchService implements Serializable {

    private static final long serialVersionUID = 1L;

    @Resource
    protected SolrServerManager solrServerManager;

    @Resource
    protected QueryHelper queryHelper;

    public List<Map<String, Object>> selectList(String query,
            FacetInfo facetInfo, int start, int rows) {
        return selectList(query, facetInfo, start, rows, true);
    }

    public List<Map<String, Object>> selectList(String query,
            FacetInfo facetInfo, int start, int rows, boolean forUser) {
        long startTime = System.currentTimeMillis();

        SolrServerGroup solrServerGroup = forUser ? solrServerManager
                .getSelectSolrServerGroup() : solrServerManager
                .getUpdateSolrServerGroup();

        QueryResponse queryResponse = null;
        SolrQuery solrQuery = new SolrQuery();
        SearchQuery searchQuery = queryHelper.build(query, forUser);
        String q = searchQuery.getQuery();
        if (StringUtil.isNotBlank(q)) {
            // fields
            solrQuery.setFields(queryHelper.getResponseFields());
            // query
            solrQuery.setQuery(q);
            solrQuery.setStart(start);
            solrQuery.setRows(rows);
            // sort
            for (SortField sortField : searchQuery.getSortFields()) {
                solrQuery.setSortField(sortField.getField(), Constants.DESC
                        .equals(sortField.getOrder()) ? SolrQuery.ORDER.desc
                        : SolrQuery.ORDER.asc);
            }
            // highlighting
            if (queryHelper.getHighlightingFields() != null
                    && queryHelper.getHighlightingFields().length != 0) {
                for (String hf : queryHelper.getHighlightingFields()) {
                    solrQuery.addHighlightField(hf);
                }
                solrQuery.setHighlightSnippets(queryHelper
                        .getHighlightSnippetSize());
            }
            // shards
            if (queryHelper.getShards() != null) {
                solrQuery.setParam("shards", queryHelper.getShards());
            }

            // facets
            if (facetInfo != null) {
                solrQuery.setFacet(true);
                if (facetInfo.field != null) {
                    for (String f : facetInfo.field) {
                        if (queryHelper.isFacetField(f)) {
                            solrQuery.addFacetField(f);
                        } else {
                            throw new FessSolrQueryException(f
                                    + " is not supported as a facet field.");
                        }
                    }
                }
                if (facetInfo.query != null) {
                    for (String fq : facetInfo.query) {
                        String facetQuery = queryHelper.buildFacetQuery(fq);
                        if (StringUtil.isNotBlank(facetQuery)) {
                            solrQuery.addFacetQuery(facetQuery);
                        } else {
                            throw new FessSolrQueryException(fq + "("
                                    + facetQuery
                                    + ") is not supported as a facet query.");
                        }
                    }
                }
                if (facetInfo.limit != null) {
                    solrQuery.setFacetLimit(Integer.parseInt(facetInfo.limit));
                }
                if (facetInfo.minCount != null) {
                    solrQuery.setFacetMinCount(Integer
                            .parseInt(facetInfo.minCount));
                }
                if (facetInfo.missing != null) {
                    solrQuery.setFacetMissing(Boolean
                            .parseBoolean(facetInfo.missing));
                }
                if (facetInfo.prefix != null) {
                    solrQuery.setFacetPrefix(facetInfo.prefix);
                }
                if (facetInfo.sort != null) {
                    if (queryHelper.isFacetSortValue(facetInfo.sort)) {
                        solrQuery.setFacetSort(facetInfo.sort);
                    }
                }
            }

            queryResponse = solrServerGroup.query(solrQuery,
                    SolrRequest.METHOD.POST);
        }
        long execTime = System.currentTimeMillis() - startTime;

        QueryResponseList queryResponseList = new QueryResponseList(
                queryResponse, rows);
        queryResponseList.setSearchQuery(q);
        queryResponseList.setSolrQuery(solrQuery.toString());
        queryResponseList.setExecTime(execTime);
        return queryResponseList;
    }

}
