/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.crud.action.admin;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.crud.CommonConstants;
import jp.sf.fess.crud.CrudMessageException;
import jp.sf.fess.crud.util.SAStrutsUtil;
import jp.sf.fess.db.exentity.LabelType;
import jp.sf.fess.form.admin.LabelTypeForm;
import jp.sf.fess.pager.LabelTypePager;
import jp.sf.fess.service.LabelTypeService;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.seasar.framework.beans.util.Beans;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.annotation.ActionForm;
import org.seasar.struts.annotation.Execute;
import org.seasar.struts.exception.ActionMessagesException;

public class BsLabelTypeAction implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Log log = LogFactory.getLog(BsLabelTypeAction.class);

    // for list

    public List<LabelType> labelTypeItems;

    // for edit/confirm/delete

    @ActionForm
    @Resource
    protected LabelTypeForm labelTypeForm;

    @Resource
    protected LabelTypeService labelTypeService;

    @Resource
    protected LabelTypePager labelTypePager;

    protected String displayList(boolean redirect) {
        // page navi
        labelTypeItems = labelTypeService.getLabelTypeList(labelTypePager);

        // restore from pager
        Beans.copy(labelTypePager, labelTypeForm.searchParams)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        if (redirect) {
            return "index?redirect=true";
        } else {
            return "index.jsp";
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String index() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "list/{pageNumber}")
    public String list() {
        // page navi
        if (StringUtil.isNotBlank(labelTypeForm.pageNumber)) {
            try {
                labelTypePager.setCurrentPageNumber(Integer
                        .parseInt(labelTypeForm.pageNumber));
            } catch (NumberFormatException e) {
                if (log.isDebugEnabled()) {
                    log.debug("Invalid value: " + labelTypeForm.pageNumber, e);
                }
            }
        }

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String search() {
        Beans.copy(labelTypeForm.searchParams, labelTypePager)
                .excludes(CommonConstants.PAGER_CONVERSION_RULE).execute();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String reset() {
        labelTypePager.clear();

        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String back() {
        return displayList(false);
    }

    @Execute(validator = false, input = "error.jsp")
    public String editagain() {
        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "confirmpage/{crudMode}/{id}")
    public String confirmpage() {
        if (labelTypeForm.crudMode != CommonConstants.CONFIRM_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.CONFIRM_MODE,
                            labelTypeForm.crudMode });
        }

        loadLabelType();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String createpage() {
        // page navi
        labelTypeForm.initialize();
        labelTypeForm.crudMode = CommonConstants.CREATE_MODE;

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "editpage/{crudMode}/{id}")
    public String editpage() {
        if (labelTypeForm.crudMode != CommonConstants.EDIT_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.EDIT_MODE,
                            labelTypeForm.crudMode });
        }

        loadLabelType();

        return "edit.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String editfromconfirm() {
        labelTypeForm.crudMode = CommonConstants.EDIT_MODE;

        loadLabelType();

        return "edit.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromcreate() {
        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String confirmfromupdate() {
        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp", urlPattern = "deletepage/{crudMode}/{id}")
    public String deletepage() {
        if (labelTypeForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            labelTypeForm.crudMode });
        }

        loadLabelType();

        return "confirm.jsp";
    }

    @Execute(validator = false, input = "error.jsp")
    public String deletefromconfirm() {
        labelTypeForm.crudMode = CommonConstants.DELETE_MODE;

        loadLabelType();

        return "confirm.jsp";
    }

    @Execute(validator = true, input = "edit.jsp")
    public String create() {
        try {
            LabelType labelType = createLabelType();
            labelTypeService.store(labelType);
            SAStrutsUtil.addSessionMessage("success.crud_create_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_create_crud_table");
        }
    }

    @Execute(validator = true, input = "edit.jsp")
    public String update() {
        try {
            LabelType labelType = createLabelType();
            labelTypeService.store(labelType);
            SAStrutsUtil.addSessionMessage("success.crud_update_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_update_crud_table");
        }
    }

    @Execute(validator = false, input = "error.jsp")
    public String delete() {
        if (labelTypeForm.crudMode != CommonConstants.DELETE_MODE) {
            throw new ActionMessagesException("errors.crud_invalid_mode",
                    new Object[] { CommonConstants.DELETE_MODE,
                            labelTypeForm.crudMode });
        }

        try {
            LabelType labelType = labelTypeService.getLabelType(createKeyMap());
            if (labelType == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { labelTypeForm.id });

            }

            labelTypeService.delete(labelType);
            SAStrutsUtil.addSessionMessage("success.crud_delete_crud_table");

            return displayList(true);
        } catch (ActionMessagesException e) {
            log.error(e.getMessage(), e);
            throw e;
        } catch (CrudMessageException e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(e.getMessageId(), e.getArgs());
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ActionMessagesException(
                    "errors.crud_failed_to_delete_crud_table");
        }
    }

    protected void loadLabelType() {

        LabelType labelType = labelTypeService.getLabelType(createKeyMap());
        if (labelType == null) {
            // throw an exception
            throw new ActionMessagesException(
                    "errors.crud_could_not_find_crud_table",

                    new Object[] { labelTypeForm.id });

        }

        Beans.copy(labelType, labelTypeForm).excludes("searchParams", "mode")
                .execute();
    }

    protected LabelType createLabelType() {
        LabelType labelType;
        if (labelTypeForm.crudMode == CommonConstants.EDIT_MODE) {
            labelType = labelTypeService.getLabelType(createKeyMap());
            if (labelType == null) {
                // throw an exception
                throw new ActionMessagesException(
                        "errors.crud_could_not_find_crud_table",

                        new Object[] { labelTypeForm.id });

            }
        } else {
            labelType = new LabelType();
        }
        Beans.copy(labelTypeForm, labelType).excludes("searchParams", "mode")
                .execute();

        return labelType;
    }

    protected Map<String, String> createKeyMap() {
        Map<String, String> keys = new HashMap<String, String>();

        keys.put("id", labelTypeForm.id);

        return keys;
    }
}
