/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.BufferedReader;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;

import javax.servlet.ServletContext;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.exec.Crawler;

import org.apache.commons.lang.SystemUtils;
import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.seasar.robot.util.CharUtil;
import org.seasar.struts.util.RequestUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SystemHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(SystemHelper.class);

    private AtomicBoolean crawlProcessStatus = new AtomicBoolean();

    private AtomicBoolean replicationProcessStatus = new AtomicBoolean();

    private String sessionId;

    private boolean forceStop = false;

    private Process currentProcess;

    private String adminRole = "fess";

    private List<String> authenticatedRoleList;

    private String[] crawlerJavaOptions = new String[] {
            "-Djava.awt.headless=true", "-server", "-XX:+UseGCOverheadLimit",
            "-XX:+UseConcMarkSweepGC", "-XX:+CMSIncrementalMode",
            "-XX:+UseTLAB", "-Dpdfbox.cjk.support=true", "-Xmx512m",
            "-XX:MaxPermSize=128m" };

    private String logFilePath = System.getProperty("fess.log.file");

    private String solrHome = System.getProperty("solr.solr.home");

    private String solrDataDir = System.getProperty("solr.data.dir");

    private String javaCommandPath = "java";

    public String filterPathEncoding = Constants.UTF_8;

    public void executeCrawler(String sessionId) {
        List<String> crawlerCmdList = new ArrayList<String>();
        String cpSeparator = SystemUtils.IS_OS_WINDOWS ? ";" : ":";
        ServletContext servletContext = SingletonS2Container
                .getComponent(ServletContext.class);

        crawlerCmdList.add(javaCommandPath);

        // -cp
        crawlerCmdList.add("-cp");
        StringBuilder buf = new StringBuilder();
        // WEB-INF/cmd/resources
        buf.append("WEB-INF");
        buf.append(File.separator);
        buf.append("cmd");
        buf.append(File.separator);
        buf.append("resources");
        buf.append(cpSeparator);
        // WEB-INF/classes
        buf.append("WEB-INF");
        buf.append(File.separator);
        buf.append("classes");
        // WEB-INF/lib
        appendJarFile(cpSeparator, servletContext, buf, "/WEB-INF/lib",
                "WEB-INF" + File.separator + "lib" + File.separator);
        // WEB-INF/cmd/lib
        appendJarFile(cpSeparator, servletContext, buf, "/WEB-INF/cmd/lib",
                "WEB-INF" + File.separator + "cmd" + File.separator + "lib"
                        + File.separator);
        crawlerCmdList.add(buf.toString());

        crawlerCmdList.add("-Dfess.crawler.process=true");
        crawlerCmdList.add("-Dsolr.solr.home=" + solrHome);
        crawlerCmdList.add("-Dsolr.data.dir=" + solrDataDir);
        crawlerCmdList.add("-Dfess.log.file=" + logFilePath);
        if (crawlerJavaOptions != null) {
            for (String value : crawlerJavaOptions) {
                crawlerCmdList.add(value);
            }
        }

        crawlerCmdList.add(Crawler.class.getCanonicalName());

        crawlerCmdList.add("--sessionId");
        crawlerCmdList.add(sessionId);

        File baseDir = new File(servletContext.getRealPath("/"));

        if (logger.isInfoEnabled()) {
            logger.info("Crawler: \nDirectory=" + baseDir + "\nOptions="
                    + crawlerCmdList);
        }

        ProcessBuilder pb = new ProcessBuilder(crawlerCmdList);
        pb.directory(baseDir);
        pb.redirectErrorStream(true);

        // TODO check if currentProcess exists
        try {
            currentProcess = pb.start();

            InputStreamThread it = new InputStreamThread(currentProcess
                    .getInputStream(), Constants.UTF_8);
            it.start();

            currentProcess.waitFor();
            it.join(5000);

            int exitValue = currentProcess.exitValue();

            if (logger.isInfoEnabled()) {
                logger.info("Crawler: Exit Code=" + exitValue
                        + " - Crawler Process Output:\n" + it.getOutput());
            }
        } catch (InterruptedException e) {
            logger.warn("Crawler Process interrupted.");
        } catch (Exception e) {
            throw new FessSystemException("Crawler Process terminated.", e);
        } finally {
            if (currentProcess != null) {
                try {
                    currentProcess.destroy();
                } catch (Exception e) {
                }
            }
            currentProcess = null;
        }
    }

    private void appendJarFile(String cpSeparator,
            ServletContext servletContext, StringBuilder buf,
            String libDirPath, String basePath) {
        File libDir = new File(servletContext.getRealPath(libDirPath));
        File[] jarFiles = libDir.listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return name.toLowerCase().endsWith(".jar");
            }
        });
        if (jarFiles != null) {
            for (File file : jarFiles) {
                buf.append(cpSeparator);
                buf.append(basePath);
                buf.append(file.getName());
            }
        }
    }

    public String getUsername() {
        String username = RequestUtil.getRequest().getRemoteUser();
        if (StringUtil.isBlank(username)) {
            username = "guest";
        }
        return username;
    }

    public Timestamp getCurrentTimestamp() {
        return new Timestamp(new Date().getTime());
    }

    public boolean readyCrawlProcess() {
        return crawlProcessStatus.compareAndSet(false, true);
    }

    public boolean isCrawlProcessRunning() {
        return crawlProcessStatus.get();
    }

    public void finishCrawlProcess() {
        crawlProcessStatus.set(false);
        sessionId = null;
    }

    public boolean readyReplicationProcess() {
        return replicationProcessStatus.compareAndSet(false, true);
    }

    public boolean isReplicationProcessRunning() {
        return replicationProcessStatus.get();
    }

    public void finishReplicationProcess() {
        replicationProcessStatus.set(false);
    }

    public String getSessionId() {
        return sessionId;
    }

    public void setSessionId(String sessionId) {
        this.sessionId = sessionId;
    }

    public File getSnapshotDir(String path) {
        File file = new File(path);
        if (!file.getName().contains("*")) {
            return file;
        }
        File targetDir = null;
        final String dirName = file.getName().replaceAll("\\.", "\\\\.")
                .replaceAll("\\*", ".*");
        for (File f : file.getParentFile().listFiles(new FilenameFilter() {
            @Override
            public boolean accept(File dir, String name) {
                return name.matches(dirName);
            }
        })) {
            if (targetDir == null
                    || targetDir.lastModified() < f.lastModified()) {
                targetDir = f;
            }
        }
        if (targetDir != null) {
            return targetDir;
        }
        return file;
    }

    public boolean isForceStop() {
        return forceStop;
    }

    public void setForceStop(boolean forceStop) {
        this.forceStop = forceStop;
        if (forceStop) {
            try {
                SingletonS2Container.getComponent(WebIndexHelper.class)
                        .stopProcesses();
                SingletonS2Container.getComponent(FileSystemIndexHelper.class)
                        .stopProcesses();
            } catch (Exception e) {
            }
        }
        if (currentProcess != null) {
            try {
                currentProcess.destroy();
            } catch (Exception e) {
            }
        }
    }

    public String getLogFilePath() {
        return logFilePath;
    }

    public void setLogFilePath(String logFilePath) {
        this.logFilePath = logFilePath;
    }

    public String getAdminRole() {
        return adminRole;
    }

    public void setAdminRole(String adminRole) {
        this.adminRole = adminRole;
    }

    public List<String> getAuthenticatedRoleList() {
        return authenticatedRoleList;
    }

    public void setAuthenticatedRoleList(List<String> authenticatedRoleList) {
        this.authenticatedRoleList = authenticatedRoleList;
    }

    public void setAuthenticatedRoles(String roles) {
        if (StringUtil.isNotBlank(roles)) {
            String[] values = roles.split(",");
            authenticatedRoleList = new ArrayList<String>();
            for (String value : values) {
                if (StringUtil.isNotBlank(value)) {
                    authenticatedRoleList.add(value.trim());
                }
            }
        }
    }

    public String[] getCrawlerJavaOptions() {
        return crawlerJavaOptions;
    }

    public void setCrawlerJavaOptions(String[] crawlerJavaOptions) {
        this.crawlerJavaOptions = crawlerJavaOptions;
    }

    public String getSolrHome() {
        return solrHome;
    }

    public void setSolrHome(String solrHome) {
        this.solrHome = solrHome;
    }

    public String getSolrDataDir() {
        return solrDataDir;
    }

    public void setSolrDataDir(String solrDataDir) {
        this.solrDataDir = solrDataDir;
    }

    protected static class InputStreamThread extends Thread {

        private BufferedReader br;

        private List<String> list = new LinkedList<String>();

        private int maxLineBuffer = 1000;

        public InputStreamThread(InputStream is, String charset) {
            try {
                br = new BufferedReader(new InputStreamReader(is, charset));
            } catch (UnsupportedEncodingException e) {
                throw new FessSystemException(e);
            }
        }

        @Override
        public void run() {
            for (;;) {
                try {
                    String line = br.readLine();
                    if (line == null) {
                        break;
                    }
                    if (logger.isDebugEnabled()) {
                        logger.debug(line);
                    }
                    list.add(line);
                    if (list.size() > maxLineBuffer) {
                        list.remove(0);
                    }
                } catch (IOException e) {
                    throw new FessSystemException(e);
                }
            }
        }

        public String getOutput() {
            StringBuilder buf = new StringBuilder();
            for (String value : list) {
                buf.append(value).append("\n");
            }
            return buf.toString();
        }
    }

    public String getJavaCommandPath() {
        return javaCommandPath;
    }

    public void setJavaCommandPath(String javaCommandPath) {
        this.javaCommandPath = javaCommandPath;
    }

    public String encodeUrlFilter(String path) {
        if (filterPathEncoding == null) {
            return path;
        }

        try {
            StringBuilder buf = new StringBuilder();
            for (int i = 0; i < path.length(); i++) {
                char c = path.charAt(i);
                if (CharUtil.isUrlChar(c)) {
                    buf.append(c);
                } else {
                    buf.append(URLEncoder.encode(String.valueOf(c),
                            filterPathEncoding));
                }
            }
            return buf.toString();
        } catch (UnsupportedEncodingException e) {
            return path;
        }
    }
}
