/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.db.exentity.FileCrawlingConfig;
import jp.sf.fess.interval.FessIntervalController;
import jp.sf.fess.service.FileCrawlingConfigService;
import jp.sf.fess.solr.IndexUpdater;
import jp.sf.fess.solr.SolrServerGroup;
import jp.sf.fess.util.FessProperties;

import org.seasar.framework.container.SingletonS2Container;
import org.seasar.framework.util.StringUtil;
import org.seasar.robot.S2Robot;
import org.seasar.robot.S2RobotContext;
import org.seasar.robot.service.DataService;
import org.seasar.robot.service.UrlFilterService;
import org.seasar.robot.service.UrlQueueService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileSystemIndexHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    private static final Logger logger = LoggerFactory
            .getLogger(FileSystemIndexHelper.class);

    @Resource
    protected FessProperties crawlerProperties;

    @Resource
    protected FileCrawlingConfigService fileCrawlingConfigService;

    @Resource
    protected CrawlingConfigHelper crawlingConfigHelper;

    public long maxAccessCount = 100000;

    public long crawlingExecutionInterval = Constants.DEFAULT_CRAWLING_EXECUTION_INTERVAL;

    public int indexUpdaterPriority = Thread.MAX_PRIORITY;

    public int crawlerPriority = Thread.NORM_PRIORITY;

    private List<S2Robot> s2RobotList = Collections
            .synchronizedList(new ArrayList<S2Robot>());

    public void crawl(String sessionId, SolrServerGroup solrServerGroup) {
        List<FileCrawlingConfig> configList = fileCrawlingConfigService
                .getAllFileCrawlingConfigList();

        if (configList.isEmpty()) {
            // nothing
            if (logger.isInfoEnabled()) {
                logger.info("No crawling target urls.");
            }
            return;
        }

        int multiprocessCrawlingCount = 5;
        String value = crawlerProperties.getProperty(
                Constants.CRAWLING_THREAD_COUNT_PROPERTY, "5");
        try {
            multiprocessCrawlingCount = Integer.parseInt(value);
        } catch (NumberFormatException e) {
            // NOP
        }

        long commitPerCount = Constants.DEFAULT_COMMIT_PER_COUNT;
        value = crawlerProperties.getProperty(
                Constants.COMMIT_PER_COUNT_PROPERTY, Long
                        .toString(Constants.DEFAULT_COMMIT_PER_COUNT));
        try {
            commitPerCount = Long.parseLong(value);
        } catch (NumberFormatException e) {
            // NOP
        }

        SystemHelper systemHelper = SingletonS2Container
                .getComponent("systemHelper");

        long startTime = System.currentTimeMillis();

        int count = 0;
        List<String> sessionIdList = new ArrayList<String>();
        s2RobotList.clear();
        crawlingConfigHelper.init();
        List<String> s2RobotStatusList = new ArrayList<String>();
        for (FileCrawlingConfig fileCrawlingConfig : configList) {
            count++;
            String sid = sessionId + "-" + count;

            crawlingConfigHelper.setCrawlingConfig(sid, fileCrawlingConfig);

            // create s2robot
            S2Robot s2Robot = SingletonS2Container.getComponent(S2Robot.class);
            s2Robot.setSessionId(sid);
            sessionIdList.add(sid);

            String pathsStr = fileCrawlingConfig.getPaths();
            if (StringUtil.isBlank(pathsStr)) {
                logger.warn("No target uris. Skipped");
                break;
            }

            int intervalTime = fileCrawlingConfig.getIntervalTime() != null ? fileCrawlingConfig
                    .getIntervalTime()
                    : Constants.DEFAULT_INTERVAL_TIME_FOR_FS;
            ((FessIntervalController) s2Robot.getIntervalController())
                    .setDelayMillisForWaitingNewUrl(intervalTime);

            String includedPathsStr = fileCrawlingConfig.getIncludedPaths();
            String excludedPathsStr = fileCrawlingConfig.getExcludedPaths();

            S2RobotContext robotContext = s2Robot.getRobotContext();
            int numOfThread = fileCrawlingConfig.getNumOfThread() != null ? fileCrawlingConfig
                    .getNumOfThread()
                    : Constants.DEFAULT_NUM_OF_THREAD_FOR_FS;
            robotContext.setNumOfThread(numOfThread);

            int depth = fileCrawlingConfig.getDepth() != null ? fileCrawlingConfig
                    .getDepth()
                    : -1;
            robotContext.setMaxDepth(depth);

            long maxCount = fileCrawlingConfig.getMaxAccessCount() != null ? fileCrawlingConfig
                    .getMaxAccessCount()
                    : maxAccessCount;
            robotContext.setMaxAccessCount(maxCount);

            // set paths
            String[] paths = pathsStr.split("[\r\n]");
            for (String u : paths) {
                if (StringUtil.isNotBlank(u)) {
                    u = u.trim();
                    if (!u.startsWith("file:")) {
                        if (u.startsWith("/")) {
                            u = "file:" + u;
                        } else {
                            u = "file:/" + u;
                        }
                    }
                    s2Robot.addUrl(u);
                }
            }

            // set included paths
            String[] includedPaths = includedPathsStr.split("[\r\n]");
            for (String u : includedPaths) {
                if (StringUtil.isNotBlank(u)) {
                    s2Robot.addIncludeFilter(systemHelper.encodeUrlFilter(u
                            .trim()));
                }
            }

            // set excluded paths
            String[] excludedPaths = excludedPathsStr.split("[\r\n]");
            for (String u : excludedPaths) {
                if (StringUtil.isNotBlank(u)) {
                    s2Robot.addExcludeFilter(systemHelper.encodeUrlFilter(u
                            .trim()));
                }
            }

            if (logger.isDebugEnabled()) {
                logger.debug("Crawling " + pathsStr);
            }

            s2Robot.setBackground(true);
            s2Robot.setThreadPriority(crawlerPriority);

            s2RobotList.add(s2Robot);
            s2RobotStatusList.add(Constants.READY);
        }

        // run index update
        IndexUpdater indexUpdater = SingletonS2Container
                .getComponent("indexUpdater");
        indexUpdater.setName("IndexUpdater");
        indexUpdater.setPriority(indexUpdaterPriority);
        indexUpdater.setSessionIdList(sessionIdList);
        indexUpdater.setSolrServerGroup(solrServerGroup);
        indexUpdater.setDaemon(true);
        indexUpdater.setCommitPerCount(commitPerCount);
        indexUpdater.start();

        int startedCrawlerNum = 0;
        int activeCrawlerNum = 0;
        while (startedCrawlerNum < s2RobotList.size()) {
            // Force to stop crawl
            if (systemHelper.isForceStop()) {
                for (S2Robot s2Robot : s2RobotList) {
                    s2Robot.stop();
                }
                break;
            }

            if (activeCrawlerNum < multiprocessCrawlingCount) {
                // start crawling
                s2RobotList.get(startedCrawlerNum).execute();
                s2RobotStatusList.set(startedCrawlerNum, Constants.RUNNING);
                startedCrawlerNum++;
                activeCrawlerNum++;
                try {
                    Thread.sleep(crawlingExecutionInterval);
                } catch (InterruptedException e) {
                    // NOP
                }
                continue;
            }

            // check status
            for (int i = 0; i < startedCrawlerNum; i++) {
                if (!s2RobotList.get(i).getRobotContext().isRunning()
                        && s2RobotStatusList.get(i).equals(Constants.RUNNING)) {
                    s2RobotList.get(i).awaitTermination();
                    s2RobotStatusList.set(i, Constants.DONE);
                    String sid = s2RobotList.get(i).getRobotContext()
                            .getSessionId();
                    indexUpdater.addFinishedSessionId(sid);
                    activeCrawlerNum--;
                }
            }
            try {
                Thread.sleep(crawlingExecutionInterval);
            } catch (InterruptedException e) {
                // NOP
            }
        }

        boolean finishedAll = false;
        while (!finishedAll) {
            finishedAll = true;
            for (int i = 0; i < s2RobotList.size(); i++) {
                s2RobotList.get(i).awaitTermination(crawlingExecutionInterval);
                if (!s2RobotList.get(i).getRobotContext().isRunning()
                        && s2RobotStatusList.get(i).equals(Constants.RUNNING)) {
                    s2RobotStatusList.set(i, Constants.DONE);
                    String sid = s2RobotList.get(i).getRobotContext()
                            .getSessionId();
                    indexUpdater.addFinishedSessionId(sid);
                }
                if (!s2RobotStatusList.get(i).equals(Constants.DONE)) {
                    finishedAll = false;
                }
            }
        }
        s2RobotList.clear();
        s2RobotStatusList.clear();

        Map<String, String> infoMap = new HashMap<String, String>();

        long execTime = System.currentTimeMillis() - startTime;
        infoMap.put(Constants.FS_CRAWLING_EXEC_TIME, Long.toString(execTime));
        if (logger.isInfoEnabled()) {
            logger.info("[EXEC TIME] crawling time: " + execTime + "ms");
        }

        indexUpdater.setFinishCrawling(true);
        try {
            indexUpdater.join();
        } catch (InterruptedException e) {
            logger.warn("Interrupted index update.", e);
        }

        infoMap.put(Constants.FS_INDEX_EXEC_TIME, Long.toString(indexUpdater
                .getExecuteTime()));
        infoMap.put(Constants.FS_INDEX_SIZE, Long.toString(indexUpdater
                .getDocumentSize()));

        // store info map
        CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                .getComponent("crawlingSessionHelper");
        crawlingSessionHelper.put(sessionId, infoMap);

        for (String sid : sessionIdList) {
            // remove config
            crawlingConfigHelper.setCrawlingConfig(sid, null);
        }

        // clear url filter
        UrlFilterService urlFilterService = SingletonS2Container
                .getComponent(UrlFilterService.class);
        urlFilterService.deleteAll();

        // clear queue
        UrlQueueService urlQueueService = SingletonS2Container
                .getComponent(UrlQueueService.class);
        urlQueueService.deleteAll();

        // clear
        DataService dataService = SingletonS2Container
                .getComponent(DataService.class);
        dataService.deleteAll();

    }

    /**
     * Stop processes.
     * (This method is not MT-safe.)
     * 
     */
    public void stopProcesses() {
        synchronized (s2RobotList) {
            for (S2Robot s2Robot : s2RobotList) {
                s2Robot.stop();
            }
        }
    }
}
