/*
 * Copyright 2009-2009 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.helper;

import java.io.Serializable;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import jp.sf.fess.Constants;
import jp.sf.fess.FessSystemException;
import jp.sf.fess.entity.Document;
import jp.sf.fess.util.FessProperties;

import org.apache.commons.lang.StringUtils;
import org.seasar.framework.util.StringUtil;
import org.seasar.struts.taglib.S2Functions;
import org.seasar.struts.util.RequestUtil;

public class ViewHelper implements Serializable {

    private static final long serialVersionUID = 1L;

    protected static final String GOOGLE_MOBILE_TRANSCODER_LINK = "http://www.google.co.jp/gwt/n?u=";

    private static final String USER_AGENT_TYPE = "ViewHelper.UserAgent";

    private static final String UA_FIREFOX = "Firefox";

    private static final String UA_OPERA = "Opera";

    private static final String UA_OTHER = "Other";

    @Resource
    protected BrowserTypeHelper browserTypeHelper;

    @Resource
    protected FessProperties crawlerProperties;

    public int mobileDescriptionLength = 50;

    public int pcDescriptionLength = 200;

    public int mobileTitleLength = 50;

    public int pcTitleLength = 50;

    public String encoding = Constants.UTF_8;

    public boolean useDigest = true;

    public String getContentTitle(Document document) {
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileTitleLength;
        } else {
            size = pcTitleLength;
        }

        String title;
        if (StringUtil.isNotBlank(document.title)) {
            title = document.title;
        } else {
            title = document.url;
        }
        return StringUtils.abbreviate(title, size);
    }

    public String getContentDescription(Document document) {
        String[] queries = (String[]) RequestUtil.getRequest().getAttribute(
                Constants.QUERIES);
        int size;
        if (browserTypeHelper.isMobile()) {
            size = mobileDescriptionLength;
        } else {
            size = pcDescriptionLength;
        }

        if (StringUtil.isNotBlank(document.digest) && useDigest) {
            return highlight(S2Functions.h(StringUtils.abbreviate(
                    document.digest, size)), queries);
        }

        if (document.cache != null) {
            if (queries != null && queries.length > 0) {
                int idx = document.cache.indexOf(queries[0]) - 10;
                return highlight(S2Functions.h(StringUtils.abbreviate(
                        document.cache, idx, size)), queries);
            } else {
                return S2Functions.h(StringUtils.abbreviate(document.cache,
                        size));
            }
        }
        return "";
    }

    protected String highlight(String content, String[] queries) {
        if (StringUtil.isBlank(content) || queries == null
                || queries.length == 0) {
            return content;
        }
        for (String query : queries) {
            content = content.replaceAll(query, "<em>" + query + "</em>");
        }
        return content;
    }

    public String getUrlLink(Document document) {
        // file protocol
        if (document.url != null && document.url.startsWith("file:")) {
            HttpServletRequest request = RequestUtil.getRequest();
            String uaType = (String) request.getAttribute(USER_AGENT_TYPE);
            if (uaType == null) {
                String userAgent = request.getHeader("user-agent");
                if (userAgent != null) {
                    if (userAgent.indexOf("Firefox") >= 0) {
                        uaType = UA_FIREFOX;
                    } else if (userAgent.indexOf("Opera") >= 0) {
                        uaType = UA_OPERA;
                    }
                }
                if (uaType == null) {
                    uaType = UA_OTHER;
                }
                request.setAttribute(USER_AGENT_TYPE, uaType);
            }

            String url = document.url;
            if (UA_FIREFOX.equals(uaType)) {
                url = url.replaceFirst("file:////", "file://///");
            } else if (UA_OPERA.equals(uaType)) {
                url = url.replaceFirst("file:////", "file://");
            }
            try {
                return URLDecoder.decode(url, encoding);
            } catch (UnsupportedEncodingException e) {
                throw new FessSystemException("Unsupported encoding: "
                        + encoding);
            }
        }

        if (browserTypeHelper.isMobile()) {
            String mobileTrasncoder = crawlerProperties.getProperty(
                    Constants.MOBILE_TRANSCODER_PROPERTY,
                    Constants.EMPTY_STRING);
            if (Constants.GOOGLE_MOBILE_TRANSCODER.equals(mobileTrasncoder)) {
                return getGoogleMobileTranscoderLink(document);
            }
        }
        return document.url;
    }

    protected String getGoogleMobileTranscoderLink(Document document) {
        StringBuilder buf = new StringBuilder(255);
        buf.append(GOOGLE_MOBILE_TRANSCODER_LINK);
        try {
            buf.append(URLEncoder.encode(document.url, Constants.UTF_8));
        } catch (UnsupportedEncodingException e) {
            return document.url;
        }
        return buf.toString();
    }
}
