/*
 * Copyright 2009-2011 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */

package jp.sf.fess.helper.impl;

import java.util.ArrayList;
import java.util.List;

import jp.sf.fess.Constants;
import jp.sf.fess.entity.SearchQuery;
import jp.sf.fess.helper.BrowserTypeHelper;
import jp.sf.fess.helper.RoleQueryHelper;

import org.seasar.extension.unit.S2TestCase;

public class QueryHelperImplTest extends S2TestCase {

    public QueryHelperImpl queryHelperImpl;

    @Override
    protected String getRootDicon() throws Throwable {
        return "jp/sf/fess/helper/query.dicon";
    }

    public void test_build() {
        assertEquals("", queryHelperImpl.buildQuery("").getQuery());

        assertEquals("(title:QUERY OR content:QUERY)", queryHelperImpl
                .buildQuery("QUERY").getQuery());
        assertEquals("(title:QUERY OR content:QUERY)", queryHelperImpl
                .buildQuery("QUERY ").getQuery());
        assertEquals("(title:QUERY OR content:QUERY)", queryHelperImpl
                .buildQuery(" QUERY").getQuery());

        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1 QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1 QUERY2 ").getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery(" QUERY1 QUERY2").getQuery());

        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\"").getQuery());
        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\" ").getQuery());
        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildQuery(" \"QUERY1 QUERY2\"").getQuery());

        assertEquals(
                "((title:QUERY1\\ QUERY2 AND title:QUERY3) OR (content:QUERY1\\ QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\" QUERY3")
                        .getQuery());

    }

    public void test_build_fullwidthSpace() {

        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1\u3000QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1\u3000QUERY2\u3000")
                        .getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("\u3000QUERY1\u3000QUERY2")
                        .getQuery());

        assertEquals(
                "(title:QUERY1\\\u3000QUERY2 OR content:QUERY1\\\u3000QUERY2)",
                queryHelperImpl.buildQuery("\"QUERY1\u3000QUERY2\"").getQuery());

        assertEquals(
                "((title:QUERY1\\\u3000QUERY2 AND title:QUERY3) OR (content:QUERY1\\\u3000QUERY2 AND content:QUERY3))",
                queryHelperImpl
                        .buildQuery("\"QUERY1\u3000QUERY2\"\u3000QUERY3")
                        .getQuery());

    }

    public void test_build_prefix() {
        assertEquals("site:QUERY1", queryHelperImpl.buildQuery("site:QUERY1")
                .getQuery());
        assertEquals("(site:QUERY1 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1 QUERY2").getQuery());
        assertEquals(
                "(site:QUERY1 AND ((title:QUERY2 AND title:QUERY3) OR (content:QUERY2 AND content:QUERY3)))",
                queryHelperImpl.buildQuery("site:QUERY1 QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "(site:QUERY1 AND host:QUERY2 AND (title:QUERY3 OR content:QUERY3))",
                queryHelperImpl.buildQuery("site:QUERY1 host:QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "(site:QUERY1\\ QUERY2 AND (title:QUERY3 OR content:QUERY3))",
                queryHelperImpl.buildQuery("site:\"QUERY1 QUERY2\" QUERY3")
                        .getQuery());

    }

    public void test_build_prefix_unknown() {
        assertEquals("(title:site\\: OR content:site\\:)", queryHelperImpl
                .buildQuery("site:").getQuery());
        assertEquals("(title:hoge\\:QUERY1 OR content:hoge\\:QUERY1)",
                queryHelperImpl.buildQuery("hoge:QUERY1").getQuery());
        assertEquals(
                "((title:hoge\\:QUERY1 AND title:QUERY2) OR (content:hoge\\:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("hoge:QUERY1 QUERY2").getQuery());
        assertEquals(
                "((title:hoge\\:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:hoge\\:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("hoge:QUERY1 QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "(host:QUERY2 AND ((title:hoge\\:QUERY1 AND title:QUERY3) OR (content:hoge\\:QUERY1 AND content:QUERY3)))",
                queryHelperImpl.buildQuery("hoge:QUERY1 host:QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "((title:hoge\\:QUERY1\\ QUERY2 AND title:QUERY3) OR (content:hoge\\:QUERY1\\ QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("hoge:\"QUERY1 QUERY2\" QUERY3")
                        .getQuery());

    }

    public void test_build_browserType() {
        queryHelperImpl.browserTypeHelper = new BrowserTypeHelper() {
            private static final long serialVersionUID = 1L;

            @Override
            public String getBrowserType() {
                return PC;
            }
        };

        assertEquals("", queryHelperImpl.build("").getQuery());

        assertEquals("(title:QUERY OR content:QUERY) AND type:pc",
                queryHelperImpl.build("QUERY").getQuery());

        queryHelperImpl.browserTypeHelper = new BrowserTypeHelper() {
            private static final long serialVersionUID = 1L;

            @Override
            public String getBrowserType() {
                return DOCOMO;
            }
        };

        assertEquals("(title:QUERY OR content:QUERY) AND type:docomo",
                queryHelperImpl.build("QUERY").getQuery());
    }

    public void test_build_roleType() {
        queryHelperImpl.roleQueryHelper = new RoleQueryHelper() {
            @Override
            public List<String> build() {
                List<String> list = new ArrayList<String>();
                list.add("guest");
                return list;
            }
        };

        assertEquals("", queryHelperImpl.build("").getQuery());

        assertEquals("(title:QUERY OR content:QUERY) AND role:guest",
                queryHelperImpl.build("QUERY").getQuery());

        queryHelperImpl.roleQueryHelper = new RoleQueryHelper() {
            @Override
            public List<String> build() {
                List<String> list = new ArrayList<String>();
                list.add("guest");
                list.add("admin");
                return list;
            }
        };

        assertEquals(
                "(title:QUERY OR content:QUERY) AND (role:guest OR role:admin)",
                queryHelperImpl.build("QUERY").getQuery());
    }

    public void test_build_browserType_and_roleType() {
        queryHelperImpl.browserTypeHelper = new BrowserTypeHelper() {
            private static final long serialVersionUID = 1L;

            @Override
            public String getBrowserType() {
                return PC;
            }
        };
        queryHelperImpl.roleQueryHelper = new RoleQueryHelper() {
            @Override
            public List<String> build() {
                List<String> list = new ArrayList<String>();
                list.add("guest");
                list.add("admin");
                return list;
            }
        };

        assertEquals("", queryHelperImpl.build("").getQuery());

        assertEquals(
                "(title:QUERY OR content:QUERY) AND type:pc AND (role:guest OR role:admin)",
                queryHelperImpl.build("QUERY").getQuery());
    }

    public void test_sortField() {
        String query;
        SearchQuery searchQuery;

        query = "";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(0, searchQuery.getSortFields().length);

        query = "sort:contentLength";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());

        query = "sort:contentLength.desc";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.DESC, searchQuery.getSortFields()[0].getOrder());

        query = "sort:contentLength.asc,lastModified";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(2, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());
        assertEquals("lastModified", searchQuery.getSortFields()[1].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[1].getOrder());

        query = "QUERY sort:contentLength";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());

        query = "QUERY sort:contentLength.desc";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.DESC, searchQuery.getSortFields()[0].getOrder());

        query = "QUERY sort:contentLength.asc,lastModified";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery.getQuery());
        assertEquals(2, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());
        assertEquals("lastModified", searchQuery.getSortFields()[1].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[1].getOrder());

        query = "QUERY site:QUERY1 sort:contentLength";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());

        query = "QUERY sort:contentLength.desc  site:QUERY1";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.DESC, searchQuery.getSortFields()[0].getOrder());

        query = "QUERY sort:contentLength.asc,lastModified site:QUERY1";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery.getQuery());
        assertEquals(2, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());
        assertEquals("lastModified", searchQuery.getSortFields()[1].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[1].getOrder());
    }

    public void test_sortField_invalid() {
        String query;
        SearchQuery searchQuery;

        query = "sort:hoge";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(0, searchQuery.getSortFields().length);

        query = "sort:contentLength.hoge";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(0, searchQuery.getSortFields().length);

        query = "sort:contentLength.asc,hoge";
        searchQuery = queryHelperImpl.buildQuery(query);
        assertEquals("", searchQuery.getQuery());
        assertEquals(1, searchQuery.getSortFields().length);
        assertEquals("contentLength", searchQuery.getSortFields()[0].getField());
        assertEquals(Constants.ASC, searchQuery.getSortFields()[0].getOrder());
    }

    public void test_wildcardSearches() {
        // *

        assertEquals("(title:\\* OR content:\\*)",
                queryHelperImpl.buildQuery("*").getQuery());
        assertEquals("(title:QUERY* OR content:QUERY*)", queryHelperImpl
                .buildQuery("QUERY* ").getQuery());
        assertEquals("(title:Q*ERY OR content:Q*ERY)", queryHelperImpl
                .buildQuery(" Q*ERY").getQuery());

        assertEquals(
                "((title:Q*ERY1 AND title:Q*ERY2) OR (content:Q*ERY1 AND content:Q*ERY2))",
                queryHelperImpl.buildQuery("Q*ERY1 Q*ERY2").getQuery());

        assertEquals("(title:Q*ERY1\\ Q*ERY2 OR content:Q*ERY1\\ Q*ERY2)",
                queryHelperImpl.buildQuery("\"Q*ERY1 Q*ERY2\"").getQuery());

        assertEquals(
                "((title:Q*ERY1\\ Q*ERY2 AND title:Q*ERY3) OR (content:Q*ERY1\\ Q*ERY2 AND content:Q*ERY3))",
                queryHelperImpl.buildQuery("\"Q*ERY1 Q*ERY2\" Q*ERY3")
                        .getQuery());

        assertEquals("site:QUERY1*", queryHelperImpl.buildQuery("site:QUERY1*")
                .getQuery());
        assertEquals("(site:QUERY1* AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1* QUERY2").getQuery());

        // ?

        assertEquals("(title:\\? OR content:\\?)",
                queryHelperImpl.buildQuery("?").getQuery());
        assertEquals("(title:QUERY? OR content:QUERY?)", queryHelperImpl
                .buildQuery("QUERY? ").getQuery());
        assertEquals("(title:Q?ERY OR content:Q?ERY)", queryHelperImpl
                .buildQuery(" Q?ERY").getQuery());

        assertEquals(
                "((title:Q?ERY1 AND title:Q?ERY2) OR (content:Q?ERY1 AND content:Q?ERY2))",
                queryHelperImpl.buildQuery("Q?ERY1 Q?ERY2").getQuery());

        assertEquals("(title:Q?ERY1\\ Q?ERY2 OR content:Q?ERY1\\ Q?ERY2)",
                queryHelperImpl.buildQuery("\"Q?ERY1 Q?ERY2\"").getQuery());

        assertEquals(
                "((title:Q?ERY1\\ Q?ERY2 AND title:Q?ERY3) OR (content:Q?ERY1\\ Q?ERY2 AND content:Q?ERY3))",
                queryHelperImpl.buildQuery("\"Q?ERY1 Q?ERY2\" Q?ERY3")
                        .getQuery());

        assertEquals("site:QUERY1?", queryHelperImpl.buildQuery("site:QUERY1?")
                .getQuery());
        assertEquals("(site:QUERY1? AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1? QUERY2").getQuery());
    }

    public void test_fuzzySearches() {
        // ~

        assertEquals("(title:QUERY~ OR content:QUERY~)", queryHelperImpl
                .buildQuery("QUERY~").getQuery());
        assertEquals(
                "((title:QUERY1~ AND title:QUERY2) OR (content:QUERY1~ AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1~ QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1~ AND title:QUERY2~) OR (content:QUERY1~ AND content:QUERY2~))",
                queryHelperImpl.buildQuery("QUERY1~ QUERY2~").getQuery());

        assertEquals("site:QUERY1~", queryHelperImpl.buildQuery("site:QUERY1~")
                .getQuery());
        assertEquals("(site:QUERY1~ AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1~ QUERY2").getQuery());

        assertEquals("(title:QUERY1\\ QUERY2~ OR content:QUERY1\\ QUERY2~)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\"~").getQuery());
        assertEquals("(title:QUERY1~ OR content:QUERY1~)", queryHelperImpl
                .buildQuery("\"QUERY1~\"").getQuery());

        // ~0.8

        assertEquals("(title:QUERY~0.8 OR content:QUERY~0.8)", queryHelperImpl
                .buildQuery("QUERY~0.8").getQuery());
        assertEquals(
                "((title:QUERY1~0.8 AND title:QUERY2) OR (content:QUERY1~0.8 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1~0.8 QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1~0.5 AND title:QUERY2~0.8) OR (content:QUERY1~0.5 AND content:QUERY2~0.8))",
                queryHelperImpl.buildQuery("QUERY1~0.5 QUERY2~0.8").getQuery());

        assertEquals("site:QUERY1~0.8",
                queryHelperImpl.buildQuery("site:QUERY1~0.8").getQuery());
        assertEquals("(site:QUERY1~0.8 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1~0.8 QUERY2").getQuery());

        assertEquals(
                "(title:QUERY1\\ QUERY2~0.8 OR content:QUERY1\\ QUERY2~0.8)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\"~0.8").getQuery());
        assertEquals("(title:QUERY1~0.8 OR content:QUERY1~0.8)",
                queryHelperImpl.buildQuery("\"QUERY1~0.8\"").getQuery());

        assertEquals("(title:QUERY1~0.8 OR content:QUERY1~0.8)",
                queryHelperImpl.buildQuery("\"QUERY1~0.8a\"").getQuery());
        assertEquals("(title:QUERY1~ OR content:QUERY1~)", queryHelperImpl
                .buildQuery("\"QUERY1~a\"").getQuery());

    }

    public void test_proximitySearches() {
        // ~10
        assertEquals("(title:\"QUERY\"~10 OR content:\"QUERY\"~10)",
                queryHelperImpl.buildQuery("QUERY~10").getQuery());
        assertEquals("(title:\"QUERY\"~1 OR content:\"QUERY\"~1)",
                queryHelperImpl.buildQuery("QUERY~1").getQuery());
        assertEquals("(title:\"QUERY\"~5 OR content:\"QUERY\"~5)",
                queryHelperImpl.buildQuery("QUERY~5.5").getQuery());
        assertEquals(
                "((title:\"QUERY1\"~10 AND title:QUERY2) OR (content:\"QUERY1\"~10 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1~10 QUERY2").getQuery());
        assertEquals(
                "((title:\"QUERY1\"~5 AND title:\"QUERY2\"~10) OR (content:\"QUERY1\"~5 AND content:\"QUERY2\"~10))",
                queryHelperImpl.buildQuery("QUERY1~5 QUERY2~10").getQuery());

        assertEquals("site:\"QUERY1\"~10",
                queryHelperImpl.buildQuery("site:QUERY1~10").getQuery());
        assertEquals(
                "(site:\"QUERY1\"~10 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1~10 QUERY2").getQuery());

        assertEquals(
                "(title:\"QUERY1\\ QUERY2\"~10 OR content:\"QUERY1\\ QUERY2\"~10)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\"~10").getQuery());
        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildQuery("\"QUERY1~10\"").getQuery());

        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildQuery("\"QUERY1~10a\"").getQuery());
        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildQuery("\"QUERY1~10.1.2").getQuery());

    }

    public void test_rangeSearches() {
        // mod_date:[20020101 TO 20030101] 
        assertEquals(
                "(title:[20020101 TO 20030101] OR content:[20020101 TO 20030101])",
                queryHelperImpl.buildQuery("[20020101 TO 20030101]").getQuery());
        assertEquals("lastModified:[20020101 TO 20030101]", queryHelperImpl
                .buildQuery("lastModified:[20020101 TO 20030101]").getQuery());
        assertEquals(
                "(lastModified:[20020101 TO 20030101] AND (title:QUERY OR content:QUERY))",
                queryHelperImpl.buildQuery(
                        "QUERY lastModified:[20020101 TO 20030101]").getQuery());

        // TODO more..

        // title:{Aida TO Carmen}
        assertEquals("(title:{Aida TO Carmen} OR content:{Aida TO Carmen})",
                queryHelperImpl.buildQuery("{Aida TO Carmen}").getQuery());
        assertEquals("lastModified:{Aida TO Carmen}", queryHelperImpl
                .buildQuery("lastModified:{Aida TO Carmen}").getQuery());
        assertEquals(
                "(title:{Aida TO Carmen} AND (title:QUERY OR content:QUERY))",
                queryHelperImpl.buildQuery("QUERY title:{Aida TO Carmen}")
                        .getQuery());

        // TODO more..
    }

    public void test_boosting() {
        // ^1000 ""^1000
        assertEquals("(title:QUERY^1000 OR content:QUERY^1000)",
                queryHelperImpl.buildQuery("QUERY^1000").getQuery());
        assertEquals(
                "((title:QUERY1^1000 AND title:QUERY2) OR (content:QUERY1^1000 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1^1000 QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1^500 AND title:QUERY2^1000) OR (content:QUERY1^500 AND content:QUERY2^1000))",
                queryHelperImpl.buildQuery("QUERY1^500 QUERY2^1000").getQuery());

        assertEquals("site:QUERY1^1000",
                queryHelperImpl.buildQuery("site:QUERY1^1000").getQuery());
        assertEquals("(site:QUERY1^1000 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildQuery("site:QUERY1^1000 QUERY2")
                        .getQuery());

        assertEquals(
                "(title:QUERY1\\ QUERY2^1000 OR content:QUERY1\\ QUERY2^1000)",
                queryHelperImpl.buildQuery("\"QUERY1 QUERY2\"^1000").getQuery());
        assertEquals("(title:QUERY1^1000 OR content:QUERY1^1000)",
                queryHelperImpl.buildQuery("\"QUERY1^1000\"").getQuery());
    }

    public void test_reserved() {
        for (int i = 0; i < Constants.RESERVED.length - 1; i++) {
            assertEquals("(title:\\" + Constants.RESERVED[i] + " OR content:\\"
                    + Constants.RESERVED[i] + ")",
                    queryHelperImpl.buildQuery(Constants.RESERVED[i])
                            .getQuery());
        }
        assertEquals("(title:\\: OR content:\\:)",
                queryHelperImpl.buildQuery(":").getQuery());
    }

    public void test_or() {
        assertEquals("(title:QUERY OR content:QUERY)", queryHelperImpl
                .buildQuery("OR QUERY").getQuery());
        assertEquals(
                "((title:QUERY1 OR title:QUERY2) OR (content:QUERY1 OR content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1 OR QUERY2").getQuery());
        assertEquals("title:QUERY", queryHelperImpl
                .buildQuery("OR title:QUERY").getQuery());
        assertEquals("(title:QUERY1 OR title:QUERY2)", queryHelperImpl
                .buildQuery("title:QUERY1 OR title:QUERY2").getQuery());
        assertEquals("(title:QUERY2 OR (title:QUERY1 OR content:QUERY1))",
                queryHelperImpl.buildQuery("QUERY1 OR title:QUERY2").getQuery());
        assertEquals("(site:QUERY1 OR title:QUERY2)", queryHelperImpl
                .buildQuery("site:QUERY1 OR title:QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1 OR title:QUERY2 AND title:QUERY3) OR (content:QUERY1 OR content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("QUERY1 OR QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "((title:QUERY1 OR title:QUERY2 OR title:QUERY3) OR (content:QUERY1 OR content:QUERY2 OR content:QUERY3))",
                queryHelperImpl.buildQuery("QUERY1 OR QUERY2 OR QUERY3")
                        .getQuery());
    }

    public void test_and() {
        assertEquals("(title:QUERY OR content:QUERY)", queryHelperImpl
                .buildQuery("AND QUERY").getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildQuery("QUERY1 AND QUERY2").getQuery());
        assertEquals("title:QUERY",
                queryHelperImpl.buildQuery("AND title:QUERY").getQuery());
        assertEquals("(title:QUERY1 AND title:QUERY2)", queryHelperImpl
                .buildQuery("title:QUERY1 AND title:QUERY2").getQuery());
        assertEquals("(title:QUERY2 AND (title:QUERY1 OR content:QUERY1))",
                queryHelperImpl.buildQuery("QUERY1 AND title:QUERY2")
                        .getQuery());
        assertEquals("(site:QUERY1 AND title:QUERY2)", queryHelperImpl
                .buildQuery("site:QUERY1 AND title:QUERY2").getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("QUERY1 AND QUERY2 QUERY3")
                        .getQuery());
        assertEquals(
                "((title:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildQuery("QUERY1 AND QUERY2 AND QUERY3")
                        .getQuery());
    }

    public void test_not() {
        assertEquals("NOT title:QUERY AND NOT content:QUERY", queryHelperImpl
                .buildQuery("NOT QUERY").getQuery());
        assertEquals("NOT title:QUERY",
                queryHelperImpl.buildQuery("NOT title:QUERY").getQuery());
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildQuery("NOT QUERY1 QUERY2").getQuery());
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildQuery("NOT QUERY1 OR QUERY2").getQuery());
        assertEquals(
                "NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY2 AND NOT content:QUERY2",
                queryHelperImpl.buildQuery("NOT QUERY1 NOT QUERY2").getQuery());
        assertEquals(
                "NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY2 AND NOT content:QUERY2",
                queryHelperImpl.buildQuery("NOT QUERY1 OR NOT QUERY2")
                        .getQuery());
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY3 AND NOT content:QUERY3",
                queryHelperImpl.buildQuery("NOT QUERY1 QUERY2 NOT QUERY3")
                        .getQuery());
        assertEquals("NOT site:QUERY",
                queryHelperImpl.buildQuery("NOT site:QUERY").getQuery());
        assertEquals("NOT site:QUERY1 AND NOT title:QUERY2", queryHelperImpl
                .buildQuery("NOT site:QUERY1 NOT title:QUERY2").getQuery());
        assertEquals("(title:QUERY2 OR content:QUERY2) AND NOT site:QUERY1",
                queryHelperImpl.buildQuery("NOT site:QUERY1 QUERY2").getQuery());
        assertEquals("site:QUERY2 AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildQuery("NOT QUERY1 site:QUERY2").getQuery());

        assertEquals("(title:QUERY AND (title:\\(NOT OR content:\\(NOT))",
                queryHelperImpl.buildQuery("(NOT title:QUERY").getQuery());
    }

    public void test_escapeValue() {
        String[] targets = new String[] { "+", "-", "&&", "||", "!", "(", ")",
                "{", "}", "[", "]", "^", "\"", "~", ":", "\\", " ", "\u3000" };
        for (String target : targets) {
            assertEquals("abc\\" + target + "123",
                    queryHelperImpl.escapeValue("abc" + target + "123"));
        }
        for (String target : targets) {
            assertEquals("abc\\" + target,
                    queryHelperImpl.escapeValue("abc" + target));
        }
        for (String target : targets) {
            assertEquals("\\" + target + "123",
                    queryHelperImpl.escapeValue(target + "123"));
        }
        for (String target : targets) {
            assertEquals(
                    "abc\\" + target + "123\\" + target + "ABC",
                    queryHelperImpl.escapeValue("abc" + target + "123" + target
                            + "ABC"));
        }
    }

    public void test_buildFacet() {
        assertEquals("", queryHelperImpl.buildFacetQuery(""));

        assertEquals("(title:QUERY OR content:QUERY)",
                queryHelperImpl.buildFacetQuery("QUERY"));
        assertEquals("(title:QUERY OR content:QUERY)",
                queryHelperImpl.buildFacetQuery("QUERY "));
        assertEquals("(title:QUERY OR content:QUERY)",
                queryHelperImpl.buildFacetQuery(" QUERY"));

        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1 QUERY2"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1 QUERY2 "));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery(" QUERY1 QUERY2"));

        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\""));
        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\" "));
        assertEquals("(title:QUERY1\\ QUERY2 OR content:QUERY1\\ QUERY2)",
                queryHelperImpl.buildFacetQuery(" \"QUERY1 QUERY2\""));

        assertEquals(
                "((title:QUERY1\\ QUERY2 AND title:QUERY3) OR (content:QUERY1\\ QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\" QUERY3"));
    }

    public void test_buildFacet_fullwidthSpace() {

        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1\u3000QUERY2"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1\u3000QUERY2\u3000"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("\u3000QUERY1\u3000QUERY2"));

        assertEquals(
                "(title:QUERY1\\\u3000QUERY2 OR content:QUERY1\\\u3000QUERY2)",
                queryHelperImpl.buildFacetQuery("\"QUERY1\u3000QUERY2\""));

        assertEquals(
                "((title:QUERY1\\\u3000QUERY2 AND title:QUERY3) OR (content:QUERY1\\\u3000QUERY2 AND content:QUERY3))",
                queryHelperImpl
                        .buildFacetQuery("\"QUERY1\u3000QUERY2\"\u3000QUERY3"));

    }

    public void test_buildFacet_prefix() {
        assertEquals("site:QUERY1",
                queryHelperImpl.buildFacetQuery("site:QUERY1"));
        assertEquals("(site:QUERY1 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1 QUERY2"));
        assertEquals(
                "(site:QUERY1 AND ((title:QUERY2 AND title:QUERY3) OR (content:QUERY2 AND content:QUERY3)))",
                queryHelperImpl.buildFacetQuery("site:QUERY1 QUERY2 QUERY3"));
        assertEquals(
                "(site:QUERY1 AND host:QUERY2 AND (title:QUERY3 OR content:QUERY3))",
                queryHelperImpl
                        .buildFacetQuery("site:QUERY1 host:QUERY2 QUERY3"));
        assertEquals(
                "(site:QUERY1\\ QUERY2 AND (title:QUERY3 OR content:QUERY3))",
                queryHelperImpl
                        .buildFacetQuery("site:\"QUERY1 QUERY2\" QUERY3"));

    }

    public void test_buildFacet_prefix_unknown() {
        assertEquals("(title:site\\: OR content:site\\:)",
                queryHelperImpl.buildFacetQuery("site:"));
        assertEquals("(title:hoge\\:QUERY1 OR content:hoge\\:QUERY1)",
                queryHelperImpl.buildFacetQuery("hoge:QUERY1"));
        assertEquals(
                "((title:hoge\\:QUERY1 AND title:QUERY2) OR (content:hoge\\:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("hoge:QUERY1 QUERY2"));
        assertEquals(
                "((title:hoge\\:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:hoge\\:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildFacetQuery("hoge:QUERY1 QUERY2 QUERY3"));
        assertEquals(
                "(host:QUERY2 AND ((title:hoge\\:QUERY1 AND title:QUERY3) OR (content:hoge\\:QUERY1 AND content:QUERY3)))",
                queryHelperImpl
                        .buildFacetQuery("hoge:QUERY1 host:QUERY2 QUERY3"));
        assertEquals(
                "((title:hoge\\:QUERY1\\ QUERY2 AND title:QUERY3) OR (content:hoge\\:QUERY1\\ QUERY2 AND content:QUERY3))",
                queryHelperImpl
                        .buildFacetQuery("hoge:\"QUERY1 QUERY2\" QUERY3"));

    }

    public void test_buildFacet_sortField() {
        String query;
        String searchQuery;

        query = "";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "sort:contentLength";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "sort:contentLength.desc";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "sort:contentLength.asc,lastModified";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "QUERY sort:contentLength";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery);

        query = "QUERY sort:contentLength.desc";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery);

        query = "QUERY sort:contentLength.asc,lastModified";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(title:QUERY OR content:QUERY)", searchQuery);

        query = "QUERY site:QUERY1 sort:contentLength";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery);

        query = "QUERY sort:contentLength.desc  site:QUERY1";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery);

        query = "QUERY sort:contentLength.asc,lastModified site:QUERY1";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("(site:QUERY1 AND (title:QUERY OR content:QUERY))",
                searchQuery);
    }

    public void test_buildFacet_sortField_invalid() {
        String query;
        String searchQuery;

        query = "sort:hoge";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "sort:contentLength.hoge";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);

        query = "sort:contentLength.asc,hoge";
        searchQuery = queryHelperImpl.buildFacetQuery(query);
        assertEquals("", searchQuery);
    }

    public void test_buildFacet_wildcardSearches() {
        // *

        assertEquals("(title:\\* OR content:\\*)",
                queryHelperImpl.buildFacetQuery("*"));
        assertEquals("(title:QUERY* OR content:QUERY*)",
                queryHelperImpl.buildFacetQuery("QUERY* "));
        assertEquals("(title:Q*ERY OR content:Q*ERY)",
                queryHelperImpl.buildFacetQuery(" Q*ERY"));

        assertEquals(
                "((title:Q*ERY1 AND title:Q*ERY2) OR (content:Q*ERY1 AND content:Q*ERY2))",
                queryHelperImpl.buildFacetQuery("Q*ERY1 Q*ERY2"));

        assertEquals("(title:Q*ERY1\\ Q*ERY2 OR content:Q*ERY1\\ Q*ERY2)",
                queryHelperImpl.buildFacetQuery("\"Q*ERY1 Q*ERY2\""));

        assertEquals(
                "((title:Q*ERY1\\ Q*ERY2 AND title:Q*ERY3) OR (content:Q*ERY1\\ Q*ERY2 AND content:Q*ERY3))",
                queryHelperImpl.buildFacetQuery("\"Q*ERY1 Q*ERY2\" Q*ERY3"));

        assertEquals("site:QUERY1*",
                queryHelperImpl.buildFacetQuery("site:QUERY1*"));
        assertEquals("(site:QUERY1* AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1* QUERY2"));

        // ?

        assertEquals("(title:\\? OR content:\\?)",
                queryHelperImpl.buildFacetQuery("?"));
        assertEquals("(title:QUERY? OR content:QUERY?)",
                queryHelperImpl.buildFacetQuery("QUERY? "));
        assertEquals("(title:Q?ERY OR content:Q?ERY)",
                queryHelperImpl.buildFacetQuery(" Q?ERY"));

        assertEquals(
                "((title:Q?ERY1 AND title:Q?ERY2) OR (content:Q?ERY1 AND content:Q?ERY2))",
                queryHelperImpl.buildFacetQuery("Q?ERY1 Q?ERY2"));

        assertEquals("(title:Q?ERY1\\ Q?ERY2 OR content:Q?ERY1\\ Q?ERY2)",
                queryHelperImpl.buildFacetQuery("\"Q?ERY1 Q?ERY2\""));

        assertEquals(
                "((title:Q?ERY1\\ Q?ERY2 AND title:Q?ERY3) OR (content:Q?ERY1\\ Q?ERY2 AND content:Q?ERY3))",
                queryHelperImpl.buildFacetQuery("\"Q?ERY1 Q?ERY2\" Q?ERY3"));

        assertEquals("site:QUERY1?",
                queryHelperImpl.buildFacetQuery("site:QUERY1?"));
        assertEquals("(site:QUERY1? AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1? QUERY2"));
    }

    public void test_buildFacet_fuzzySearches() {
        // ~

        assertEquals("(title:QUERY~ OR content:QUERY~)",
                queryHelperImpl.buildFacetQuery("QUERY~"));
        assertEquals(
                "((title:QUERY1~ AND title:QUERY2) OR (content:QUERY1~ AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1~ QUERY2"));
        assertEquals(
                "((title:QUERY1~ AND title:QUERY2~) OR (content:QUERY1~ AND content:QUERY2~))",
                queryHelperImpl.buildFacetQuery("QUERY1~ QUERY2~"));

        assertEquals("site:QUERY1~",
                queryHelperImpl.buildFacetQuery("site:QUERY1~"));
        assertEquals("(site:QUERY1~ AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1~ QUERY2"));

        assertEquals("(title:QUERY1\\ QUERY2~ OR content:QUERY1\\ QUERY2~)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\"~"));
        assertEquals("(title:QUERY1~ OR content:QUERY1~)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~\""));

        // ~0.8

        assertEquals("(title:QUERY~0.8 OR content:QUERY~0.8)",
                queryHelperImpl.buildFacetQuery("QUERY~0.8"));
        assertEquals(
                "((title:QUERY1~0.8 AND title:QUERY2) OR (content:QUERY1~0.8 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1~0.8 QUERY2"));
        assertEquals(
                "((title:QUERY1~0.5 AND title:QUERY2~0.8) OR (content:QUERY1~0.5 AND content:QUERY2~0.8))",
                queryHelperImpl.buildFacetQuery("QUERY1~0.5 QUERY2~0.8"));

        assertEquals("site:QUERY1~0.8",
                queryHelperImpl.buildFacetQuery("site:QUERY1~0.8"));
        assertEquals("(site:QUERY1~0.8 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1~0.8 QUERY2"));

        assertEquals(
                "(title:QUERY1\\ QUERY2~0.8 OR content:QUERY1\\ QUERY2~0.8)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\"~0.8"));
        assertEquals("(title:QUERY1~0.8 OR content:QUERY1~0.8)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~0.8\""));

        assertEquals("(title:QUERY1~0.8 OR content:QUERY1~0.8)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~0.8a\""));
        assertEquals("(title:QUERY1~ OR content:QUERY1~)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~a\""));

    }

    public void test_buildFacet_proximitySearches() {
        // ~10
        assertEquals("(title:\"QUERY\"~10 OR content:\"QUERY\"~10)",
                queryHelperImpl.buildFacetQuery("QUERY~10"));
        assertEquals("(title:\"QUERY\"~1 OR content:\"QUERY\"~1)",
                queryHelperImpl.buildFacetQuery("QUERY~1"));
        assertEquals("(title:\"QUERY\"~5 OR content:\"QUERY\"~5)",
                queryHelperImpl.buildFacetQuery("QUERY~5.5"));
        assertEquals(
                "((title:\"QUERY1\"~10 AND title:QUERY2) OR (content:\"QUERY1\"~10 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1~10 QUERY2"));
        assertEquals(
                "((title:\"QUERY1\"~5 AND title:\"QUERY2\"~10) OR (content:\"QUERY1\"~5 AND content:\"QUERY2\"~10))",
                queryHelperImpl.buildFacetQuery("QUERY1~5 QUERY2~10"));

        assertEquals("site:\"QUERY1\"~10",
                queryHelperImpl.buildFacetQuery("site:QUERY1~10"));
        assertEquals(
                "(site:\"QUERY1\"~10 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1~10 QUERY2"));

        assertEquals(
                "(title:\"QUERY1\\ QUERY2\"~10 OR content:\"QUERY1\\ QUERY2\"~10)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\"~10"));
        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~10\""));

        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~10a\""));
        assertEquals("(title:\"QUERY1\"~10 OR content:\"QUERY1\"~10)",
                queryHelperImpl.buildFacetQuery("\"QUERY1~10.1.2"));

    }

    public void test_buildFacet_rangeSearches() {
        // mod_date:[20020101 TO 20030101] 
        assertEquals(
                "(title:[20020101 TO 20030101] OR content:[20020101 TO 20030101])",
                queryHelperImpl.buildFacetQuery("[20020101 TO 20030101]"));
        assertEquals("lastModified:[20020101 TO 20030101]",
                queryHelperImpl
                        .buildFacetQuery("lastModified:[20020101 TO 20030101]"));
        assertEquals(
                "(lastModified:[20020101 TO 20030101] AND (title:QUERY OR content:QUERY))",
                queryHelperImpl
                        .buildFacetQuery("QUERY lastModified:[20020101 TO 20030101]"));

        // TODO more..

        // title:{Aida TO Carmen}
        assertEquals("(title:{Aida TO Carmen} OR content:{Aida TO Carmen})",
                queryHelperImpl.buildFacetQuery("{Aida TO Carmen}"));
        assertEquals("lastModified:{Aida TO Carmen}",
                queryHelperImpl
                        .buildFacetQuery("lastModified:{Aida TO Carmen}"));
        assertEquals(
                "(title:{Aida TO Carmen} AND (title:QUERY OR content:QUERY))",
                queryHelperImpl.buildFacetQuery("QUERY title:{Aida TO Carmen}"));

        // TODO more..
    }

    public void test_buildFacet_boosting() {
        // ^1000 ""^1000
        assertEquals("(title:QUERY^1000 OR content:QUERY^1000)",
                queryHelperImpl.buildFacetQuery("QUERY^1000"));
        assertEquals(
                "((title:QUERY1^1000 AND title:QUERY2) OR (content:QUERY1^1000 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1^1000 QUERY2"));
        assertEquals(
                "((title:QUERY1^500 AND title:QUERY2^1000) OR (content:QUERY1^500 AND content:QUERY2^1000))",
                queryHelperImpl.buildFacetQuery("QUERY1^500 QUERY2^1000"));

        assertEquals("site:QUERY1^1000",
                queryHelperImpl.buildFacetQuery("site:QUERY1^1000"));
        assertEquals("(site:QUERY1^1000 AND (title:QUERY2 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("site:QUERY1^1000 QUERY2"));

        assertEquals(
                "(title:QUERY1\\ QUERY2^1000 OR content:QUERY1\\ QUERY2^1000)",
                queryHelperImpl.buildFacetQuery("\"QUERY1 QUERY2\"^1000"));
        assertEquals("(title:QUERY1^1000 OR content:QUERY1^1000)",
                queryHelperImpl.buildFacetQuery("\"QUERY1^1000\""));
    }

    public void test_buildFacet_reserved() {
        for (int i = 0; i < Constants.RESERVED.length - 1; i++) {
            assertEquals("(title:\\" + Constants.RESERVED[i] + " OR content:\\"
                    + Constants.RESERVED[i] + ")",
                    queryHelperImpl.buildFacetQuery(Constants.RESERVED[i]));
        }
        assertEquals("(title:\\: OR content:\\:)",
                queryHelperImpl.buildFacetQuery(":"));
    }

    public void test_buildFacet_or() {
        assertEquals("(title:QUERY OR content:QUERY)",
                queryHelperImpl.buildFacetQuery("OR QUERY"));
        assertEquals(
                "((title:QUERY1 OR title:QUERY2) OR (content:QUERY1 OR content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1 OR QUERY2"));
        assertEquals("title:QUERY",
                queryHelperImpl.buildFacetQuery("OR title:QUERY"));
        assertEquals("(title:QUERY1 OR title:QUERY2)",
                queryHelperImpl.buildFacetQuery("title:QUERY1 OR title:QUERY2"));
        assertEquals("(title:QUERY2 OR (title:QUERY1 OR content:QUERY1))",
                queryHelperImpl.buildFacetQuery("QUERY1 OR title:QUERY2"));
        assertEquals("(site:QUERY1 OR title:QUERY2)",
                queryHelperImpl.buildFacetQuery("site:QUERY1 OR title:QUERY2"));
        assertEquals(
                "((title:QUERY1 OR title:QUERY2 AND title:QUERY3) OR (content:QUERY1 OR content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildFacetQuery("QUERY1 OR QUERY2 QUERY3"));
        assertEquals(
                "((title:QUERY1 OR title:QUERY2 OR title:QUERY3) OR (content:QUERY1 OR content:QUERY2 OR content:QUERY3))",
                queryHelperImpl.buildFacetQuery("QUERY1 OR QUERY2 OR QUERY3"));
    }

    public void test_buildFacet_and() {
        assertEquals("(title:QUERY OR content:QUERY)",
                queryHelperImpl.buildFacetQuery("AND QUERY"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2) OR (content:QUERY1 AND content:QUERY2))",
                queryHelperImpl.buildFacetQuery("QUERY1 AND QUERY2"));
        assertEquals("title:QUERY",
                queryHelperImpl.buildFacetQuery("AND title:QUERY"));
        assertEquals("(title:QUERY1 AND title:QUERY2)",
                queryHelperImpl
                        .buildFacetQuery("title:QUERY1 AND title:QUERY2"));
        assertEquals("(title:QUERY2 AND (title:QUERY1 OR content:QUERY1))",
                queryHelperImpl.buildFacetQuery("QUERY1 AND title:QUERY2"));
        assertEquals("(site:QUERY1 AND title:QUERY2)",
                queryHelperImpl.buildFacetQuery("site:QUERY1 AND title:QUERY2"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildFacetQuery("QUERY1 AND QUERY2 QUERY3"));
        assertEquals(
                "((title:QUERY1 AND title:QUERY2 AND title:QUERY3) OR (content:QUERY1 AND content:QUERY2 AND content:QUERY3))",
                queryHelperImpl.buildFacetQuery("QUERY1 AND QUERY2 AND QUERY3"));
    }

    public void test_buildFacet_not() {
        assertEquals("NOT title:QUERY AND NOT content:QUERY",
                queryHelperImpl.buildFacetQuery("NOT QUERY"));
        assertEquals("NOT title:QUERY",
                queryHelperImpl.buildFacetQuery("NOT title:QUERY"));
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 QUERY2"));
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 OR QUERY2"));
        assertEquals(
                "NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY2 AND NOT content:QUERY2",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 NOT QUERY2"));
        assertEquals(
                "NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY2 AND NOT content:QUERY2",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 OR NOT QUERY2"));
        assertEquals(
                "(title:QUERY2 OR content:QUERY2) AND NOT title:QUERY1 AND NOT content:QUERY1 AND NOT title:QUERY3 AND NOT content:QUERY3",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 QUERY2 NOT QUERY3"));
        assertEquals("NOT site:QUERY",
                queryHelperImpl.buildFacetQuery("NOT site:QUERY"));
        assertEquals("NOT site:QUERY1 AND NOT title:QUERY2",
                queryHelperImpl
                        .buildFacetQuery("NOT site:QUERY1 NOT title:QUERY2"));
        assertEquals("(title:QUERY2 OR content:QUERY2) AND NOT site:QUERY1",
                queryHelperImpl.buildFacetQuery("NOT site:QUERY1 QUERY2"));
        assertEquals("site:QUERY2 AND NOT title:QUERY1 AND NOT content:QUERY1",
                queryHelperImpl.buildFacetQuery("NOT QUERY1 site:QUERY2"));

        assertEquals("(title:QUERY AND (title:\\(NOT OR content:\\(NOT))",
                queryHelperImpl.buildFacetQuery("(NOT title:QUERY"));
    }
}
