﻿using System;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Xml;
using System.Net;
using System.Text;
using System.Text.RegularExpressions;
using System.IO;
using System.Web;

namespace FeedGenerator.lib
{
    class Entry
    {
        private string id;
        private string title;
        private string summary;
        private DateTime published;
        private DateTime updated;
        private string link;
        private string content;
        private List<string> others;
        private List<string> error;

        private bool isInsertBrTag;

        // プロパティ
        public string Id
        {
            set { this.id = value; }
            get { return this.id; }
        }
        public string Title
        {
            set { this.title = value; }
            get { return this.title; }
        }
        public string Summary
        {
            set { this.summary = value; }
            get { return this.summary; }
        }
        public DateTime Published
        {
            set { this.published = value; }
            get { return this.published; }
        }
        public DateTime Updated
        {
            set { this.updated = value; }
            get { return this.updated; }
        }
        public string Link
        {
            set { this.link = value; }
            get { return this.link; }
        }
        public string Content
        {
            set { this.content = value; }
            get { return this.content; }
        }
        public List<string> Others
        {
            set { this.others = value; }
            get { return this.others; }
        }
        public List<string> Error
        {
            set { this.error = value; }
            get { return this.error; }
        }
        public bool IsInsertBrTag
        {
            set { this.isInsertBrTag = value; }
            get { return this.isInsertBrTag; }
        }

        /// <summary>
        /// Entryオブジェクトを初期化します
        /// </summary>
        public Entry(bool isInsertBrTag)
        {
            title = "New Title";
            published = DateTime.Now;
            updated = DateTime.Now;
            content = string.Empty;
            others = new List<string>();
            error = new List<string>();
            this.isInsertBrTag = isInsertBrTag;
        }

        /// <summary>
        /// 指定されたIDを基礎XML設定としてEntryオブジェクトを初期化します
        /// </summary>
        /// <param name="feedId"></param>
        public Entry(string feedId, bool isInsertBrTag)
        {
            title = "New Title";
            published = DateTime.Now;
            updated = DateTime.Now;
            id = feedId + "#" + published.ToString("s") + "+09:00";
            content = string.Empty;
            others = new List<string>();
            error = new List<string>();
            this.isInsertBrTag = isInsertBrTag;
        }

        /// <summary>
        /// Entryオブジェクトに値を設定します。updateの値はメソッドを呼び出した瞬間のDateTime.Nowの値になります。
        /// </summary>
        /// <param name="id"></param>
        /// <param name="title"></param>
        /// <param name="summary"></param>
        /// <param name="link"></param>
        /// <param name="content"></param>
        public void setEntry(string id, string title, string summary, string link, string content)
        {
            this.id = id;
            this.title = title;
            this.summary = summary;
            this.updated = DateTime.Now;
            this.link = link;
            this.content = content;
        }

        /// <summary>
        /// Entryオブジェクトに値を設定します。指定されたDateTimeオブジェクトでupdateの値を更新します
        /// </summary>
        /// <param name="id"></param>
        /// <param name="title"></param>
        /// <param name="summary"></param>
        /// <param name="updated"></param>
        /// <param name="link"></param>
        /// <param name="content"></param>
        public void setEntry(string id, string title, string summary, DateTime updated, string link, string content)
        {
            this.id = id;
            this.title = title;
            this.summary = summary;
            this.updated = updated;
            this.link = link;
            this.content = content;
        }

        /// <summary>
        /// Entryオブジェクトに値を設定します。指定されたDateTimeオブジェクトでpublishedとupdateの値を更新します
        /// </summary>
        /// <param name="id"></param>
        /// <param name="title"></param>
        /// <param name="summary"></param>
        /// <param name="published"></param>
        /// <param name="updated"></param>
        /// <param name="link"></param>
        /// <param name="content"></param>
        public void setEntry(string id, string title, string summary, DateTime published, DateTime updated, string link, string content)
        {
            this.id = id;
            this.title = title;
            this.summary = summary;
            this.published = published;
            this.updated = updated;
            this.link = link;
            this.content = content;
        }

        /// <summary>
        /// オーバーロードされたEqualsメソッドです
        /// </summary>
        /// <param name="anotherEntry">比較するEntryオブジェクト</param>
        /// <returns>全てのメンバが一致した場合true、そうでない場合false</returns>
        public bool Equals(Entry anotherEntry)
        {
            if (this.id != anotherEntry.id ||
                this.title != anotherEntry.title ||
                this.summary != anotherEntry.summary ||
                this.published.ToString("s") != anotherEntry.published.ToString("s") ||
                this.updated.ToString("s") != anotherEntry.updated.ToString("s") ||
                this.link != anotherEntry.link ||
                this.content != anotherEntry.content)
            {
                return false;
            }
            return true;
        }

        /// <summary>
        /// XMLを読み込んだ後に解析して、オブジェクトに格納するメソッド
        /// </summary>
        /// <param name="filePath">XMLのファイルフルパス</param>
        public static List<Entry> readXml(string filePath, bool isInsertBrTag)
        {
            // 実際にXMLを読み込んで解析する
            XmlTextReader reader = new XmlTextReader(filePath);
            List<Entry> entryList = new List<Entry>();
            Entry tempEntry = new Entry(isInsertBrTag);
            bool isInInnerEntry = false;
            // 頭が悪いReadOuterXMLのための制御用変数
            bool isActiveReadOuterXmlFlag = false;

            while (isActiveReadOuterXmlFlag || reader.Read())
            {
                isActiveReadOuterXmlFlag = false;
                try
                {
                    // 開始タグだった場合の処理(/ではない場合)
                    if (reader.NodeType == XmlNodeType.Element)
                    {
                        switch (reader.LocalName)
                        {
                            case "entry":
                                isInInnerEntry = true;
                                tempEntry = new Entry(isInsertBrTag);
                                break;

                            case "id":
                                tempEntry.id = reader.ReadString();
                                break;

                            case "title":
                                tempEntry.title = reader.ReadString();
                                break;

                            case "summary":
                                tempEntry.summary = reader.ReadString();
                                break;

                            case "published":
                                tempEntry.published = RFC3339.getDateTime(reader.ReadString());
                                break;

                            case "updated":
                                tempEntry.updated = RFC3339.getDateTime(reader.ReadString());
                                break;

                            case "link":
                                if (reader.HasAttributes)
                                {
                                    reader.MoveToAttribute("href");
                                    tempEntry.link = reader.Value;
                                }
                                break;

                            case "content":
                                tempEntry.content = tempEntry.unescapeHTML(Regex.Replace(reader.ReadString(), "[^\r]\n", "\r\n"));
                                break;

                            default:
                                if (reader.LocalName == string.Empty)
                                    break;

                                // 未知の定義要素を読み取る
                                if (isInInnerEntry == true)
                                {
                                    isActiveReadOuterXmlFlag = true;
                                    // 自動挿入される不要な属性を削除
                                    tempEntry.others.Add(reader.ReadOuterXml().Replace(" xmlns=\"http://www.w3.org/2005/Atom\"", ""));
                                }

                                break;
                        }
                    }
                    // 終了タグだった場合(/で閉じる場合)
                    else if (reader.NodeType == XmlNodeType.EndElement)
                    {
                        switch (reader.LocalName)
                        {
                            case "entry":
                                isInInnerEntry = false;
                                entryList.Add(tempEntry);
                                break;
                        }
                    }
                }
                catch (Exception)
                {
                    // XML形式エラー出力用
                    tempEntry.error.Add("WARNING : LINE " + reader.LineNumber.ToString());
                }
            }

            reader.Close();

            return entryList;
        }

        /// <summary>
        /// オーバーライドされたメソッドです。Feed形式のentry要素を返却します。
        /// </summary>
        /// <returns></returns>
        public override string ToString()
        {
            StringBuilder buffer = new StringBuilder();
            buffer.AppendLine("<entry>");
            buffer.AppendLine("<id>" + this.id + "</id>");
            buffer.AppendLine("<title>" + this.title + "</title>");
            buffer.AppendLine("<summary>" + this.summary + "</summary>");
            buffer.AppendLine("<published>" + this.published.ToString("s") + "+09:00</published>");
            buffer.AppendLine("<updated>" + this.updated.ToString("s") + "+09:00</updated>");
            buffer.AppendLine("<link href=\"" + this.link + "\" />");
            // 挿入されていたほかの要素を挿入
            foreach (string element in others)
            {
                buffer.AppendLine(element);
            }
            buffer.AppendLine("<content type=\"html\">" + escapeHTML(this.content) + "</content>");
            buffer.AppendLine("</entry>");

            return buffer.ToString();
        }

        /// <summary>
        /// 与えられた文字列をアンエスケープします
        /// </summary>
        /// <param name="input">アンエスケープしたい文字列</param>
        /// <returns>アンエスケープされた文字列</returns>
        private string unescapeHTML(string input)
        {
            if (isInsertBrTag == true)
                return Regex.Replace(HttpUtility.HtmlDecode(input), "<br>(\r\n|\n)", "\r\n", RegexOptions.Multiline);
            else
                return HttpUtility.HtmlDecode(input);
        }

        /// <summary>
        /// 与えられた文字列をエスケープします
        /// </summary>
        /// <param name="input">エスケープする文字列</param>
        /// <returns>エスケープされた文字列</returns>
        private string escapeHTML(string input)
        {
            if (isInsertBrTag == true)
                input = Regex.Replace(input, "(?<!\\<\\/(p|h\\d|div)\\>)\r\n", "<br>\r\n", RegexOptions.Multiline);

            return HttpUtility.HtmlEncode(input);
        }
    }
}
