﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using FeedGenerator.lib;
using System.IO;
using FeedGenerator.subform;

namespace FeedGenerator
{
    public partial class MainForm : Form
    {
        private string fileName;
        private Feed feed;
        private List<TextBox> entryTextBoxList, xmlSettingTextBoxList;
        private Config config;

        /// <summary>
        /// アプリケーションを起動した際に初期化処理を行います
        /// </summary>
        public MainForm()
        {
            InitializeComponent();

            // 日記設定項目上の全テキストボックスをアクセス用のリストに格納
            entryTextBoxList = new List<TextBox>();
            entryTextBoxList.Add(entryTitleTextbox);
            entryTextBoxList.Add(entrySummaryTextBox);
            entryTextBoxList.Add(entryIdTextBox);
            entryTextBoxList.Add(entryLinkTextBox);

            // XML設定項目上の全テキストボックスをアクセス用のリストに格納
            xmlSettingTextBoxList = new List<TextBox>();
            xmlSettingTextBoxList.Add(xmlPathTextBox);
            xmlSettingTextBoxList.Add(xmlTitleTextBox);
            xmlSettingTextBoxList.Add(xmlSubtitleTextBox);
            xmlSettingTextBoxList.Add(xmlLinkTextBox);
            xmlSettingTextBoxList.Add(xmlIdTextBox);
            xmlSettingTextBoxList.Add(xmlRightTextBox);
            xmlSettingTextBoxList.Add(xmlNameTextBox);
            xmlSettingTextBoxList.Add(xmlEmailTextBox);

            // 設定を読み込み、適用する
            config = new Config();
            appliConfig(config);

            // 初期化
            newXml();

            // ハイライトモードをXMLに設定
            contentAzukiControl.Highlighter = Sgry.Azuki.Highlighter.Highlighters.Xml;
            contentAzukiControl.ViewWidth = contentAzukiControl.ClientSize.Width;
            // サイズ変更用デリゲート
            contentAzukiControl.Resize += delegate
            {
                contentAzukiControl.ViewWidth = contentAzukiControl.ClientSize.Width;
            };
        }

        /// <summary>
        /// 設定ファイルから読み込んだ内容を適用します
        /// </summary>
        public void appliConfig(Config config)
        {
            contentAzukiControl.Font = config.contentFont;
        }

        /// <summary>
        /// 現在保有されている全ての情報を破棄します
        /// </summary>
        private void newXml()
        {
            // 既存の全データを破棄
            fileName = string.Empty;
            fileNameStripStatusLabel.Text = fileName;
            feed = new Feed(config.isInsertBrTag);

            // 日記編集画面を初期化
            entryListComboBox.Text = string.Empty;
            entryListComboBox.SelectedIndex = -1;
            entryListComboBox.Items.Clear();
            foreach (TextBox textBox in entryTextBoxList)
            {
                textBox.Clear();
            }
            // AzukiControlの初期化。READONLYを解除しないとTEXT要素を操作できない
            contentAzukiControl.ClearHistory();
            if (contentAzukiControl.IsReadOnly == true)
            {
                contentAzukiControl.IsReadOnly = false;
                contentAzukiControl.Text = string.Empty;
                contentAzukiControl.IsReadOnly = true;
            }
            else
            {
                contentAzukiControl.Text = string.Empty;
            }
            entryDatePicker.Value = DateTime.Now;
            entryTimePicker.Value = DateTime.Now;

            // XML設定項目を初期化
            foreach (TextBox textBox in xmlSettingTextBoxList) {
                textBox.Clear();
            }
        }

        /// <summary>
        /// ダイアログを開き、指定されたXMLファイルからデータを読み込んで、各種オブジェクトに格納します
        /// </summary>
        private void readXml()
        {
            // ダイアログを開いてユーザーにファイルを選択させる
            OpenFileDialog openDialog = new OpenFileDialog();
            openDialog.FileName = "diary";
            openDialog.Filter = "XMLファイル(*.xml)|*.xml";

            DialogResult result = openDialog.ShowDialog();
            if (result != DialogResult.Cancel)
            {
                // 全データを破棄
                newXml();

                // ファイルから日記データを読み込む
                fileName = openDialog.FileName;
                fileNameStripStatusLabel.Text = openDialog.FileName;
                try
                {
                    feed = new Feed(openDialog.FileName, config.isInsertBrTag);
                    feed.feedinfo.readXml(openDialog.FileName);
                    feed.entryList = Entry.readXml(openDialog.FileName, config.isInsertBrTag);
                }
                catch (Exception)
                {
                    MessageBox.Show("XMLの読み込み中に致命的なエラーが発生しまし為、ファイル読み込みを中断します", "FATAL ERROR", MessageBoxButtons.OK, MessageBoxIcon.Error);
                    return;
                }
            }
            else
            {
                return;
            }

            // 日記リストをコンボボックスに追加する
            foreach (Entry entry in feed.entryList)
            {
                entryListComboBox.Items.Add(entry.Title + " / " + entry.Published);
            }

            // 基本情報をXML設定タブに設定する
            xmlPathTextBox.Text = feed.feedinfo.Self;
            xmlTitleTextBox.Text = feed.feedinfo.Title;
            xmlSubtitleTextBox.Text = feed.feedinfo.Subtitle;
            xmlLinkTextBox.Text = feed.feedinfo.Link;
            xmlIdTextBox.Text = feed.feedinfo.Id;
            xmlRightTextBox.Text = feed.feedinfo.Rights;
            xmlNameTextBox.Text = feed.feedinfo.Authorname;
            xmlEmailTextBox.Text = feed.feedinfo.Authoremail;

            List<string> errorList = new List<string>();

            // エラーが発生していた場合、そのエラーを全て格納する
            foreach (string error in feed.feedinfo.Error)
                errorList.Add(error);

            foreach (Entry entry in feed.entryList)
            {
                foreach (string error in entry.Error)
                    errorList.Add(error);
            }

            // 完了ダイアログを表示する
            if (errorList.ToArray().Length == 0)
            {
                MessageBox.Show("XMLの読み込みが正常に完了しました", "正常終了");
            }
            else
            {
                StringBuilder buffer = new StringBuilder();
                foreach (string error in errorList)
                    buffer.AppendLine(error);

                MessageBox.Show("XMLの読み込みは完了しましたが、以下の位置でエラーが発生した為正常にファイルが読み込まれない可能性があります。\r\n" + buffer.ToString(), "注意", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }

            // 自動的に最新の日記を表示する
            if (feed.entryList.ToArray().Length > 0)
            {
                entryListComboBox.SelectedIndex = 0;
            }
                
        }

        /// <summary>
        /// Entry配列から指定のインデックスのEntryオブジェクトを読み出し、フォーム上に配置します
        /// </summary>
        /// <param name="index"></param>
        private void editXml(int index)
        {
            Entry entry = feed.entryList[index];

            entryTitleTextbox.Text = entry.Title;
            entrySummaryTextBox.Text = entry.Summary;

            if (config.isIdGenerate)
                entryIdTextBox.Text = feed.feedinfo.Id + "?" + entry.Published.ToString("s") + "+09:00";
            else
                entryIdTextBox.Text = entry.Id;

            if (config.isUrlGenerate)
                entryLinkTextBox.Text = feed.feedinfo.Link + "#" + feed.feedinfo.Id + "?" + entry.Published.ToString("s") + "+09:00";
            else
                entryLinkTextBox.Text = entry.Link;

            if (config.isUpdateGenerate)
            {
                entryDatePicker.Value = DateTime.Now;
                entryTimePicker.Value = DateTime.Now;
            }
            else
            {
                entryDatePicker.Value = entry.Updated;
                entryTimePicker.Value = entry.Updated;
            }

            contentAzukiControl.IsReadOnly = false;
            contentAzukiControl.Text = entry.Content;
        }

        /// <summary>
        /// フォーム上に配置されたEntryオブジェクトで、指定のインデックスのEntryオブジェクトを上書きします
        /// </summary>
        /// <param name="index"></param>
        private void updateXml(int index)
        {
            // リストが範囲外の場合は即時リターン
            if (index < 0)
                return;

            // 2つのDateTimePickerを結合
            DateTime updateTime = new DateTime(entryDatePicker.Value.Year, entryDatePicker.Value.Month, entryDatePicker.Value.Day, entryTimePicker.Value.Hour, entryTimePicker.Value.Minute, entryTimePicker.Value.Second);
            feed.entryList[index].setEntry(entryIdTextBox.Text, entryTitleTextbox.Text, entrySummaryTextBox.Text, updateTime, entryLinkTextBox.Text, contentAzukiControl.Text);

            // コンボボックスに変更を反映
            if (entryListComboBox.SelectedIndex > -1)
            {
                int selectedIndex = entryListComboBox.SelectedIndex;
                entryListComboBox.Items.Clear();
                updateComboBox();
                entryListComboBox.SelectedIndex = selectedIndex;
            }
        }

        /// <summary>
        /// フォームの入力内容からEntryオブジェクトを生成します
        /// </summary>
        /// <returns></returns>
        private Entry getNowXml()
        {
            Entry entry = new Entry(config.isInsertBrTag);
            entry.Title = entryTitleTextbox.Text;
            entry.Summary = entrySummaryTextBox.Text;
            entry.Id = entryIdTextBox.Text;
            entry.Link = entryLinkTextBox.Text;
            DateTime updateTime = new DateTime(entryDatePicker.Value.Year, entryDatePicker.Value.Month, entryDatePicker.Value.Day, entryTimePicker.Value.Hour, entryTimePicker.Value.Minute, entryTimePicker.Value.Second);
            entry.Updated = updateTime;
            entry.Published = updateTime;
            entry.Content = contentAzukiControl.Text;
            return entry;
        }

        /// <summary>
        /// フォームの入力内容を、Feedオブジェクトに反映させます
        /// </summary>
        private void updateFeedObject(ref FeedInfo feed) {
            feed.Self = xmlPathTextBox.Text;
            feed.Title = xmlTitleTextBox.Text;
            feed.Subtitle = xmlSubtitleTextBox.Text;
            feed.Link = xmlLinkTextBox.Text;
            feed.Updated = DateTime.Now;
            feed.Id = xmlIdTextBox.Text;
            feed.Rights = xmlRightTextBox.Text;
            feed.Authorname = xmlNameTextBox.Text;
            feed.Authoremail = xmlEmailTextBox.Text;
        }

        /// <summary>
        /// Feedオブジェクト及びEntryオブジェクト配列からXMLを書き出します
        /// </summary>
        /// <param name="filePath"></param>
        private void writeXmlFile(string filePath)
        {
            // 入力内容をFeedオブジェクトに反映させる
            updateFeedObject(ref feed.feedinfo);

            // ファイルへの書き込みを行う
            try
            {
                feed.writeXml(filePath);
            }
            catch (Exception)
            {
                MessageBox.Show("ファイルの保存に失敗しました", "ERROR", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        /// <summary>
        /// AzukiControl上の選択された状態の文字列の手前と後の位置にタグを埋め込むメソッドです
        /// </summary>
        /// <param name="tagStringList">埋め込みたい文字列の配列。[0]に選択位置の手前に埋め込むタグ、[1]に選択位置の後ろに埋め込むタグを指定</param>
        private void insertTag(List<string> tagStringList)
        {
            if (tagStringList[1] == null || tagStringList[0] == null)
            {
                MessageBox.Show("タグの挿入文字列が不正です", "ERROR");
                return;
            }
            else if (entryListComboBox.SelectedIndex == -1)
            {
                return;
            }

            int selectStart, selectEnd;
            contentAzukiControl.GetSelection(out selectStart, out selectEnd);

            contentAzukiControl.Text = contentAzukiControl.Text.Insert(selectEnd, tagStringList[1]);
            contentAzukiControl.Text = contentAzukiControl.Text.Insert(selectStart, tagStringList[0]);

            // キャレットの位置を戻す
            contentAzukiControl.SetSelection(selectStart + tagStringList[0].Length, selectStart + tagStringList[0].Length);
            contentAzukiControl.ScrollToCaret();
        }

        /// <summary>
        /// コンボボックスを更新します
        /// </summary>
        private void updateComboBox()
        {
            entryListComboBox.Items.Clear();
            foreach (Entry entry in feed.entryList)
            {
                entryListComboBox.Items.Add(entry.Title + " / " + entry.Published);
            }
        }

        /// <summary>
        /// コンボボックスの値を変更した際に呼び出されます
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void entryListComboBox_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (entryListComboBox.SelectedIndex != -1)
            {
                editXml(entryListComboBox.SelectedIndex);
            }
        }

        /// <summary>
        /// 開くボタンを押した場合の処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void openXmlToolStripMenuItem_Click(object sender, EventArgs e)
        {
            readXml();
        }

        /// <summary>
        /// 新規作成ボタンを押した場合の処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void newXmlToolStripMenuItem_Click(object sender, EventArgs e)
        {
            newXml();
        }

        /// <summary>
        /// 上書き保存ボタンを押した場合の処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void updateXmlToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // 上書き前に反映を行うか尋ねる
            DialogResult result = MessageBox.Show("現在編集している日記の内容を保存するファイルに反映させますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Asterisk);
            if (result == DialogResult.Yes)
            {
                updateXml(entryListComboBox.SelectedIndex);
            }

            // ファイル名が存在しなかった場合、ダイアログで問い合わせを行う
            if (fileName == string.Empty)
            {
                SaveFileDialog saveDialog = new SaveFileDialog();
                saveDialog.FileName = "diary";
                saveDialog.Filter = "XMLファイル(*.xml)|*.xml";

                result = saveDialog.ShowDialog();
                if (result != DialogResult.Cancel)
                {
                    writeXmlFile(saveDialog.FileName);
                }
            }
            else
            {
                writeXmlFile(fileName);
            }
            
        }

        /// <summary>
        /// 名前をつけて保存ボタンを押した場合の処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void saveXmlToolStripMenuItem_Click(object sender, EventArgs e)
        {
            // 上書き前に反映を行うか尋ねる
            DialogResult result = MessageBox.Show("現在編集している日記の内容を保存するファイルに反映させますか？", "確認", MessageBoxButtons.YesNo, MessageBoxIcon.Asterisk);
            if (result == DialogResult.Yes)
            {
                updateXml(entryListComboBox.SelectedIndex);
            }

            // ダイアログを開いてユーザーにファイルを選択させる
            SaveFileDialog saveDialog = new SaveFileDialog();
            saveDialog.FileName = "diary";
            saveDialog.Filter = "XMLファイル(*.xml)|*.xml";

            result = saveDialog.ShowDialog();
            if (result != DialogResult.Cancel)
            {
                writeXmlFile(saveDialog.FileName);
            }
        }

        /// <summary>
        /// 反映ボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void updateEntryButton_Click(object sender, EventArgs e)
        {
            // コンボボックス上で日記が選択されていない場合は何も行わない
            if (entryListComboBox.SelectedIndex == -1)
                return;

            DialogResult result = MessageBox.Show("編集内容を反映させますか？", "確認", MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
            if (result == DialogResult.Cancel)
                return;

            if (entryListComboBox.SelectedIndex != -1)
            {
                updateXml(entryListComboBox.SelectedIndex);
            }
        }

        /// <summary>
        /// 新規追加ボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void newEntryButton_Click(object sender, EventArgs e)
        {
            DialogResult result = MessageBox.Show("新しい日記を追加しますか？", "確認", MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
            if (result == DialogResult.Cancel)
                return;

            Entry newEntry = new Entry(feed.feedinfo.Id, config.isInsertBrTag);

            feed.entryList.Insert(0, newEntry);
            entryListComboBox.Items.Clear();
            updateComboBox();

            // 追加した日記へ移動する
            entryListComboBox.SelectedIndex = 0;
        }

        /// <summary>
        /// 削除ボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void removeEntryButton_Click(object sender, EventArgs e)
        {
            // コンボボックス上で日記が選択されていない場合は何も行わない
            if (entryListComboBox.SelectedIndex == -1)
                return;

            DialogResult result = MessageBox.Show("現在編集中の日記を削除しますか？", "確認", MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
            if (result == DialogResult.Cancel)
                return;

            if (entryListComboBox.SelectedIndex >= 0)
            {
                feed.entryList.RemoveAt(entryListComboBox.SelectedIndex);
            }
            updateComboBox();

            entryListComboBox.Text = string.Empty;

            entryListComboBox.SelectedIndex = -1;
            foreach (TextBox textBox in entryTextBoxList)
            {
                textBox.Clear();
            }
            contentAzukiControl.Text = string.Empty;
            entryDatePicker.Value = DateTime.Now;
            entryTimePicker.Value = DateTime.Now;
        }

        /// <summary>
        /// 終了ボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void exitToolStripMenuItem_Click(object sender, EventArgs e)
        {
            DialogResult result = MessageBox.Show("終了しますか？", "確認", MessageBoxButtons.OKCancel, MessageBoxIcon.Question);
            if (result == DialogResult.OK)
            {
                Environment.Exit(0);
            }
        }

        /// <summary>
        /// コンボボックスに無効な値が挿入されていた場合、テキストボックスなどをロックします
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void entryListComboBox_TextChanged(object sender, EventArgs e)
        {
            if (entryListComboBox.SelectedIndex == -1)
            {
                foreach (TextBox textBox in entryTextBoxList)
                {
                    textBox.ReadOnly = true;
                }
                contentAzukiControl.IsReadOnly = true;
            }
            else
            {
                foreach (TextBox textBox in entryTextBoxList)
                {
                    textBox.ReadOnly = false;
                }
                contentAzukiControl.IsReadOnly = false;
            }
        }

        /// <summary>
        /// バージョン情報を表示します
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void versionInfoToolStripMenuItem_Click(object sender, EventArgs e)
        {
            StringBuilder buffer = new StringBuilder();
            buffer.AppendLine(Application.ProductName +" " + Application.ProductVersion);

            MessageBox.Show(buffer.ToString(), "バージョン情報", MessageBoxButtons.OK, MessageBoxIcon.Information);
        }

        /// <summary>
        /// ツールボックスのAボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void aTagToolStripButton1_Click(object sender, EventArgs e)
        {
            List<string> aTag = new List<string>();

            aTag.Add("<a href=\"\" target=\"_blank\" >");
            aTag.Add("</a>");
            insertTag(aTag);
        }

        /// <summary>
        /// ツールボックスのBボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void spanBTagStripButton_Click(object sender, EventArgs e)
        {
            List<string> bTag = new List<string>();
            bTag.Add("<strong>");
            bTag.Add("</strong>");
            insertTag(bTag);
        }

        /// <summary>
        /// ツールボックスのSボタンを押したときの処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void spanTagStripButton_Click(object sender, EventArgs e)
        {
            List<string> sTag = new List<string>();
            sTag.Add("<span style=\"\">");
            sTag.Add("</span>");
            insertTag(sTag);
        }

        /// <summary>
        /// ツールボックスのイメージボタンを押した時の処理です
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void imgTagStripButton1_Click(object sender, EventArgs e)
        {
            if (entryListComboBox.SelectedIndex == -1)
                return;

            ImgForm imgForm = new ImgForm();
            imgForm.StartPosition = FormStartPosition.CenterScreen;
            DialogResult result = imgForm.ShowDialog();
            if (result == DialogResult.OK)
            {
                string imgPath;
                int width, height;
                imgForm.returnImgStatus(out imgPath, out width, out height);
                List<string> imgTag = new List<string>();
                imgTag.Add("<img src=\"" + imgPath + "\" width=\"" + width + "\" height=\"" + height + "\">");
                imgTag.Add("");
                insertTag(imgTag);
            }
            else
            {
                return;
            }
        }

        /// <summary>
        /// オプションボタンを押した際の処理です。設定用のフォームを表示し、設定内容を反映します。
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void optionToolStripMenuItem_Click(object sender, EventArgs e)
        {
            config.contentFont = contentAzukiControl.Font;

            ConfigForm configForm = new ConfigForm(config);
            DialogResult result = configForm.ShowDialog(this);
            if (result == DialogResult.OK)
            {
                this.config = configForm.config;
                appliConfig(config);
                // config.xmlを更新する
                config.writeConfigFormFile();
            }
        }

    }
}
