/*
 * Projrct F-11 - Web SCADA for Java
 * Copyright (C) 2002 Freedom, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

package org.F11.scada.server.deploy;

import java.io.File;
import java.io.FileFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.SortedSet;
import java.util.Timer;
import java.util.TimerTask;
import java.util.TreeSet;

import org.F11.scada.EnvironmentManager;
import org.apache.log4j.Logger;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

/**
 * y[W`t@C̔zXLi[NXłB
 * 
 * @author Hideaki Maekawa <frdm@users.sourceforge.jp>
 */
public class PageFileDeploymentScanner {
	/** MOAPI */
	private static Logger logger = Logger
			.getLogger(PageFileDeploymentScanner.class);
	/** XLJԂԊułB */
	private volatile long period;
	/** XL^C}[IuWFNg */
	private Timer timer;
	/** z[gfBNg̃Xgł */
	private List fileList = Collections.synchronizedList(new ArrayList());
	/**
	 * fBNggqjavãt@C𒊏otB^[ł
	 */
	private static FileFilter FILTER = new FileFilter() {
		public boolean accept(File pathname) {
			return pathname.isDirectory()
					|| pathname.getName().endsWith(".xml") ? true : false;
		}
	};
	/** zς݃y[W`t@CSetIuWFNgł */
	private Set deployedSet = Collections.synchronizedSet(new HashSet());
	/** zt@CNX̃Rp[^ł */
	private static Comparator SORTER = new Comparator() {
		public int compare(Object o1, Object o2) {
			DeployedFile f1 = (DeployedFile) o1;
			DeployedFile f2 = (DeployedFile) o2;
			return f1.file.compareTo(f2.file);
		}
	};
	/** zIuWFNg̎Q */
	private Deployer deployer;
	/** bNIuWFNg */
	private Lock lock;
	/** bNϐ */
	private Condition condition;

	/**
	 * y[W`zIuWFNgw肵ď܂B
	 * 
	 * @param deployer y[W`zIuWFNg
	 */
	public PageFileDeploymentScanner(Deployer deployer, long period) {
		this.deployer = deployer;
		this.lock = new ReentrantLock();
		this.condition = lock.newCondition();
		this.period = period;
		reSchedule();
	}

	/**
	 * y[W`zIuWFNgw肵ď܂B
	 * 
	 * @param deployer y[W`zIuWFNg
	 */
	public PageFileDeploymentScanner(
			Deployer deployer,
			Lock lock,
			Condition condition) {
		this.deployer = deployer;
		this.lock = lock;
		this.condition = condition;
		period = createPeripod();
		reSchedule();
	}

	private long createPeripod() {
		return Long.parseLong(EnvironmentManager.get(
				"/server/deploy/period",
				"69896"));
	}

	private void reSchedule() {
		timer = new Timer();
		timer.schedule(new ScanTimerTask(), new Date(), period);
	}

	/**
	 * y[W`̃XLjO~܂B
	 */
	public void terminate() {
		timer.cancel();
	}

	/**
	 * XLjOpx~bŕԂ܂
	 * 
	 * @return XLjOpx~bŕԂ܂
	 */
	public long getPeriod() {
		return period;
	}

	/**
	 * XLjOpx~bŐݒ肵܂
	 * 
	 * @param period XLjOpx~bŐݒ肵܂
	 */
	public void setPeriod(long period) {
		this.period = period;
		terminate();
		reSchedule();
	}

	/*
	 * yagni z[gfBNgݒ肵܂ @param list z[gfBNg̃Xg public void
	 * setFileList(List list) { if (list == null) { throw new
	 * IllegalArgumentException("list is null."); }
	 * 
	 * fileList.clear();
	 * 
	 * for (Iterator it = list.iterator(); it.hasNext(); ) { File file = (File)
	 * it.next(); addFile(file); } }
	 */

	/**
	 * z[gfBNgǉ܂B
	 * 
	 * @param file ǉz[gfBNg
	 */
	public void addFile(File file) {
		if (file == null) {
			throw new IllegalArgumentException("file is null.");
		}

		fileList.add(file);
	}

	/**
	 * z[gfBNg폜܂B
	 * 
	 * @param file 폜z[gfBNg
	 */
	public void removeFile(File file) {
		if (file == null) {
			throw new IllegalArgumentException("file is null.");
		}

		fileList.remove(file);
	}

	/**
	 * XL^C}[^XN̎NXłB
	 * 
	 * @author Hideaki Maekawa <frdm@users.sourceforge.jp>
	 */
	private class ScanTimerTask extends TimerTask {

		/**
		 * t@C𒲍āAzEzs܂B
		 */
		public void run() {
			// t@Cꗗ
			SortedSet filesToDeploy = new TreeSet();

			synchronized (fileList) {
				for (Iterator it = fileList.iterator(); it.hasNext();) {
					File root = (File) it.next();
					if (root.exists()) {
						if (root.isDirectory()) {
							FileLister lister = new FileLister();
							filesToDeploy
									.addAll(lister.listFiles(root, FILTER));
						} else {
							filesToDeploy.add(root);
						}
					}
				}
			}

			// VKt@Cƍ폜t@C𔻒
			List filesToRemove = new ArrayList();
			List filesToCheckForUpdate = new ArrayList();
			synchronized (deployedSet) {
				// 폜t@CƍXV`FbNt@CU蕪
				for (Iterator i = deployedSet.iterator(); i.hasNext();) {
					DeployedFile deployedFile = (DeployedFile) i.next();
					if (filesToDeploy.contains(deployedFile.file)) {
						filesToCheckForUpdate.add(deployedFile);
					} else {
						filesToRemove.add(deployedFile);
					}
				}
			}

			lock.lock();
			try {
				deployAndUndeploy(
						filesToDeploy,
						filesToRemove,
						filesToCheckForUpdate);
				condition.signal();
			} finally {
				lock.unlock();
			}
		}

		private void deployAndUndeploy(
				SortedSet filesToDeploy,
				List filesToRemove,
				List filesToCheckForUpdate) {
			// 폜t@Cz
			for (Iterator i = filesToRemove.iterator(); i.hasNext();) {
				DeployedFile deployedFile = (DeployedFile) i.next();
				if (logger.isDebugEnabled()) {
					logger.debug("Removing " + deployedFile.file);
				}
				undeploy(deployedFile);
			}

			// XV`FbNt@CXVt@C𒊏o
			ArrayList filesToUpdate = new ArrayList(filesToCheckForUpdate
					.size());
			for (Iterator i = filesToCheckForUpdate.iterator(); i.hasNext();) {
				DeployedFile deployedFile = (DeployedFile) i.next();
				if (deployedFile.isModified()) {
					if (logger.isDebugEnabled()) {
						logger.debug("Re-deploying " + deployedFile.file);
					}
					filesToUpdate.add(deployedFile);
				}
			}
			// XVt@C\[g
			Collections.sort(filesToUpdate, SORTER);
			// XVt@CĔz
			for (int i = filesToUpdate.size() - 1; i >= 0; i--) {
				undeploy((DeployedFile) filesToUpdate.get(i));
			}
			for (int i = 0; i < filesToUpdate.size(); i++) {
				deploy((DeployedFile) filesToUpdate.get(i));
			}

			// VKt@Cz
			// Collections.sort(filesToDeploy);
			for (Iterator i = filesToDeploy.iterator(); i.hasNext();) {
				File file = (File) i.next();
				DeployedFile deployedFile = new DeployedFile(file);
				if (!deployedSet.contains(deployedFile)) {
					if (logger.isDebugEnabled()) {
						logger.debug("Deploying " + deployedFile.file);
					}
					deploy(deployedFile);
				}
			}
		}
	}

	/**
	 * z
	 * 
	 * @param df zt@CIuWFNg
	 */
	private void deploy(final DeployedFile df) {
		if (deployer == null) {
			logger.fatal("Deployer not set.");
			return;
		}

		if (logger.isDebugEnabled()) {
			logger.debug("Deploying: " + df);
		}

		try {
			deployer.deploy(df.file);
			// } catch (IncompleteDeploymentException e) {
			// lastIncompleteDeploymentException = e;
		} catch (DeploymentException e) {
			logger.error("Failed to deploy: " + df + e);
			e.printStackTrace();
			return;
		} // end of try-catch

		df.deployed();

		if (!deployedSet.contains(df)) {
			deployedSet.add(df);
		}
	}

	/**
	 * z
	 * 
	 * @param df zt@CIuWFNg
	 */
	private void undeploy(final DeployedFile df) {
		try {
			if (logger.isDebugEnabled()) {
				logger.debug("Undeploying: " + df);
			}
			deployer.undeploy(df.file);
			deployedSet.remove(df);
		} catch (Exception e) {
			logger.error("Failed to undeploy: " + df, e);
		}
	}

	/**
	 * zς݂̃t@CIuWFNg\NXłB
	 */
	private class DeployedFile {
		/** zy[W`t@C */
		public File file;
		/** zŏIt */
		public long deployedLastModified;

		/**
		 * RXgN^
		 * 
		 * @param file y[W`t@C
		 */
		public DeployedFile(final File file) {
			this.file = file;
		}

		/**
		 * ̃IuWFNg̔zs܂B
		 */
		public void deployed() {
			deployedLastModified = getLastModified();
		}

		/**
		 * ̃t@CIuWFNgAt@C\ꍇ true Ԃ܂B
		 * 
		 * @return ̃t@CIuWFNgAt@C\ꍇ true Ԃ܂B
		 */
		public boolean isFile() {
			return file.isFile();
		}

		/**
		 * t@CIuWFNgԂ܂B
		 * 
		 * @return t@CIuWFNgԂ܂B
		 */
		public File getFile() {
			return file;
		}

		/**
		 * ̔zς݃y[W`t@C폜Ăꍇ true Ԃ܂B
		 * 
		 * @return ̔zς݃y[W`t@C폜Ăꍇ true Ԃ܂B
		 */
		public boolean isRemoved() {
			if (isFile()) {
				return !file.exists();
			}
			return false;
		}

		/**
		 * t@C̍ŏIXVtԂ܂
		 * 
		 * @return t@C̍ŏIXVtԂ܂
		 */
		public long getLastModified() {
			return file.lastModified();
		}

		/**
		 * t@CXVĂꍇ true Ԃ܂
		 * 
		 * @return t@CXVĂꍇ true Ԃ܂
		 */
		public boolean isModified() {
			long lastModified = getLastModified();
			if (lastModified == -1) {
				return false;
			}
			return deployedLastModified != lastModified;
		}

		/**
		 * nbVR[hԂ܂
		 * 
		 * @return nbVR[hԂ܂
		 */
		public int hashCode() {
			return file.hashCode();
		}

		/**
		 * ̃IuWFNgƂ̃IuWFNgrʂԂ܂B
		 * 
		 * @param other rIuWFNg
		 */
		public boolean equals(final Object other) {
			if (other instanceof DeployedFile) {
				return ((DeployedFile) other).file.equals(this.file);
			}
			return false;
		}

		/**
		 * ̃IuWFNg̕\Ԃ܂B
		 */
		public String toString() {
			return super.toString() + "{ file=" + file
					+ ", deployedLastModified=" + deployedLastModified + " }";
		}
	}
}
