/*************************************************************************
 *
 * Copyright 2009 by bBreak Systems.
 *
 * ExCella Reports - Excelt@C𗘗p[c[
 *
 * $Id: ReportProcessor.java 17 2009-06-24 03:19:58Z tomo-shibata $
 * $Revision: 17 $
 *
 * This file is part of ExCella Reports.
 *
 * ExCella Reports is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3
 * only, as published by the Free Software Foundation.
 *
 * ExCella Reports is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License version 3 for more details
 * (a copy is included in the COPYING.LESSER file that accompanied this code).
 *
 * You should have received a copy of the GNU Lesser General Public License
 * version 3 along with ExCella Reports .  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0-standalone.html>
 * for a copy of the LGPLv3 License.
 *
 ************************************************************************/
package org.bbreak.excella.reports.processor;

import java.io.FileInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.poi.ss.usermodel.Sheet;
import org.apache.poi.ss.usermodel.Workbook;
import org.apache.poi.ss.usermodel.WorkbookFactory;
import org.bbreak.excella.core.BookController;
import org.bbreak.excella.core.BookData;
import org.bbreak.excella.core.SheetData;
import org.bbreak.excella.core.exception.ExportException;
import org.bbreak.excella.core.exception.ParseException;
import org.bbreak.excella.core.exporter.book.BookExporter;
import org.bbreak.excella.reports.exporter.ReportBookExporter;
import org.bbreak.excella.reports.listener.RemoveControlRowAdapter;
import org.bbreak.excella.reports.listener.ReportProcessListener;
import org.bbreak.excella.reports.model.ConvertConfiguration;
import org.bbreak.excella.reports.model.ReportBook;
import org.bbreak.excella.reports.model.ReportSheet;
import org.bbreak.excella.reports.tag.ReportsTagParser;
import org.bbreak.excella.reports.util.ReportsUtil;

/**
 * [vZbT
 * 
 * @since 1.0
 */
public class ReportProcessor {

    /**
     * ^Op[T[Map<BR>
     * L[F^O<BR>
     * lF钠[p^Op[T[
     */
    private Map<String, ReportsTagParser<?>> parsers = new HashMap<String, ReportsTagParser<?>>();

    /**
     * GNX|[^[Map<BR>
     * L[Fo̓^Cv<BR>
     * lFGNX|[^[
     */
    private Map<String, ReportBookExporter> exporters = new HashMap<String, ReportBookExporter>();

    /**
     * Xi[
     */
    private List<ReportProcessListener> listeners = new ArrayList<ReportProcessListener>();

    /**
     * RXgN^
     */
    public ReportProcessor() {
        // ftHg̃^Op[T[̍쐬
        // Parser쐬
        parsers.putAll( ReportCreateHelper.createDefaultParsers());

        // ftHg̃GNX|[^[̍쐬
        // Exporter쐬
        exporters.putAll( ReportCreateHelper.createDefaultExporters());
    }

    /**
     * ϊsB
     * 
     * @param reportBooks [
     * @throws IOException t@C̓ǂݍ݂Ɏsꍇ
     * @throws ParseException ϊɎsꍇ
     * @throws ExportException o͏Ɏsꍇ
     */
    public void process( ReportBook... reportBooks) throws Exception {
        // o̓ubNf[^ɏ
        for ( ReportBook reportBook : reportBooks) {
            processBook( reportBook);
        }
    }

    /**
     * [NubN̕ϊsB
     * 
     * @param reportBook [NubN̒u
     * @throws IOException t@C̓ǂݍ݂Ɏsꍇ
     * @throws ParseException ϊɎsꍇ
     * @throws ExportException o͏Ɏsꍇ
     */
    private void processBook( ReportBook reportBook) throws Exception {

        if ( reportBook == null) {
            return;
        }

        Workbook workbook = getTemplateWorkbook( reportBook.getTemplateFileName());

        for ( ReportProcessListener listener : listeners) {
            listener.preBookParse( workbook, reportBook);
        }

        try {

            checkReportBook( reportBook);

            // ev[gWJ
            Set<Integer> delTemplateIndexs = expandTemplate( workbook, reportBook);

            // o̓t@C擾
            BookController controller = new BookController( workbook);

            // Parser̐ݒ
            for ( ReportsTagParser<?> tagParser : parsers.values()) {
                controller.addTagParser( tagParser);
            }

            // Xi[̐ݒ
            controller.addSheetParseListener( new RemoveControlRowAdapter());
            for ( ReportProcessListener listener : listeners) {
                controller.addSheetParseListener( listener);
            }

            // Exporter̐ݒ
            for ( ConvertConfiguration configuration : reportBook.getConfigurations()) {
                if ( configuration == null) {
                    continue;
                }
                for ( ReportBookExporter reportExporter : exporters.values()) {
                    if ( configuration.getFormatType().equals( reportExporter.getFormatType())) {
                        reportExporter.setConfiguration( configuration);
                        reportExporter.setFilePath( reportBook.getOutputFileName() + reportExporter.getExtention());
                        controller.addBookExporter( reportExporter);
                    }
                }
            }

            ReportsParserInfo reportsParserInfo = new ReportsParserInfo();
            reportsParserInfo.setReportParsers( new ArrayList<ReportsTagParser<?>>( parsers.values()));
            reportsParserInfo.setReportBook( reportBook);

            BookData bookData = controller.getBookData();
            bookData.clear();
            for ( String sheetName : controller.getSheetNames()) {
                if ( sheetName.startsWith( BookController.COMMENT_PREFIX)) {
                    continue;
                }
                ReportSheet reportSheet = ReportsUtil.getReportSheet( sheetName, reportBook);
                if ( reportSheet != null) {

                    reportsParserInfo.setParamInfo( reportSheet.getParamInfo());

                    // ͂̎s
                    SheetData sheetData = controller.parseSheet( sheetName, reportsParserInfo);
                    // ʂ̒ǉ
                    controller.getBookData().putSheetData( sheetName, sheetData);
                }
            }

            // svev[g폜
            for ( Integer deleteSheetIndex : delTemplateIndexs) {
                workbook.removeSheetAt( deleteSheetIndex);
            }

            // o͏̎s
            for ( BookExporter exporter : controller.getExporter()) {
                if ( exporter != null) {
                    exporter.setup();
                    try {
                        exporter.export( workbook, bookData);
                    } finally {
                        exporter.tearDown();
                    }
                }
            }
        } finally {
            for ( ReportProcessListener listener : listeners) {
                listener.postBookParse( workbook, reportBook);
            }
        }

    }

    /**
     * [NubN`FbN
     * 
     * @param reportBook [NubN̒u
     */
    private void checkReportBook( ReportBook reportBook) {
    }

    /**
     * ev[g̃[NubN擾B
     * 
     * @param filepath ev[gt@CpX
     * @return ev[g[NubN
     * @throws IOException t@C̓ǂݍ݂Ɏsꍇ
     */
    private Workbook getTemplateWorkbook( String filepath) throws Exception {

        FileInputStream fileIn = null;
        Workbook wb = null;

        try {
            fileIn = new FileInputStream( filepath);
            wb = WorkbookFactory.create( fileIn);
        } finally {
            if ( fileIn != null) {
                fileIn.close();
            }
        }

        return wb;
    }

    /**
     * ^Op[T[ǉB
     * 
     * @param tagParser ǉ^Op[T[
     */
    public void addReportsTagParser( ReportsTagParser<?> tagParser) {
        parsers.put( tagParser.getTag(), tagParser);
    }

    /**
     * ^Õ^Op[T[폜B
     * 
     * @param tag ^O
     */
    public void removeReportsTagParser( String tag) {
        parsers.remove( tag);
    }

    /**
     * ׂẴ^Op[T[폜B
     */
    public void clearReportsTagParser() {
        parsers.clear();
    }

    /**
     * GNX|[^[ǉB
     * 
     * @param exporter ǉGNX|[^[
     */
    public void addReportBookExporter( ReportBookExporter exporter) {
        exporters.put( exporter.getFormatType(), exporter);
    }

    /**
     * tH[}bg^CṽGNX|[^[폜B
     * 
     * @param formatType tH[}bg^Cv
     */
    public void removeReportBookExporter( String formatType) {
        exporters.remove( formatType);
    }

    /**
     * ׂẴGNX|[^[폜B
     */
    public void clearReportBookExporter() {
        exporters.clear();
    }

    /**
     * Xi[ǉB
     * 
     * @param listener Xi[
     */
    public void addReportProcessListener( ReportProcessListener listener) {
        listeners.add( listener);
    }

    /**
     * Xi[폜B
     * 
     * @param listener Xi[
     */
    public void removeReportProcessListener( ReportProcessListener listener) {
        listeners.remove( listener);
    }

    /**
     * ev[g[NubÑV[gA[o͒PʂɕϊB
     * 
     * @param workbook ev[g[NubN
     * @param reportBook [[NubN
     * @return 폜Kvȃev[gV[gCfbNX
     */
    private Set<Integer> expandTemplate( Workbook workbook, ReportBook reportBook) {

        Set<Integer> delTemplateIndexs = new TreeSet<Integer>(Collections.reverseOrder());
        Set<Integer> useTemplateIndexs = new HashSet<Integer>();

        // o̓V[gPʂɃRs[
        for ( ReportSheet reportSheet : reportBook.getReportSheets()) {

            if ( reportSheet != null) {
                if ( reportSheet.getSheetName().equals( reportSheet.getTemplateName())) {
                    // ev[go̓V[g
                    useTemplateIndexs.add( workbook.getSheetIndex( reportSheet.getSheetName()));
                } else {
                    int tempIdx = workbook.getSheetIndex( reportSheet.getTemplateName());

                    Sheet sheet = workbook.cloneSheet( tempIdx);
                    workbook.setSheetName( workbook.getSheetIndex( sheet), reportSheet.getSheetName());
                    delTemplateIndexs.add( tempIdx);
                }
            }
        }

        delTemplateIndexs.removeAll( useTemplateIndexs);

        return delTemplateIndexs;

    }

}
