﻿/*
 * Created by SharpDevelop.
 * User: banana
 * Date: 2013/10/11
 * Time: 10:56
 * 
 * To change this template use Tools | Options | Coding | Edit Standard Headers.
 */
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Linq;
using System.Windows.Forms;
using EWatch;
using EWatch.Models.Configurations;
using EWatch.Utils;
using EWatchUI.Controls;

namespace EWatchUI
{
	/// <summary>
	/// Description of MainForm.
	/// </summary>
	public partial class MainForm : Form
	{
		private Configuration config;
		private WatchType ShownType {
			get {
				if(radioButtonEventLog.Checked) return WatchType.EventLog;
				if(radioButtonFile.Checked) return WatchType.File;
				if(radioButtonSyslog.Checked) return WatchType.Syslog;
				throw new Exception("Invalid radio button selection");
			}
		}
		
		private bool _dirty = false;
		private bool Dirty {
			get { return _dirty; }
			set {
				_dirty = value;
				buttonSave.Enabled = _dirty;
				buttonReload.Enabled = _dirty;
			}
		}
		
		public MainForm()
		{
			//
			// The InitializeComponent() call is required for Windows Forms designer support.
			//
			InitializeComponent();
			config = (Configuration)(new LoadConfig()).Execute();
			SetListItems();
			Dirty = false;
		}
		
		private void SetListItems()
		{
			listViewConfig.Items.Clear();
			AddConfigListItem(ShownType, GetCurrentActionDictionary());
		}
		
		private void AddConfigListItem(WatchType type, Dictionary<string,List<ActionTable>> rules)
		{
			List<string> keys = new List<string>(rules.Keys);
			keys.Sort();
			foreach(string key in keys) {
				int index = 0;
				foreach(ActionTable table in rules[key]) {
					listViewConfig.Items.Add(ConvertToListViewItem(type.ToString(), key, table, index++));
				}
			}
		}
		
		private ListViewItem ConvertToListViewItem(string type, string target, ActionTable table, int tag)
		{
			ListViewItem item = new ListViewItem(type);
			item.SubItems.Add(target);
			item.SubItems.Add(table.LogLevel);
			item.SubItems.Add(table.EventSource);
			item.SubItems.Add(table.EventId != null ? table.EventId.ToString() : null);
			item.SubItems.Add(table.HostName);
			item.SubItems.Add(table.MatchPattern);
			item.SubItems.Add(table.IgnorePattern);
			item.SubItems.Add(table.Actions.AsEnumerable().Count().ToString());
			item.Tag = tag;
			return item;
		}
		
		private ListViewItem GetSelectedItem()
		{
			if(listViewConfig.SelectedItems.Count == 1){
				return listViewConfig.SelectedItems[0];
			}
			return null;
		}
		
		private ActionTable GetActionTable(Dictionary<string,List<ActionTable>> actionDic, ListViewItem item)
		{
			string key = item.SubItems[1].Text;
			return actionDic[key][(int)item.Tag];
		}
		
		private Dictionary<string,List<ActionTable>> GetCurrentActionDictionary()
		{
			switch(ShownType){
				case WatchType.EventLog:
					return config.EventLog;
				case WatchType.File:
					return config.File;
				case WatchType.Syslog:
					return config.Syslog;
				default:
					throw new Exception("Invalid value for WatchType");
			}
		}

		private List<ActionTable> GetActionTableList(Dictionary<string,List<ActionTable>> rules, string target)
		{
			if(!rules.ContainsKey(target)){
				List<ActionTable> result = new List<ActionTable>();
				rules.Add(target, result);
			}
			return rules[target];
		}
		

		private void SetFormRuleEditData(FormRuleEdit form, ListViewItem item)
		{
			Dictionary<string,List<ActionTable>> actionDic = GetCurrentActionDictionary();
			form.TargetValues = new SortedSet<string>(actionDic.Keys).ToArray();
			form.Type = ShownType;
			if(item != null){
				ActionTable table = new ActionTable(GetActionTable(actionDic, item));
				string target = item.SubItems[1].Text;
				form.Target = target;
				form.Table = table;
			}
		}
		
		void ButtonAddClick(object sender, EventArgs e)
		{
			FormRuleEdit form = new FormRuleEdit();
			SetFormRuleEditData(form, null);
			if(form.ShowDialog() == DialogResult.OK){
				List<ActionTable> tables = GetActionTableList(GetCurrentActionDictionary(), form.Target);
				tables.Add(form.Table);
				SetListItems();
				Dirty = true;
			}
		}
		
		void ButtonCopyClick(object sender, EventArgs e)
		{
			ListViewItem item = GetSelectedItem();
			if(item != null){
				FormRuleEdit form = new FormRuleEdit();
				SetFormRuleEditData(form, item);
				if(form.ShowDialog() == DialogResult.OK){
					List<ActionTable> tables = GetActionTableList(GetCurrentActionDictionary(), form.Target);
					tables.Add(form.Table);
					SetListItems();
					Dirty = true;
				}
			}
		}
		
		void ButtonEditClick(object sender, EventArgs e)
		{
			ListViewItem item = GetSelectedItem();
			if(item != null){
				FormRuleEdit form = new FormRuleEdit();
				SetFormRuleEditData(form, item);
				if(form.ShowDialog() == DialogResult.OK){
					List<ActionTable> tables = GetActionTableList(GetCurrentActionDictionary(), form.Target);
					tables[(int)item.Tag] = form.Table;
					SetListItems();
					Dirty = true;
				}
			}
		}

		void ListViewConfigDoubleClick(object sender, EventArgs e)
		{
			ButtonEditClick(sender, e);
		}
		
		void ButtonDeleteClick(object sender, EventArgs e)
		{
			if(listViewConfig.SelectedItems.Count > 0){
				foreach(ListViewItem item in listViewConfig.SelectedItems){
					listViewConfig.Items.Remove(item);
				}
				Dirty = true;
			}
		}
		
		private void ChangeIndexTable(int oldIndex, int newIndex, List<ActionTable> tables)
		{
			if(oldIndex >= 0 && oldIndex < tables.Count &&
			   newIndex >= 0 && newIndex < tables.Count){
				ActionTable table = tables[oldIndex];
				tables.RemoveAt(oldIndex);
				tables.Insert(newIndex, table);
			}
		}
		
		private void ChangeIndexListView(int oldIndex, int newIndex, int oldTag, int newTag)
		{
			if(oldIndex < listViewConfig.Items.Count &&
			   oldIndex < listViewConfig.Items.Count){
				ListViewItem target = listViewConfig.Items[oldIndex];
				listViewConfig.Items.Remove(target);
				listViewConfig.Items.Insert(newIndex, target);
				target.Tag = newTag;
				target.Selected = true;
				listViewConfig.Items[oldIndex].Tag = oldTag;
			}
		}
		
		void ButtonUpClick(object sender, EventArgs e)
		{
			ListViewItem item = GetSelectedItem();
			if(item != null){
				string target = item.SubItems[1].Text;
				List<ActionTable> tables = GetActionTableList(GetCurrentActionDictionary(), target);
				int index = (int)item.Tag;
				if(index > 0){
					ChangeIndexTable(index, index-1, tables);
					ChangeIndexListView(item.Index, item.Index-1, index, index-1);
					Dirty = true;
				}
			}
		}
		
		void ButtonDownClick(object sender, EventArgs e)
		{
			ListViewItem item = GetSelectedItem();
			if(item != null){
				string target = item.SubItems[1].Text;
				List<ActionTable> tables = GetActionTableList(GetCurrentActionDictionary(), target);
				int index = (int)item.Tag;
				if(index < tables.Count-1){
					ChangeIndexTable(index, index+1, tables);
					ChangeIndexListView(item.Index, item.Index+1, index, index+1);
					Dirty = true;
				}
			}
		}
		
		void ButtonSaveClick(object sender, EventArgs e)
		{
			string error = (string)(new SaveConfig{ Config = config }).Execute();
			if(!Util.IsNullOrEmpty(error)){
				MessageBox.Show(string.Format("Save error.\n{0}", error),
				                "ERROR", MessageBoxButtons.OK,
				                MessageBoxIcon.Error);
				return;
			}
			Dirty = false;
		}

		void ButtonReloadClick(object sender, EventArgs e)
		{
			config = (Configuration)(new LoadConfig()).Execute();
			SetListItems();
			Dirty = false;
		}

		void ListViewConfigSelectedIndexChanged(object sender, EventArgs e)
		{
			
		}
		
		void RadioButtonEventLogClick(object sender, EventArgs e)
		{
			SetListItems();
		}
		
		void RadioButtonFileClick(object sender, EventArgs e)
		{
			SetListItems();
		}
		
		void RadioButtonSyslogClick(object sender, EventArgs e)
		{
			SetListItems();
		}
		
	}
}
