<?php
// vim: foldmethod=marker
/**
 *  Smarty3.php
 *
 *  @author     Sotaro Karasawa <sotaro.k@gmail.com>
 *  @license    http://www.opensource.org/licenses/bsd-license.php The BSD License
 *  @package    Ethna
 *  @version    $Id: 9e5c7b4e96b6e9ee7dc385bbb842533e76cb171c $
 */

require_once 'Smarty/Smarty.class.php';

// {{{ Ethna_Renderer_Smarty3
/**
 *  Smarty 3.x
 *
 *  @author     Sotaro Karasawa <sotaro.k@gmail.com>
 *  @package    Ethna
 */
class Ethna_Renderer_Smarty3 extends Ethna_Renderer
{
    /** @var    string compile directory  */
    var $compile_dir;

    /**
     *  Constructor for Ethna_Renderer_Smarty3
     *
     *  @access public
     */
    public function __construct($controller)
    {
        parent::__construct($controller);

        $this->engine = new Smarty();

        // Configurerd by controller
        $template_dir = $controller->getTemplatedir();
        $compile_dir = $controller->getDirectory('template_c');

        $this->setTemplateDir($template_dir);
        $this->compile_dir = $compile_dir;

        $this->engine->template_dir = $template_dir;
        $this->engine->compile_dir = $compile_dir;
        $this->engine->compile_id = md5($this->template_dir);

        //  デリミタは Ethna_Config を見る
        $smarty_config = isset($this->config['smarty3'])
                       ? $this->config['smarty3']
                       : array();
        if (isset($smarty_config['left_delimiter'])) {
            $this->engine->left_delimiter = $smarty_config['left_delimiter'];
        }
        if (isset($smarty_config['right_delimiter'])) {
            $this->engine->right_delimiter = $smarty_config['right_delimiter'];
        }

        // make compile dir
        if (is_dir($this->engine->compile_dir) === false) {
            Ethna_Util::mkdir($this->engine->compile_dir, 0755);
        }

        $this->engine->plugins_dir = array_merge(
            $controller->getDirectory('plugins'),
            array(ETHNA_BASE . '/class/Plugin/Smarty', SMARTY_DIR . 'plugins')
        );
    }

    /**
     *  Display the template
     *
     *  @param  string  $template   template name
     *  @param  bool    $capture    if true, not display but return as string
     *
     *  @access public
     */
    public function perform($template = null, $capture = false)
    {
        if ($template === null && $this->template === null) {
            return Ethna::raiseWarning('template is not defined');
        }

        if ($template !== null) {
            $this->template = $template;
        }

        try {
            if ((is_absolute_path($this->template) && is_readable($this->template))
                || is_readable($this->template_dir . $this->template)) {
                    if ($capture === true) {
                        $captured = $this->engine->fetch($this->template);
                        return $captured;
                    } else {
                        $this->engine->display($this->template);
                    }
            } else {
                return Ethna::raiseWarning('template not found ' . $this->template);
            }
        } catch (SmartyCompilerException $e) {
            $this->logger->log(LOG_ERR, "smarty compile error: msg='{$e->getMessage()}'");
            return Ethna::raiseError("smarty compile error: msg='{$e->getMessage()}'", 500);
        }
    }

    /**
     * get tamplate variable
     *
     *  @param      string      $name  variable name
     *  @return     mixed       variables
     *  @access     public
     */
    public function getProp($name = null)
    {
        $property = $this->engine->get_template_vars($name);

        if ($property !== null) {
            return $property;
        }
        return null;
    }

    /**
     *  remove template variable
     *
     *  @param  name    variable name
     *
     *  @access public
     */
    public function removeProp($name)
    {
        $this->engine->clear_assign($name);
    }

    /**
     *  set array to template variable
     *
     *  @param  array   $array
     *
     *  @access public
     */
    public function setPropArray($array)
    {
        $this->engine->assign($array);
    }

    /**
     *  set array to template variable by reference
     *
     *  @param  array   $array
     *  @access public
     */
    public function setPropArrayByRef(&$array)
    {
        $this->engine->assignByRef($array);
    }

    /**
     *  set template variable
     *
     *  @param  string  $name   variable name
     *  @param  mixed   $value  value
     *
     *  @access public
     */
    public function setProp($name, $value)
    {
        $this->engine->assign($name, $value);
    }

    /**
     *  set template variable by reference
     *
     *  @param  string  $name   variable name
     *  @param  mixed   $value  value
     *
     *  @access public
     */
    public function setPropByRef($name, &$value)
    {
        $this->engine->assignByRef($name, $value);
    }

    /**
     *  プラグインをセットする
     *
     *  @param  string  $name   plugin name
     *  @param  string  $type   plugin type
     *  @param  mixed   $plugin plugin
     *  @TODO   i don't know whether this is working or not
     *  @access public
     */
    public function setPlugin($name, $type, $plugin)
    {
        //プラグイン関数の有無をチェック
        if (is_callable($plugin) === false) {
            return Ethna::raiseWarning('Does not exists.');
        }

        //プラグインの種類をチェック
        $register_method = 'register_' . $type;
        if (method_exists($this->engine, $register_method) === false) {
            return Ethna::raiseWarning('This plugin type does not exist');
        }

        // フィルタは名前なしで登録
        if ($type === 'prefilter' || $type === 'postfilter' || $type === 'outputfilter') {
            parent::setPlugin($name, $type, $plugin);
            $this->engine->$register_method($plugin);
            return;
        }

        // プラグインの名前をチェック
        if ($name === '') {
            return Ethna::raiseWarning('Please set plugin name');
        }

        // プラグインを登録する
        parent::setPlugin($name, $type, $plugin);
        $this->engine->$register_method($name, $plugin);
    }
}
// }}}
