##############################################################################
#    pvm
#    software development environment
#
#    Copyright (C) 1997  Andrew Guryanov
#    andrew-guryanov@usa.net
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
##############################################################################

#===========================================
#	Global variables
#===========================================

global main; 
global env;

global titlepane
set titlepane ".t"
wm withdraw .
toplevel $titlepane -bd 4 -relief ridge
image create photo "title" -file $env(APP_HOME)/images/image.gif
wm overrideredirect $titlepane 1
label $titlepane.l -image title
pack $titlepane.l -expand 1 -fill both
set x [expr ([winfo screenwidth .] - [image width title])/2]
set y [expr ([winfo screenheight .] - [image height title])/2]
wm geometry $titlepane +$x+$y
wm deiconify $titlepane
after 3000 {catch {destroy $titlepane}}
update idletasks

#===========================================
#	Application initialization
#===========================================

proc main:InitApp {app_home} {
#--------------------------
    global main;
    global env;

    wm focusmodel . passive
    wm geometry . 200x200+0+0
    wm maxsize . 1265 994
    wm minsize . 1 1
    wm overrideredirect . 0
    wm resizable . 1 1
    wm withdraw .
    wm title . ""

    set main(app,title) "Paradigm Visual Make"
    set main(app,version) 1.23
    set main(app,home) $app_home
    set main(app,frame) ""
    set main(app,profile) "$env(HOME)/.pvm.res.$env(USER)"
    set main(app,icon) $app_home/images/icon.bmp
    set main(app,toolbar) 1
    set main(app,statusbar) 1
    set main(position,menubar) 0
    set main(position,toolbar) 0
    set main(position,statusbar) 2
    set main(position,leftright) 1
    set main(position,topbottom) 1

    set main(doc,types) ""
    set main(doc,count) 0

    set main(menu,frame) ""

    set main(tool,btn) ""

    set main(font,menu) "fixed"
    set main(font,tool) "fixed"
    set main(font,status) "fixed"
    set main(font,caption) "fixed"
    set main(font,def) "fixed"

    set main(color,workspacebg) "#dfdfdf"
    set main(color,activefg) "#ffffff"
    set main(color,activebg) "#0000ff"
    set main(color,inactivefg) "#000000"
    set main(color,inactivebg) "#ffffff"
    set main(color,tooltipfg) "#000000"
    set main(color,tooltipbg) "#ffff00"
    set main(color,textfg) "#000000"
    set main(color,textbg) "#ffffff"
    set main(color,textselfg) "#ffffff"
    set main(color,textselbg) "#000000"
    set main(color,projectfg) "#000000"
    set main(color,projectbg) "#ffffff"
    set main(color,projectselfg) "#ffffff"
    set main(color,projectselbg) "#000000"
    set main(color,outputfg) "#000000"
    set main(color,outputbg) "#ffffff"
    set main(color,outputselfg) "#ffffff"
    set main(color,outputselbg) "#000000"

    set main(recent,list) ""

    image create bitmap main(tool,new) -file $app_home/images/new.bmp
    image create bitmap main(tool,open) -file $app_home/images/open.bmp
    image create bitmap main(tool,save) -file $app_home/images/save.bmp
    image create bitmap main(tool,cut) -file $app_home/images/cut.bmp
    image create bitmap main(tool,copy) -file $app_home/images/copy.bmp
    image create bitmap main(tool,paste) -file $app_home/images/paste.bmp
    image create bitmap main(tool,compile) -file $app_home/images/compile.bmp
    image create bitmap main(tool,build) -file $app_home/images/build.bmp
    image create bitmap main(tool,rebuild) -file $app_home/images/rebuild.bmp
    image create bitmap main(tool,stop) -file $app_home/images/stop.bmp
    image create bitmap main(tool,exit) -file $app_home/images/exit.bmp


    set main(pipe,cmnd) ""
    set main(pipe,result) 1
    set main(single,find) ""
    set main(single,tool) ""


    bind btnPop <Enter> {%W config -relief raised}
    bind btnPop <Leave> {%W config -relief flat}
}

#===========================================
#	Application main frame event handlers
#===========================================

proc main:OnFileNew {} {
#--------------------------
    main:DoOpenFile ""
}

proc main:OnFileOpen {} {
#--------------------------
    global main;
    global tk_version
    if {$tk_version < 4.2} {
        set filename [tk_filesel $main(app,frame).filesel -title "Open file"]
    } else {
        set filename [tk_getOpenFile -parent $main(app,frame) -title "Open file"]
    }
    if {$filename != ""} {
        main:DoOpenFile $filename
    }
}

proc main:OnOpenRecentFile {filename} {
#--------------------------------------
    main:DoOpenFile $filename
}

proc main:OnFileClose {} {
#--------------------------------
    global main;
    if {[main:DocDispatch SaveIfModified]} {
        main:DocInWindowMenu "" remove
        return [mdiclient:DestroyActiveFrame]
    }
    return false
}

proc main:OnFileCloseAll {} {
#--------------------------------
    while {[main:OnFileClose]} {}
}

proc main:OnFileSave {} {
#--------------------------
    main:DoSaveDocument [mdiclient:GetActiveFrame]
}

proc main:OnFileSaveAs {} {
#--------------------------
    main:DoSaveDocument [mdiclient:GetActiveFrame] 1
}

proc main:OnFileTool {} {
#--------------------------
    global main
    if {![winfo exists $main(single,tool)]} {
        set main(single,tool) [dialog-tool:CreateDialog $main(app,frame)]
    } else {
        wm deiconify $main(single,tool)
        raise $main(single,tool)
    }
}

proc main:DoSaveDocument {fr_name {askpath 0}} {
#------------------------------------------------------
    global main;
    global tk_version
    if {$fr_name != ""} {
        set title [main:DocDispatch GetDocData $fr_name title]
        set filename [main:DocDispatch GetDocData $fr_name path]
        if {$filename == "" || $askpath} {
            set dir [pwd]
            if {$filename != ""} {
                set dir [file dirname $filename]
            }
            while {1} {
                if {$tk_version < 4.2} {
                    set filename [tk_filesel $main(app,frame).filesel -title "SaveAs file" -directory "$dir"]
                } else {
                    set filename [tk_getSaveFile -parent $main(app,frame) -title "SaveAs file" -initialdir "$dir"]
                }
                if {$filename == ""} {
                    return false
                }
                if {$tk_version < 4.2} {
                    if {[file exists $filename]} {
                        set ret [dialog:MessageBox $fr_name $title \
                            "$filename\n\nFile exists\nOverwrite it?" "Cancel-No-Yes"]
                        if {$ret == "no"} {
                            continue
                        }
                        if {$ret == "cancel"} {
                            return false
                        }
                    }
                }
                break
            }
        }
        if {$filename != ""} {
            main:DocDispatch OnSaveDocument $fr_name $filename
            main:DocInWindowMenu $fr_name remove
            main:DocInWindowMenu $fr_name add $filename
            main:UpdateRecentFiles $filename
            return true
        }
    } else {
        puts "there is nothing to save"
    }
    return false
}

proc main:OnFileExit {} {
#-------------------------------
    if {[mdiclient:SaveAllModified]} {
        main:SaveProfile
        exit;
    }
}

proc main:OnEditFind {} {
#------------------------------
    global main
    if {![winfo exists $main(single,find)]} {
        set main(single,find) [dialog-find:CreateDialog $main(app,frame)]
    } else {
        wm deiconify $main(single,find)
        raise $main(single,find)
    }
}

proc main:OnHelpAbout {} {
#--------------------------------
    global main
    set dlg [dialog-about:CreateDialog $main(app,frame)]
    main:DlgDispatch DoModal $dlg
}


proc main:OnMenuActivate {W} {
#---------------------------------
    main:OnMenuSelect $W
}

proc main:OnMenuDeactivate {W} {
#---------------------------------
    main:SetStatusBarText ""
}

proc main:OnMenuSelect {W} {
#----------------------------------------
    global main;
    set menubar [lindex [split $W .] 3]
    set menuitem ""
    set textfound 0
    if {$menubar != ""} {
        set menuitem [$W entrycget active -label]
        set menuitem [string tolower [string trimright $menuitem .]]
        regsub -all " " $menuitem "" menuitem
        if {$menuitem != ""} {
            if {[info exists main(menu,$menubar$menuitem)]} {
                main:SetStatusBarText "$main(menu,$menubar$menuitem)"
                set textfound 1
            } else {
                if {[info exists main(menu,$menubar)]} {
                    main:SetStatusBarText "$main(menu,$menubar)"
                    set textfound 1
                }
            }
        }
    }
    if {!$textfound} {
        main:SetStatusBarText ""
    }
    return
}

proc main:OnToolEnter {toolbtn} {
#--------------------------------
    global main
    set main(tool,btn) $toolbtn
    after 500 {main:TooltipShow}
}

proc main:OnToolLeave {toolbtn} {
#--------------------------------
    if {![main:TooltipDestroy]} {
        after cancel {main:TooltipShow}
    }
}

proc main:OnToolMove {toolbtn} {
#--------------------------------------------
    global main
    if {$main(tool,btn) != ""} {
        after cancel {main:TooltipShow}
        after 500 {main:TooltipShow}
    }
}

proc main:OnToolButton {toolbtn} {
#--------------------------------
    global main
    if {![main:TooltipDestroy]} {
        after cancel {main:TooltipShow}
        set main(tool,btn) ""
    }
}

proc main:OnWindowStatusbar {} {
#--------------------------------
    global main
    set bar $main(app,frame).status
    if {[winfo exists ${bar}]} {
        destroy ${bar}
        set main(app,statusbar) 0
    } else {
        if {$main(position,leftright)} {
            set anchor_0 "w"
            set side_0 "left"
            set side_1 "right";
        } else {
            set anchor_0 "e"
            set side_0 "right"
            set side_1 "left";
        }
        frame ${bar}  -borderwidth 1 -height 30 -relief sunken -width 30 
        label ${bar}.l0  -anchor $anchor_0 -relief groove -width 27 
        label ${bar}.l1  -anchor center -relief groove -width 10 
        if {$main(position,statusbar) == 2} {
            pack ${bar}  -anchor center -expand 0 -fill x -side bottom
        } else {
            pack ${bar}  -anchor center -expand 0 -fill x -side top -before $main(app,frame).client
        }
        pack ${bar}.l0  -anchor center -expand 1 -fill both -side $side_0
        pack ${bar}.l1  -anchor center -expand 0 -side $side_1
        main:ConfigureStatusbarAppearance
        set main(app,statusbar) 1
    }
}

proc main:OnWindowToolbar {} {
#---------------------------------
    global main
    set bar $main(app,frame).tool
    if {[winfo exists ${bar}]} {
        destroy ${bar}
        set main(app,toolbar) 0
    } else {
        frame ${bar} -borderwidth 1 -height 1 -relief sunken -width 1 
        switch $main(position,toolbar) {
        "1"         {if {$main(position,topbottom)} {set side_btn "top"} {set side_btn "bottom"}; set after_frame $main(menu,frame); set fill_frame "y"; set side_frame "left";}
        "2"         {if {$main(position,leftright)} {set side_btn "left"} {set side_btn "right"}; set after_frame $main(app,frame).client; set fill_frame "x"; set side_frame "top";}
        "3"         {if {$main(position,topbottom)} {set side_btn "top"} {set side_btn "bottom"}; set after_frame $main(menu,frame); set fill_frame "y"; set side_frame "right";}
        "default"   {if {$main(position,leftright)} {set side_btn "left"} {set side_btn "right"}; set after_frame $main(menu,frame); set fill_frame "x"; set side_frame "top";}
        }
        pack ${bar} -after $after_frame -anchor center -expand 0 -fill $fill_frame -side $side_frame
        set idle 0
        foreach i { \
                new open save separator \
                cut copy paste separator \
                compile build rebuild stop separator \
                exit} {
            switch $i {
            "new"     {set cmnd "main:OnFileNew"}
            "open"    {set cmnd "main:OnFileOpen"}
            "save"    {set cmnd "main:OnFileSave"}
            "cut"     {set cmnd "main:DocDispatch OnEditCut"}
            "copy"    {set cmnd "main:DocDispatch OnEditCopy"}
            "paste"   {set cmnd "main:DocDispatch OnEditPaste"}
            "compile" {set cmnd {main:DocTypeDispatch doc-project OnProjectBuild "" deb select}}
            "build"   {set cmnd {main:DocTypeDispatch doc-project OnProjectBuild}}
            "rebuild" {set cmnd {main:DocTypeDispatch doc-project OnProjectBuild "" deb clean all}}
            "stop"    {set cmnd "main:DocTypeDispatch doc-project OnStopBuild"}
            "exit"    {set cmnd "main:OnFileExit"}
            "default" {incr idle; set cmnd ""}
            }
            if {$cmnd != ""} {
                button $bar.$i -padx 8 -pady 2 -borderwidth 1 -relief flat -image main(tool,${i}) -command $cmnd
                bindtags $bar.$i "btnPop [bindtags $bar.$i]"
                main:BindToolbutton ${bar}.${i}
                pack $bar.$i -anchor center -expand 0 -fill none -padx 0 -side $side_btn
            } else {
                button $bar.$idle -padx 0 -pady 0 -relief flat -state disabled -borderwidth 1
                pack $bar.$idle -anchor center -expand 0 -fill none -padx 0 -side $side_btn
            }
        }
        set main(app,toolbar) 1
    }
}

proc main:BindToolbutton {toolbtn} {
#--------------------------------------------
    if {[winfo class $toolbtn] == "Button"} {
        bind $toolbtn <Enter>  {main:OnToolEnter %W}
        bind $toolbtn <Leave>  {main:OnToolLeave %W}
        bind $toolbtn <Motion> {main:OnToolMove %W}
        bind $toolbtn <Button> {main:OnToolButton %W}
    }
}

proc main:OnWindowPreferences {} {
#--------------------------------------------
    global main
    set dlg [dialog-prefs:CreateDialog $main(app,frame)]
    main:DlgDispatch DoModal $dlg
}

#===========================================
#	Application main frame methods
#===========================================

proc main:GetActiveToplevel {} {
#-------------------------------------------
    set w [focus]
    while {[winfo exists $w]} {
        set classname [winfo class $w]
        if {$classname == "Toplevel" || $classname == "Dialog"} {
            return $w
        } else {
            set w [winfo parent $w]
        }
    }
    return ""
}

proc main:GetParentType {type} {
#-------------------------------------------
    set typelist [split $type "-"]
    set count [llength $typelist]
    incr count -1
    if {$count > 0} {
        for {set i 1; set result [lindex $typelist 0]} {$i < $count} {incr i} {
            set result "$result-[lindex $typelist $i]"
        }
        return $result
    }
    return ""
}

proc main:GetWindowType {window} {
#-------------------------------------------
    if {[winfo exists $window]} {
        set names [split $window "."]
        set type [lindex $names [expr [llength $names]-1]]
        return [main:GetParentType $type]
    }
    return ""
}

proc main:GetWindowId {window} {
#---------------------------------
    if {[winfo exists $window]} {
        set class [split $window "-"]
        set id [lindex $class [expr [llength $class]-1]]
        return $id
    }
    return 0
}

proc main:DlgDispatch {dispfunc args} {
#----------------------------------------------
    set ret ""
    set prm(0) ""
    set paramcount 0
    foreach param $args {
        set prm($paramcount) $param
        incr paramcount
    }
    if {$paramcount == 0 || $prm(0) == ""} {
        set window [main:GetActiveToplevel]
    } else {
        set window $prm(0)
    }
    set windowtype [main:GetWindowType $window]
    while {$windowtype != ""} {
        set tryfunc "$windowtype:$dispfunc"
        if {[info procs $tryfunc] != ""} {
            switch $paramcount {
                0 {set ret [$tryfunc $window]}
                1 {set ret [$tryfunc $window]}
                2 {set ret [$tryfunc $window $prm(1)]}
                3 {set ret [$tryfunc $window $prm(1) $prm(2)]}
                4 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3)]}
                5 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4)]}
                6 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5)]}
                7 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5) $prm(6)]}
                default {puts  "ERROR:too many params for $tryfunc"}
            }
            break
        }
        set windowtype [main:GetParentType $windowtype]
    }
    return $ret
}

proc main:DocDispatch {dispfunc args} {
#----------------------------------------------
    set okay 0
    set ret ""
    set prm(0) ""
    set paramcount 0
    foreach param $args {
        set prm($paramcount) $param
        incr paramcount
    }
    if {$paramcount == 0 || $prm(0) == ""} {
        set window [mdiclient:GetActiveFrame]
    } else {
        set window $prm(0)
    }
    if {![winfo exists $window]} {
        puts "failed to dispatch \"$dispfunc\""
        return ""
    }
    set windowtype [main:GetWindowType $window]
    while {$windowtype != ""} {
        set tryfunc "$windowtype:$dispfunc"
        if {[info procs $tryfunc] != ""} {
            set okay 1
            switch $paramcount {
                0 {set ret [$tryfunc $window]}
                1 {set ret [$tryfunc $window]}
                2 {set ret [$tryfunc $window $prm(1)]}
                3 {set ret [$tryfunc $window $prm(1) $prm(2)]}
                4 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3)]}
                5 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4)]}
                6 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5)]}
                7 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5) $prm(6)]}
                default {puts  "ERROR:too many params for $tryfunc"}
            }
            break
        }
        set windowtype [main:GetParentType $windowtype]
    }
    if {!$okay} {
        puts "failed to dispatch \"$dispfunc\""
    }
    return $ret
}

proc main:DocTypeDispatch {doctype dispfunc args} {
#----------------------------------------------
    set okay 0
    set ret ""
    set prm(0) ""
    set paramcount 0
    foreach param $args {
        set prm($paramcount) $param
        incr paramcount
    }
    set windowtype $doctype
    if {$paramcount == 0 || $prm(0) == ""} {
        set window [mdiclient:GetActiveFrame]
        if {$window != ""} {
            set activetype [main:GetWindowType $window]
            if {$activetype != $doctype} {
                set window [mdiclient:GetFrameByType $doctype]
            }
        }
    } else {
        set window $prm(0)
    }
    if {![winfo exists $window]} {
        puts "failed to dispatch \"$dispfunc\""
        return ""
    }
    while {$windowtype != ""} {
        set tryfunc "$windowtype:$dispfunc"
        if {[info procs $tryfunc] != ""} {
            set okay 1
            switch $paramcount {
                0 {set ret [$tryfunc $window]}
                1 {set ret [$tryfunc $window]}
                2 {set ret [$tryfunc $window $prm(1)]}
                3 {set ret [$tryfunc $window $prm(1) $prm(2)]}
                4 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3)]}
                5 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4)]}
                6 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5)]}
                7 {set ret [$tryfunc $window $prm(1) $prm(2) $prm(3) $prm(4) $prm(5) $prm(6)]}
                default {puts  "ERROR:too many params for $tryfunc"}
            }
            break
        }
        set windowtype [main:GetParentType $windowtype]
    }
    if {!$okay} {
        puts "failed to dispatch \"$dispfunc\""
    }
    return $ret
}

proc main:StaticDispatch {windowtype dispfunc args} {
#---------------------------------------------------
    set ret ""
    set prm(0) ""
    set paramcount 0
    foreach param $args {
        set prm($paramcount) $param
        incr paramcount
    }
    while {$windowtype != ""} {
        set tryfunc "$windowtype:$dispfunc"
        if {[info procs $tryfunc] != ""} {
            switch $paramcount {
                0 {set ret [$tryfunc]}
                1 {set ret [$tryfunc $prm(0)]}
                2 {set ret [$tryfunc $prm(0) $prm(1)]}
                3 {set ret [$tryfunc $prm(0) $prm(1) $prm(2)]}
                4 {set ret [$tryfunc $prm(0) $prm(1) $prm(2) $prm(3)]}
                5 {set ret [$tryfunc $prm(0) $prm(1) $prm(2) $prm(3) $prm(4)]}
                6 {set ret [$tryfunc $prm(0) $prm(1) $prm(2) $prm(3) $prm(4) $prm(5)]}
                7 {set ret [$tryfunc $prm(0) $prm(1) $prm(2) $prm(3) $prm(4) $prm(5) $prm(6)]}
                default {puts  "ERROR:too many params for $tryfunc"}
            }
            break
        }
        set windowtype [main:GetParentType $windowtype]
    }
    return $ret
}

proc main:StartExternalProcess {outwindow eol update program args} {
#--------------------------------------------------------------
    global main
    if {$main(pipe,cmnd) != ""} {
        puts "output channel is busy"
        return ""
    }
    set arglist $args
    set main(pipe,result) 1
    set main(pipe,cmnd) [open "|$program $arglist 2>@stdout" "r"]
    if {$main(pipe,cmnd) != ""} {
        fileevent $main(pipe,cmnd) readable "main:ReadProcessOutput $outwindow $eol $update"
    }
    return $main(pipe,cmnd)
}

proc main:WaitExternalProcess {} {
#------------------------------------
    global main
    if {$main(pipe,cmnd) == ""} {
        return 1
    }
    tkwait variable main(pipe,cmnd)
    return $main(pipe,result)
}

proc main:AbortExternalProcess {} {
#------------------------------------
    global main
    if {$main(pipe,cmnd) == ""} {
        return false
    }
    catch "close $main(pipe,cmnd)"
    set main(pipe,cmnd) ""
    return true
}

proc main:ReadProcessOutput {window insert_eol update} {
#----------------------------------------------------------------------
    global main
    if {[eof $main(pipe,cmnd)]} {
        if {[catch "close $main(pipe,cmnd)"]} {
            set main(pipe,result) 0
        }
        set main(pipe,cmnd) ""
    } else {
        gets $main(pipe,cmnd) line
        if {$line != ""} {
            $window insert end $line
        }
        if {$insert_eol} {
            $window insert end \n
        }
        if {$update} {
            $window see end
#            update
        }
    }
}

proc main:SaveProfile {} {
#----------------------------
    global main
    set file [open $main(app,profile) "w"]
    if {$file != ""} {
        set geom [wm geometry $main(app,frame)]
        puts $file "#    $main(app,title)"
        puts $file "#    software development environment"
        puts $file "#    auto-generated Tcl/Tk profile"
        puts $file "#"
        puts $file "wm geometry $main(app,frame) $geom"
        puts $file "set main(app,toolbar) $main(app,toolbar)"
        puts $file "set main(app,statusbar) $main(app,statusbar)"

        foreach name [array names main position,*] {
            puts $file "set main($name) \"$main($name)\""
        }
        foreach name [array names main font,*] {
            puts $file "set main($name) \"$main($name)\""
        }
        foreach name [array names main color,*] {
            puts $file "set main($name) \"$main($name)\""
        }
        if {[llength $main(recent,list)] > 0} {
            foreach name $main(recent,list) {
                puts $file "lappend main(recent,list) \"$name\""
            }
        }
        close $file
    }
}

proc main:LoadProfile {} {
#----------------------------
    global main
    if {[file exists $main(app,profile)]} {
        source $main(app,profile)
    }
}

proc main:AddRecentFilesToMenu {} {
#------------------------------------
    global main
    if {[llength $main(recent,list)] > 0} {
        set i 0
        foreach name $main(recent,list) {
            if {$i >= $main(recent,maxcount)} {
                break
            }
            $main(menu,frame).file.m add command -command "main:OnOpenRecentFile $name" -label $name
            incr i
        }
    }
}

proc main:UpdateRecentFiles {filename} {
#----------------------------------------
    global main
    if {$filename != ""} {
        set index [lsearch -exact $main(recent,list) $filename]
        if {$index >= 0} {
            $main(menu,frame).file.m delete [lindex $main(recent,list) $index]
            set main(recent,list) [lreplace $main(recent,list) $index $index]
        }
        lappend main(recent,list) $filename
        $main(menu,frame).file.m add command -command "main:OnOpenRecentFile $filename" -label $filename
        if {[llength $main(recent,list)] > $main(recent,maxcount)} {
            $main(menu,frame).file.m delete [lindex $main(recent,list) 0]
            set main(recent,list) [lreplace $main(recent,list) 0 0]
        }
    }
}

proc main:GetMainData {section entry} {
#---------------------------------------
    global main
    if {[info exists main($section,$entry)]} {
        return $main($section,$entry)
    }
    return ""
}

proc main:SetMainData {section entry data} {
#-------------------------------------------
    global main
    set main($section,$entry) $data
}

proc main:ChangeAllAppearance {} {
#-------------------------------
    main:ConfigureMenuAppearance
    main:ConfigureStatusbarAppearance
    mdiclient:ConfigureFrameAppearance
}

proc main:ChangeAllPositions {} {
#-------------------------------
    global main
    set t $main(app,toolbar)
    set s $main(app,statusbar)
    if {$t} { main:OnWindowToolbar}
    if {$s} { main:OnWindowStatusbar}
    main:ConfigureMenuPosition 1
    if {$t} { main:OnWindowToolbar}
    if {$s} { main:OnWindowStatusbar}
}

proc main:RegisterDocType {doctype} {
#--------------------------------------
    global main
    lappend main(doc,types) $doctype
}

proc main:DoOpenFile {filename {doctype ""}} {
#----------------------------------------------
    global main;
    if {$filename != ""} {
        set fr_name [mdiclient:GetFrameByPath $filename]
        if {$fr_name != ""} {
            mdiclient:ActivateFrame $fr_name
            return $fr_name
        }
    }
    set fr_name ""
    if {[file exists $filename] || $filename == ""} {
        if {$doctype == ""} {
            set doctype [main:SelectDocType $filename]
            if {$doctype == ""} {
                return ""
            }
        }
        set fr_name [main:CreateDocFrame $doctype]
        if {$fr_name != ""} {
            tkwait visibility $fr_name
            mdiclient:ActivateFrame $fr_name
            if {$filename == ""} {
                main:DocDispatch OnNewDocument $fr_name
            } else {
                main:DocDispatch OnOpenDocument $fr_name $filename
            }
            main:DocInWindowMenu $fr_name remove
            main:DocInWindowMenu $fr_name add $filename
#            tkwait visibility $fr_name
#            mdiclient:ActivateFrame $fr_name
        }
        main:UpdateRecentFiles $filename
    } else {
        dialog:MessageBox $main(app,frame) OpenFile "$filename\n\nFile not found" "Ok"
    }
    return $fr_name
}

proc main:SelectDocType {filename} {
#--------------------------------------
    global main
    set bestmatch 0
    set besttype ""
    if {[llength $main(doc,types)] > 0} {
        foreach type $main(doc,types) {
            set match [main:StaticDispatch $type MatchDocType $filename]
            if {$match != "" && $match > $bestmatch} {
                set bestmatch $match
                set besttype $type
            }
        }
        return $besttype
    }
    return ""
}

proc main:CreateDocFrame {doctype} {
#-------------------------------------------
    global main;
    if {$doctype != ""} {
        incr main(doc,count);
        set fr_name [mdiclient:CreateNewFrame "$doctype-$main(doc,count)"]
        return $fr_name
    }
    return ""
}

proc main:DocInWindowMenu {fr_name add {filename ""}} {
#-------------------------------------------------
    global main;
    if {$add == "add"} {
        main:DocDispatch RenameDocument $fr_name $filename
        mdiclient:AdjustFrameName $fr_name
        set fr_caption [main:DocDispatch GetDocData $fr_name title]
        $main(menu,frame).window.m add command -command "mdiclient:OnWindowActivate $fr_caption" -label $fr_caption
    } else {
        set title [main:DocDispatch GetDocData $fr_name title]
        if {$title != ""} {
            $main(menu,frame).window.m delete $title
        }
    }
}

proc main:SetStatusBarText {text {pane 0}} {
#--------------------------------------------
    global main
    set bar $main(app,frame).status.l$pane
    if {[winfo exists ${bar}]} {
        set prev [$bar cget -text]
        if {[string compare $prev $text]} {
            $bar config -text $text
        }
    }
}

proc main:ConfigureStatusbarAppearance {} {
#-------------------------------------------
    global main
    foreach pane [winfo children $main(app,frame).status] {
        $pane config -font $main(font,status)
    }
}

proc main:TooltipShow {} {
#--------------------------------
    global main
    if {$main(tool,btn) != ""} {
        set item [lindex [split $main(tool,btn) .] 3]
        if {$item != ""} {
            if {[info exists main(tool,$item)]} {
                set x [expr [winfo rootx $main(tool,btn)] + ([winfo width $main(tool,btn)]/2)]
                set y [expr [winfo rooty $main(tool,btn)] + [winfo height $main(tool,btn)] + 4]
                toplevel $main(app,frame).tooltip -bg black
                wm overrideredirect $main(app,frame).tooltip 1
                label $main(app,frame).tooltip.l -text $main(tool,$item) -relief flat \
                    -bg $main(color,tooltipbg) -fg $main(color,tooltipfg) \
                    -padx 2 -pady 0 -anchor w -font $main(font,tool)
                pack $main(app,frame).tooltip.l -side left -padx 1 -pady 1
                wm geometry $main(app,frame).tooltip +${x}+${y}
            }
        }
        set main(tool,btn) ""
    }
}

proc main:TooltipDestroy {} {
#------------------------------------
    global main
    if {[winfo exists $main(app,frame).tooltip]} {
        destroy $main(app,frame).tooltip
        return 1
    }
    return 0
}

proc main:ShowWindow {name cmnd} {
#-----------------------------------
    if {![winfo exists $name]} {return}
    switch $cmd {
        "show"    { wm deiconify $name; }
        "hide"    { wm withdraw $name; }
        "iconify" { wm iconify $name; }
        "destroy" { destroy $name; }
    }
}

proc main:CreateAppMenu {this} {
#-----------------------------------
    global main
    set main(menu,frame) $this
    frame $this -borderwidth 1 -height 30 -relief sunken -width 30 

    menubutton $this.file -menu $this.file.m -padx 4 -pady 3 -text File -underline 0 
    menu $this.file.m -tearoff 0 
    $this.file.m add command  -command main:OnFileNew -label New -underline 0  -accelerator "Ctrl+N"
    $this.file.m add command  -command main:OnFileOpen -label Open... -underline 0 -accelerator "Ctrl+O"
    $this.file.m add command  -command main:OnFileClose -label Close -underline 0  -accelerator "Ctrl+F4"
    $this.file.m add command  -command main:OnFileCloseAll -label {Close All}
    $this.file.m add separator
    $this.file.m add command  -command main:OnFileSave -label Save -underline 0  -accelerator "Ctrl+S"
    $this.file.m add command  -command main:OnFileSaveAs -label {Save As...} -underline 5
    $this.file.m add command  -command {mdiclient:SaveAllModified 0} -label {Save All}
    $this.file.m add separator
    $this.file.m add command  -command main:OnFileTool -label {Run Tool...} -underline 0
    $this.file.m add separator
    $this.file.m add command  -command main:OnFileExit -label Exit -underline 1 
    $this.file.m add separator

    menubutton $this.edit -menu $this.edit.m -padx 4 -pady 3 -text Edit -underline 0 
    menu $this.edit.m -tearoff 0 
    $this.edit.m add command  -command {main:DocDispatch OnEditUndo} -label Undo
    $this.edit.m add command  -command {main:DocDispatch OnEditRedo} -label Redo
    $this.edit.m add command  -command {main:DocDispatch OnEditCut} -label Cut -underline 2 -accelerator "Ctrl+X"
    $this.edit.m add command  -command {main:DocDispatch OnEditCopy} -label Copy -underline 0 -accelerator "Ctrl+C"
    $this.edit.m add command  -command {main:DocDispatch OnEditPaste} -label Paste -underline 0 -accelerator "Ctrl+V"
    $this.edit.m add command  -command {main:DocDispatch OnEditDelete} -label Delete -underline 0 -accelerator "Del"
    $this.edit.m add separator
    $this.edit.m add command  -command main:OnEditFind -label Find... -underline 0 -accelerator "Ctrl+F"
    $this.edit.m add command  -command {main:DocDispatch OnEditFindSelected} -label {Find Sel} -accelerator "Ctrl+F3"
    $this.edit.m add command  -command {main:DocDispatch OnEditFindAgain} -label {Find Again} -accelerator "F3"
    $this.edit.m add command  -command {main:DocDispatch OnEditGoto} -label {Go to...} -underline 0 -accelerator "Ctrl+G"
    $this.edit.m add separator
    $this.edit.m add command  -command {main:DocDispatch OnEditSelectAll} -label {Select All} -underline 7

    menubutton $this.project -menu $this.project.m -padx 4 -pady 3 -text Project -underline 0 
    menu $this.project.m -tearoff 0 
    $this.project.m add command  -command {main:DoOpenFile "" "doc-project"; main:DocTypeDispatch doc-project OnProjectEdit} -label New -underline 0 
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectEdit} -label Edit... -underline 0
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectDependencies} -label {Dependencies}
    $this.project.m add separator
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild "" deb select} -label {Build} -accelerator "Ctrl+F7"
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild} -label {Build All} -accelerator "F7"
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild "" deb rebuildselect} -label {Rebuild}
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild "" deb clean all} -label {Rebuild All}
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild "" rel select} -label {Release}
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectBuild "" rel clean all} -label {Release All}
    $this.project.m add separator
    $this.project.m add command  -command {main:DocTypeDispatch doc-project OnProjectDebug} -label {Debug} -underline 0  -accelerator "F5"

    menubutton $this.window  -menu $this.window.m -padx 4 -pady 3 -text Window -underline 0 
    menu $this.window.m -tearoff 0 
    $this.window.m add command  -command mdiclient:OnWindowCascade -label Cascade -underline 0 
    $this.window.m add command  -command mdiclient:OnWindowTileHorizontal -label {Tile Horizontal}  -underline 5 
    $this.window.m add command  -command mdiclient:OnWindowTileVertical -label {Tile Vertical} -underline 5 
    $this.window.m add separator
    $this.window.m add command  -command mdiclient:OnWindowNext -label {Next} -underline 0 -accelerator "F6"
    $this.window.m add command  -command mdiclient:OnWindowPrev -label {Previous} -underline 0 -accelerator "Shift+F6"
    $this.window.m add separator
    $this.window.m add command  -command main:OnWindowToolbar -label {Tool Bar} -underline 0 
    $this.window.m add command  -command main:OnWindowStatusbar -label {Status Bar} -underline 0 
    $this.window.m add command  -command main:OnWindowPreferences -label {Preferences...}
    $this.window.m add separator

    menubutton $this.help -menu $this.help.m -padx 4 -pady 3 -text Help -underline 0 
    menu $this.help.m -tearoff 0 
    $this.help.m add command  -command main:OnHelpAbout -label About... -state active -underline 0 

    foreach menubtn [winfo children $this] {
        main:BindMenubutton $menubtn
    }

    main:ConfigureMenuPosition 0
    main:ConfigureMenuAppearance
}

proc main:BindMenubutton {menubtn} {
#--------------------------------------------
    if {[winfo class $menubtn] == "Menubutton"} {
        foreach item [winfo children $menubtn] {
            if {[winfo class $item] == "Menu"} {
                bind $item <KeyRelease> {main:OnMenuSelect %W}
                bind $item <FocusIn>    {main:OnMenuActivate %W}
                bind $item <FocusOut>   {main:OnMenuDeactivate %W}
                bind $item <Motion>     {main:OnMenuSelect %W}
            }
        }
    }
}

proc main:ConfigureMenuPosition {reconfig} {
#-------------------------------------------
    global main
    set this $main(menu,frame)

    if {$reconfig} {
        pack forget $this.file
        pack forget $this.edit
        pack forget $this.project
        pack forget $this.window
        pack forget $this.help
        pack forget $this
    }

    switch $main(position,menubar) {
    "1"         {if {$main(position,topbottom)} {set side_btn "top"} {set side_btn "bottom"}; set fill_frame "y"; set side_frame "left";}
    "2"         {if {$main(position,leftright)} {set side_btn "left"} {set side_btn "right"}; set fill_frame "x"; set side_frame "bottom";}
    "3"         {if {$main(position,topbottom)} {set side_btn "top"} {set side_btn "bottom"}; set fill_frame "y"; set side_frame "right";}
    "default"   {if {$main(position,leftright)} {set side_btn "left"} {set side_btn "right"}; set fill_frame "x"; set side_frame "top";}
    }
    pack $this         -anchor center -expand 0 -fill $fill_frame -side $side_frame -before $main(app,frame).client
    foreach item { file edit project window help } {
        pack $this.$item    -anchor center -expand 0 -fill none -side $side_btn
    }
}

proc main:ConfigureMenuAppearance {} {
#-------------------------------------------
    global main
    foreach menubtn [winfo children $main(menu,frame)] {
        $menubtn configure -font $main(font,menu)
        foreach menuitem [winfo children $menubtn] {
            $menuitem configure -font $main(font,menu)
        }
    }
}

proc main:CreateAppFrame {this} {
#-----------------------------------
    global main
    set main(app,frame) $this

    global titlepane
    if {[winfo exists $titlepane]} {
        tkwait window $titlepane
    }

    toplevel $this -class Toplevel
    wm focusmodel $this passive
    wm geometry $this 600x400+200+200
    wm maxsize $this 1265 994
    wm minsize $this 1 1
    wm overrideredirect $this 0
    wm resizable $this 1 1
    wm deiconify $this
    wm title $this $main(app,title)
    wm iconbitmap $this @$main(app,icon)
    main:LoadProfile

    mdiclient:CreateWindow $this.client $main(app,home)
    main:CreateAppMenu $this.menu
    main:AddRecentFilesToMenu

    if {$main(app,toolbar)} {
        main:OnWindowToolbar
    }
    if {$main(app,statusbar)} {
        main:OnWindowStatusbar
    }
}

bind all <Control-n>   {main:OnFileNew}
bind all <Control-N>   {main:OnFileNew}
bind all <Control-o>   {main:OnFileOpen}
bind all <Control-O>   {main:OnFileOpen}
bind all <Control-s>   {main:OnFileSave}
bind all <Control-S>   {main:OnFileSave}
bind all <Control-F4>  {main:OnFileClose}
bind all <Control-F7>  {main:DocTypeDispatch doc-project OnProjectBuild "" deb select}
bind all <F7>          {main:DocTypeDispatch doc-project OnProjectBuild}
bind all <F5>          {main:DocTypeDispatch doc-project OnProjectDebug}
bind all <F6>          {mdiclient:OnWindowNext}
bind all <Shift-F6>    {mdiclient:OnWindowPrev}
