/*
 * Copyright (c) 2004-2011 Marco Maccaferri and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     Marco Maccaferri - initial API and implementation
 */

package org.eclipsetrader.archipelago;

import java.util.ArrayList;
import java.util.List;

import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.ListenerList;
import org.eclipse.core.runtime.Status;
import org.eclipsetrader.core.feed.IBook;
import org.eclipsetrader.core.feed.IFeedIdentifier;
import org.eclipsetrader.core.feed.IFeedSubscription2;
import org.eclipsetrader.core.feed.ILastClose;
import org.eclipsetrader.core.feed.IQuote;
import org.eclipsetrader.core.feed.ISubscriptionListener;
import org.eclipsetrader.core.feed.ITodayOHL;
import org.eclipsetrader.core.feed.ITrade;
import org.eclipsetrader.core.feed.QuoteDelta;
import org.eclipsetrader.core.feed.QuoteEvent;

public class FeedSubscription implements IFeedSubscription2 {

    private Level2Feed connector;
    private String symbol;
    private IFeedIdentifier identifier;

    private ITrade trade;
    private IQuote quote;
    private ITodayOHL todayOHL;
    private ILastClose lastClose;
    private IBook book;

    private ListenerList listeners = new ListenerList(ListenerList.IDENTITY);

    private List<QuoteDelta> deltaList = new ArrayList<QuoteDelta>();
    private int instanceCount = 0;
    private int level2InstanceCount = 0;

    public FeedSubscription(Level2Feed connector, IFeedIdentifier identifier) {
        this.connector = connector;
        this.identifier = identifier;
        this.symbol = identifier.getSymbol();
    }

    public FeedSubscription(Level2Feed connector, String symbol) {
        this.connector = connector;
        this.symbol = symbol;
    }

    public void setIdentifier(IFeedIdentifier identifier) {
        this.identifier = identifier;
        this.symbol = identifier.getSymbol();
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#dispose()
     */
    @Override
    public void dispose() {
        connector.disposeSubscription(this);
    }

    protected int incrementInstanceCount() {
        instanceCount++;
        return instanceCount;
    }

    protected int decrementInstanceCount() {
        instanceCount--;
        return instanceCount;
    }

    protected int getInstanceCount() {
        return instanceCount;
    }

    protected int incrementLevel2InstanceCount() {
        level2InstanceCount++;
        return level2InstanceCount;
    }

    protected int decrementLevel2InstanceCount() {
        level2InstanceCount--;
        return level2InstanceCount;
    }

    public int getLevel2InstanceCount() {
        return level2InstanceCount;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#addSubscriptionListener(org.eclipsetrader.core.feed.ISubscriptionListener)
     */
    @Override
    public void addSubscriptionListener(ISubscriptionListener listener) {
        listeners.add(listener);
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#removeSubscriptionListener(org.eclipsetrader.core.feed.ISubscriptionListener)
     */
    @Override
    public void removeSubscriptionListener(ISubscriptionListener listener) {
        listeners.remove(listener);
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getIdentifier()
     */
    @Override
    public IFeedIdentifier getIdentifier() {
        return identifier;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getSymbol()
     */
    @Override
    public String getSymbol() {
        return symbol;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getQuote()
     */
    @Override
    public IQuote getQuote() {
        return quote;
    }

    public void setQuote(IQuote quote) {
        this.quote = quote;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getTodayOHL()
     */
    @Override
    public ITodayOHL getTodayOHL() {
        return todayOHL;
    }

    public void setTodayOHL(ITodayOHL todayOHL) {
        this.todayOHL = todayOHL;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getTrade()
     */
    @Override
    public ITrade getTrade() {
        return trade;
    }

    public void setTrade(ITrade trade) {
        this.trade = trade;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription#getLastClose()
     */
    @Override
    public ILastClose getLastClose() {
        return lastClose;
    }

    public void setLastClose(ILastClose lastClose) {
        this.lastClose = lastClose;
    }

    /* (non-Javadoc)
     * @see org.eclipsetrader.core.feed.IFeedSubscription2#getBook()
     */
    @Override
    public IBook getBook() {
        return book;
    }

    public void setBook(IBook book) {
        this.book = book;
    }

    public void addDelta(QuoteDelta delta) {
        synchronized (deltaList) {
            deltaList.add(delta);
        }
    }

    public boolean hasListeners() {
        return listeners.size() != 0;
    }

    public void fireNotification() {
        QuoteDelta[] deltas;
        synchronized (deltaList) {
            if (deltaList.isEmpty()) {
                return;
            }
            deltas = deltaList.toArray(new QuoteDelta[deltaList.size()]);
            deltaList.clear();
        }
        QuoteEvent event = new QuoteEvent(connector, getIdentifier(), deltas);
        Object[] l = listeners.getListeners();
        for (int i = 0; i < l.length; i++) {
            try {
                ((ISubscriptionListener) l[i]).quoteUpdate(event);
            } catch (Exception e) {
                Status status = new Status(IStatus.ERROR, ArchipelagoPlugin.PLUGIN_ID, 0, "Error notifying a quote update", e);
                ArchipelagoPlugin.log(status);
            } catch (LinkageError e) {
                Status status = new Status(IStatus.ERROR, ArchipelagoPlugin.PLUGIN_ID, 0, "Error notifying a quote update", e);
                ArchipelagoPlugin.log(status);
            }
        }
    }
}
