#include <QScrollBar>
#include <QSlider>
#include <QLabel>
#include <QLineEdit>
#include <QItemDelegate>
#include <QPainter>
#include <QPaintEvent>
#include <QShowEvent>

#include "TrackView.h"

#include "ViewSettings.h"
#include "EventItem.h"
#include "TreeItem.h"
#include "TrackItem.h"
#include "ControlItem.h"
#include "SequenceModel.h"
#include "TrackDelegate.h"
#include "../configure.h"

#include "command/LyricChangeCommand.h"
#include "helper/PitchPainter.h"

using namespace stand::view;
using namespace stand::model;

TrackView::TrackView(QWidget *parent) :
    AbstractPianoroll(parent)
{
    _model = NULL;
    _autoScroll = false;
    setEditTriggers(QAbstractItemView::DoubleClicked);
    stand::delegate::TrackDelegate *delegate = new stand::delegate::TrackDelegate(this);
    setItemDelegate(delegate);
}

TrackView::~TrackView()
{
}

/*==============================================*/
/*                                              */
/*  Reimplementation of QAbstractItem is below  */
/*                                              */
/*==============================================*/
QModelIndex TrackView::indexAt(const QPoint &point) const
{
    if(!model())
    {
        return QModelIndex();
    }
    QModelIndex i = model()->index(4, 0);
    i = model()->index(0, 0, i);
    if(i.isValid())
    {
        int tick = _settings->tickAt(point.x() - _widget->x());
        int note = _settings->noteAt(point.y() - _widget->y());
        TreeItem *s = static_cast<TreeItem *>(i.internalPointer());
        for(int j = 0; j < s->childCount(); j++)
        {
            EventItem *e = (EventItem *)(s->child(j));
            int eTick = e->data(EventItem::TickData, 0).toInt();
            int eLength = e->data(EventItem::LengthData, 0).toInt();
            int eNote = e->data(EventItem::NoteData, 0).toInt();
            if(eNote == note && eTick <= tick && tick <= eTick + eLength)
            {
                QModelIndex current = model()->index(j, 0, i);
                return current;
            }
        }
    }
    return QModelIndex();
}

void TrackView::setModel(QAbstractItemModel *model)
{
    if(_model)
    {
        disconnect(_model, SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(dataChanged(QModelIndex,QModelIndex)));
    }
    _model = dynamic_cast<SequenceModel *>(model);
    if(!model)
    {
        return;
    }
    connect(_model, SIGNAL(dataChanged(QModelIndex,QModelIndex)), this, SLOT(dataChanged(QModelIndex,QModelIndex)));
    QAbstractItemView::setModel(model);
    //setState(QAbstractItemView::NoState);
    setSelectionModel(new QItemSelectionModel(_model));

    if(_settings)
    {
        _settings->setModel(_model);
    }
    QModelIndex i = model->index(4, 0);
    i = model->index(0, 0, i);
    i = model->index(0, 0, i);
    scrollTo(i, QAbstractItemView::PositionAtTop);
    _widget->update();
}

int TrackView::length()
{
    if(!_model)
    {
        return _settings->beatWidth() * 4 * 12;
    }
    return _model->minimumViewWidth(_settings->trackId())  * _settings->beatWidth() / stand::sequence::DefaultTicksPerBeat + _settings->beatWidth() * 16;
}

void TrackView::scrollTo(const QModelIndex &index, ScrollHint hint)
{
    QRect r(visualRect(index));
    switch(hint)
    {
    case QAbstractItemView::PositionAtTop:
        horizontalScrollBar()->setValue(r.x() - _widget->x() - _settings->beatWidth());
        verticalScrollBar()->setValue(r.y() - _widget->y() - height() / 2);
        break;
    case QAbstractItemView::PositionAtBottom:
        horizontalScrollBar()->setValue(r.x() - _widget->x() + width() - _settings->beatWidth());
        verticalScrollBar()->setValue(r.y() - _widget->y() - height() / 2);
    case QAbstractItemView::EnsureVisible:
        if(r.left() < _settings->beatWidth() || r.right() > width() - _settings->beatWidth())
        {
            horizontalScrollBar()->setValue(r.x() - _widget->x() - _settings->beatWidth());
        }

        if(r.bottom() < 0 || r.top() > height())
        {
            verticalScrollBar()->setValue(r.y() - _widget->y() - height() / 2);
        }
        break;
    }

}

QRect TrackView::visualRect(const QModelIndex &index) const
{
    if(isIndexHidden(index))
    {
        return QRect();
    }
    TreeItem *item = static_cast<TreeItem *>(index.internalPointer());
    if(!item)
    {
        return QRect();
    }
    int tick = item->data(EventItem::TickData, 0).toInt();
    int length = item->data(EventItem::LengthData, 0).toInt();
    int note = item->data(EventItem::NoteData, 0).toInt();

    int x = _settings->xAt(tick);
    int w = _settings->xAt(tick + length) - x;
    int y = _settings->yAt(note);
    x += _widget->x();
    y += _widget->y();

    return QRect(x, y, w, _settings->noteHeight());
}

int TrackView::horizontalOffset() const
{
    return 0;
}

bool TrackView::isIndexHidden(const QModelIndex &index) const
{
    return false;
    if(!index.isValid() || !index.parent().isValid())
    {
        return true;
    }
    QModelIndex i(index);
    while(i.parent().isValid())
    {
        i = i.parent();
    }
    if(i.row() != 4 + _settings->trackId())
    {
        return true;
    }
    return false;
}

QModelIndex TrackView::moveCursor(CursorAction cursorAction, Qt::KeyboardModifiers modifiers)
{
    if(modifiers != Qt::NoModifier)
    {
        return QModelIndex();
    }

    QModelIndex current = selectionModel()->currentIndex();
    QModelIndex parent;
    if(current.isValid() && (parent = current.parent()).isValid())
    {
        switch(cursorAction)
        {
        case QAbstractItemView::MoveLeft:
            current = model()->index(std::max(0, current.row()) - 1, 0, parent);
            break;
        case QAbstractItemView::MoveRight:
            current = model()->index(current.row() + 1, 0, parent);
            break;
        }
    }

    return current;
}

void TrackView::setSelection(const QRect &rect, QItemSelectionModel::SelectionFlags command)
{
    int beginTick = _settings->tickAt(rect.left() - _widget->x());
    int endTick = _settings->tickAt(rect.right() - _widget->x());
    int beginNote = _settings->noteAt(rect.top() - _widget->y());
    int endNote = _settings->noteAt(rect.bottom() - _widget->y());
    int tmp = std::max(beginNote, endNote);
    beginNote = std::min(beginNote, endNote);
    endNote = tmp;
    tmp = std::max(beginTick, endTick);
    beginTick = std::min(beginTick, endTick);
    endTick = tmp;

    QModelIndex index = model()->index(4, 0);
    index = model()->index(0, 0, index);
    EventItem *events = (EventItem *)(_model->track(_settings->trackId())->findChildItem("Events"));

    for(int i = 0; i < events->childCount(); i++)
    {
        EventItem *item = (EventItem *)(events->child(i));

        int tick = item->data(EventItem::TickData, 0).toInt();
        int length = item->data(EventItem::LengthData, 0).toInt();
        int note = item->data(EventItem::NoteData, 0).toInt();

        if(beginNote <= note && note <= endNote && ((tick <= beginTick && beginTick <= tick + length)
                 || (tick <= endTick && endTick <= tick + length || ( beginTick <= tick && tick + length <= endTick))))
        {
            QModelIndex current = model()->index(i, 0, index);
            if(!selectionModel()->isSelected(current))
            {
                selectionModel()->select(current, command & QItemSelectionModel::Select);
            }
        }
        else
        {
            QModelIndex current = model()->index(i, 0, index);
            if(selectionModel()->isSelected(current))
            {
                selectionModel()->select(current, QItemSelectionModel::Deselect);
            }
        }
    }
}

void TrackView::selectionChanged(const QItemSelection &selected, const QItemSelection &deselected)
{
    QRegion rgn(visualRegionForSelection(selected));
    rgn = rgn + visualRegionForSelection(deselected);
    _widget->update(rgn);
}

int TrackView::verticalOffset() const
{
    return 0;
}

QRegion TrackView::visualRegionForSelection(const QItemSelection &selection) const
{
    QRegion rgn;
    for(int i = 0; i < selection.size(); i++)
    {
        QRect r(visualRect(selection.indexes().at(i)));
        r = QRect(r.x() - _widget->x(), r.y() - _widget->y(), r.width(), r.height());
        rgn = rgn + r;
    }
    return rgn;
}

bool TrackView::eventFilter(QObject *o, QEvent *e)
{
    bool retval = AbstractPianoroll::eventFilter(o, e);

    if(o == _widget && e->type() == QEvent::Paint)
    {
        paintEvent((QPaintEvent *)e);
    }

    return retval;
}

int TrackView::_beginId(EventItem *events, int beginTick, int beginNote, int endNote)
{
    for(int i = 0; i < events->childCount(); i++)
    {
        EventItem *item = (EventItem *)(events->child(i));
        if(!item)
        {
            continue;
        }
        int tick = item->data(EventItem::TickData, 0).toInt();
        int length = item->data(EventItem::LengthData, 0).toInt();
        int note = item->data(EventItem::NoteData, 0).toInt();
        if(tick + length < beginTick || note < beginNote || note > endNote)
        {
            continue;
        }
        return i;
    }
    return events->childCount();
}

void TrackView::mousePressEvent(QMouseEvent *e)
{
    _isDragged = false;
    QModelIndex index = indexAt(e->pos());
    if(!index.isValid() && ((e->button() & Qt::LeftButton) || e->button() == Qt::NoButton))
    {
        _leftClickedPosition = _leftDraggedPosition = QPoint(e->pos().x() - _widget->x(), e->pos().y() - _widget->y());
        _clickKind = ClickBackground;
    }
    else
    {
        _clickKind = ClickNothing;
    }
    AbstractPianoroll::mousePressEvent(e);
}

void TrackView::mouseMoveEvent(QMouseEvent *e)
{
    _isDragged = true;
    if(e->x() < _settings->beatWidth() / 2)
    {
        horizontalScrollBar()->setValue(horizontalScrollBar()->value() - 16);
    }
    if(e->x() > width() - _settings->beatWidth() / 2)
    {
        horizontalScrollBar()->setValue(horizontalScrollBar()->value() + 16);
    }

    // 左クリックでドラッグ
    switch(_clickKind)
    {
    case ClickBackground:
        _mouseDragOnBackground(e);
        break;
    }

    AbstractPianoroll::mouseMoveEvent(e);
}

void TrackView::_mouseDragOnBackground(QMouseEvent *e)
{
    _leftDraggedPosition = QPoint(e->pos().x() - _widget->x(), e->pos().y() - _widget->y());
    QPoint p1(std::min(_leftClickedPosition.x(), _leftDraggedPosition.x()) + _widget->x(), std::min(_leftClickedPosition.y(), _leftDraggedPosition.y()) + _widget->y());
    QPoint p2(std::max(_leftClickedPosition.x(), _leftDraggedPosition.x()) + _widget->x(), std::max(_leftClickedPosition.y(), _leftDraggedPosition.y()) + _widget->y());
    setSelection(QRect(p1, p2), QItemSelectionModel::Select | QItemSelectionModel::Clear);
    QRegion rgn(_previousRect.x() - 1, _previousRect.y() - 1 , _previousRect.width() + 2, _previousRect.height() + 2);
    QRect r(p1, p2);
    rgn += QRect(r.x() - 1, r.y() - 1, r.width() + 2, r.height() + 2);
    _widget->update(rgn);
}

void TrackView::mouseReleaseEvent(QMouseEvent *e)
{
    switch(_clickKind)
    {
    case ClickBackground:
        _clickKind = ClickNothing;
        _widget->update(_previousRect.x() - 1, _previousRect.y() - 1 , _previousRect.width() + 2, _previousRect.height() + 2);
        AbstractPianoroll::mouseReleaseEvent(NULL);
        break;
    default:
        _clickKind = ClickNothing;
        AbstractPianoroll::mouseReleaseEvent(e);
    }
}

void TrackView::paintEvent(QPaintEvent *e)
{
    if(!_model || !_model->track(_settings->trackId()) || _model->track(_settings->trackId())->childCount() < 1)
    {
        return;
    }

    //
    QPainter p(_widget);
    _drawNotes(&p, e);

    switch(_clickKind)
    {
    case ClickBackground:
        _drawSelectionBox(&p, e);
        break;
    }

    QList<TreeItem *> controls = _settings->tracks()[_settings->trackId()]->controls();
    QList<ControlItem *> pitches;
    for(int i = 0; i < controls.size(); i++)
    {
        if(controls[i]->data(ControlItem::ViewType, Qt::EditRole).toInt() == ControlItem::TypePitch)
        {
            pitches << static_cast<ControlItem *>(controls[i]->child(0));
        }
    }
    helper::PitchPainter::paint(pitches, _settings, &p, e);
}

void TrackView::_drawNotes(QPainter *p, QPaintEvent *e)
{
    const int beginTick = _settings->tickAt(e->rect().left());
    const int endTick = _settings->tickAt(e->rect().right());
    const int beginNote = _settings->noteAt(e->rect().bottom());
    const int endNote = _settings->noteAt(e->rect().top());

    EventItem *events = (EventItem *)(_model->track(_settings->trackId())->findChildItem("Events"));
    QStyleOptionViewItem opt;
    opt.initFrom(this);
    opt.textElideMode = Qt::ElideRight;
    opt.palette = QPalette(palette());

    for(int i = _beginId(events, beginTick, beginNote, endNote); i < events->childCount(); i++)
    {
        if(events->child(i)->data(EventItem::TickData, 0).toInt() > endTick)
        {
            break;
        }
        QModelIndex index = model()->index(4, 0);
        index = model()->index(0, 0, index);
        index = model()->index(i, 0, index);
        opt.rect = visualRect(index);
        opt.rect = QRect(opt.rect.x() - _widget->x(), opt.rect.y() - _widget->y(), opt.rect.width(), opt.rect.height());

        if(selectionModel()->isSelected(index))
        {
            opt.palette.setColor(QPalette::Background, QColor(128, 128, 192));
        }
        else
        {
            opt.palette.setColor(QPalette::Background, QColor(192, 192, 255));
        }
        itemDelegate()->paint(p, opt, index);
    }
}

void TrackView::_drawSelectionBox(QPainter *p, QPaintEvent *e)
{
    QPoint p1(std::min(_leftClickedPosition.x(), _leftDraggedPosition.x()), std::min(_leftClickedPosition.y(), _leftDraggedPosition.y()));
    QPoint p2(std::max(_leftClickedPosition.x(), _leftDraggedPosition.x()) - 1, std::max(_leftClickedPosition.y(), _leftDraggedPosition.y()));
    p->fillRect(_previousRect = QRect(p1, p2), QColor(255, 255, 128, 128));
}

void TrackView::showEvent(QShowEvent *e)
{
    if(!model())
    {
        return;
    }
    QModelIndex index = model()->index(4, 0);
    index = model()->index(0, 0, index);
    index = model()->index(0, 0, index);
    scrollTo(index, QAbstractItemView::PositionAtTop);
}

void TrackView::dataChanged(const QModelIndex &topLeft, const QModelIndex &bottomRight)
{
    if(!topLeft.isValid() || !bottomRight.isValid() || !topLeft.parent().isValid() || topLeft.parent() != bottomRight.parent())
    {
        return;
    }

    QRegion rgn;
    for(int i = topLeft.row(); i <= bottomRight.row(); i++)
    {
        QModelIndex index = model()->index(i, 0, topLeft.parent());
        QRect r(visualRect(index));
        r = QRect(r.x() - _widget->x(), r.y() - _widget->y(), r.width(), r.height());
        rgn = rgn + r;
    }
    _widget->update(rgn);
}

