#include "ControlView.h"

#include "ViewSettings.h"
#include "../configure.h"

#include <QHash>
#include <QMouseEvent>
#include <QPainter>
#include <QPaintEvent>
#include <QResizeEvent>
#include <QUndoStack>
#include <cmath>

using namespace stand::view;
using namespace stand::model;

ControlView::ControlView(TreeItem *item, ViewSettings *s, QWidget *parent) :
    QWidget(parent)
{
    _controlProperty = item;
    _control = static_cast<ControlItem *>(item->child(0));
    _settings = NULL;
    setSettings(s);
    setAutoFillBackground(false);
    _bottom = -1;
    _top = 1;
    _dragBegin = _dragEnd = INT_MIN;
    _isEditable = (item->flags() & Qt::ItemIsEditable);
    setFocusPolicy(Qt::ClickFocus);
    setRange(item->data(ControlItem::MinimumValue, 0).toDouble(), item->data(ControlItem::MaximumValue, 0).toDouble());
    _isTop = false;
}

ControlView::~ControlView()
{
    if(_settingsIsDeletable)
    {
        delete _settings;
    }
}

void ControlView::releaseSettings()
{
    disconnect(this, SLOT(releaseSettings()));
    _settings = new ViewSettings(1, 16, 64, this);
    _settingsIsDeletable = true;
}

void ControlView::setSettings(ViewSettings *s)
{
    disconnect(this, SLOT(releaseSettings()));
    disconnect(this, SLOT(BarMoved(int,int)));
    disconnect(this, SLOT(widthChanged(int)));
    if(_settingsIsDeletable)
    {
        delete _settings;
    }
    _settings = s;
    if(s)
    {
        connect(_settings, SIGNAL(AboutToDelete()), this, SLOT(releaseSettings()));
        _settingsIsDeletable = false;
    }
    else
    {
        _settings = new ViewSettings(1, 16, 64, this);
        _settingsIsDeletable = true;
    }
    connect(_settings, SIGNAL(BarMoved(int,int)), this, SLOT(BarMoved(int,int)));
    connect(_settings, SIGNAL(widthChanged(int)), this, SLOT(widthChanged(int)));
}

void ControlView::setRange(double bottom, double top)
{
    _bottom = bottom;
    _top = top;
}

void ControlView::BarMoved(int oldX, int newX)
{
    QRect r1(oldX, 0, 1, height()), r2(newX, 0, 1, height());
    QRegion rgn;
    rgn = rgn + r1 + r2;
    update(rgn);
}

void ControlView::widthChanged(int w)
{
    setMinimumWidth(std::max(w, minimumWidth()));
}

double ControlView::valueAt(int y) const
{
    double val = (double)(height() - y) / (double)height();
    val = _bottom + (_top - _bottom) * val;
    return val;
}

int ControlView::yAt(double value) const
{
    int y = (1.0 - (value - _top) / (_top - _bottom)) * height();
    return y;
}

void ControlView::paintEvent(QPaintEvent *e)
{
    if(!_control)
    {
        return;
    }
    QPainter p(this);
    QPen pen(QColor(255, 255, 255, 128));
    p.setPen(pen);
    const QList<stand::utility::ControlPoint> &l = _control->contour();
    if(l.empty())
    {
        return;
    }

    int index = 0;
    if(_isTop)
    {
        pen.setColor(QColor(128, 128, 255, 192));
    }
    for(int i = e->rect().left() - 1; i <= e->rect().right(); i++)
    {
        int tick = _settings->tickAt(i);
        while(index < l.size() - 1 && l[index+1].tick <= tick)
        {
            index++;
        }
        double val = l[index].value;
        int top;
        double r = (_top - val) / (_top - _bottom);
        top = r * height();
        p.drawLine(i, top, i, height());
        if(_isTop)
        {
            p.setPen(Qt::white);
            p.drawPoint(i, top);
            p.setPen(pen);
        }
    }
    if(!_isTop)
    {
        return;
    }
    p.fillRect(_dragBegin, 0, _dragEnd - _dragBegin, height(), QColor(255, 255, 255, 96));
    p.setPen(QColor(128, 255, 128, 128));
    if(_isEditable)
    {
        for(int i = _dragBegin; i <= _dragEnd; i++)
        {
            if(_dragPos.contains(i))
            {
                p.drawLine(i, _dragPos[i], i, height());
            }
        }
    }
    pen.setColor(QColor(255, 128, 128, 128));
    p.setPen(pen);
    p.drawLine(_settings->currentBarX(), 0, _settings->currentBarX(), height());
}

void ControlView::mousePressEvent(QMouseEvent *e)
{
    if(e->button() != Qt::LeftButton)
    {
        return;
    }
    _dragPos.clear();
    _prevClicked = e->pos();
    _dragPos[e->x()] = e->y();
    _dragBegin = _dragEnd = e->x();
    _isClicked = true;
}

void ControlView::mouseMoveEvent(QMouseEvent *e)
{
    if(!_isClicked)
    {
        return;
    }

    int smaller, bigger;
    int smallerY, biggerY;
    if(e->x() < _prevClicked.x())
    {
        smaller = e->x();
        smallerY = e->y();
        bigger = _prevClicked.x();
        biggerY = _prevClicked.y();
    }
    else
    {
        smaller = _prevClicked.x();
        smallerY = _prevClicked.y();
        bigger = e->x();
        biggerY = e->y();
    }
    for(int i = smaller; i < bigger; i++)
    {
        double r = (double)(bigger - i) / (double)(bigger - smaller);
        _dragPos[i] = r * smallerY + (1 - r) * biggerY;
    }
    if(smaller == bigger)
    {
        _dragPos[smaller] = smallerY;
    }

    _dragBegin = std::min(smaller, _dragBegin);
    _dragEnd = std::max(bigger, _dragEnd);

    QRect r(smaller - 1, 0, bigger - smaller + 2, height());
    QWidget *p = dynamic_cast<QWidget *>(parent());
    if(p)
    {
        p->update();//p->repaint(r.x() - x(), r.y() - y(), r.width(), r.height());
    }
    else
    {
        update(r);
    }
    _prevClicked = e->pos();
}

void ControlView::mouseReleaseEvent(QMouseEvent *e)
{
    if(!_isClicked)
    {
        return;
    }
    _isClicked = false;

    if(!_isEditable)
    {
        _dragPos.clear();
        int begin = _dragBegin;
        _dragBegin = _dragEnd = -1;
        update(begin, 0, width() - begin, height());
        return;
    }

    ControlCommand *c = new ControlCommand();
    c->control = _control;
    c->view = this;

    int prevTick = INT_MIN, prevY = INT_MIN;
    for(int i = _dragBegin; i <= _dragEnd; i++)
    {
        if(!_dragPos.contains(i))
        {
            continue;
        }
        int tick = _settings->tickAt(i);
        tick = tick - tick % _settings->controlResolution();
        if(prevTick == tick)
        {
            continue;
        }
        int yAtTick = _dragPos[i];
        if(prevY == yAtTick)
        {
            continue;
        }
        double val = valueAt(yAtTick);

        stand::utility::ControlPoint p = {tick, val};
        c->newP.append(p);
        prevY    = yAtTick;
        prevTick = tick;
    }
    if(c->newP.empty())
    {
        delete c;
        return;
    }
    c->oldP = _control->contour(c->newP.first().tick, c->newP.last().tick);

    _dragPos.clear();
    int begin = _dragBegin;
    _dragBegin = _dragEnd = -1;
    update(begin, 0, width() - begin, height());

    if(_settings->undoStack())
    {
        _settings->undoStack()->push(c);
    }
    else
    {
        c->redo();
        delete c;
    }
}

ControlView::ControlCommand::ControlCommand(QUndoCommand *parent)
    : QUndoCommand(parent)
{

}

void ControlView::ControlCommand::undo()
{
    if(oldP.empty())
    {
        return;
    }
    qDebug("ControlCommand::undo();");
    qDebug(" insert control points from %d to %d.", oldP.first().tick, oldP.last().tick);
    control->replace(oldP, newP.first().tick, newP.last().tick);
    QWidget *p = dynamic_cast<QWidget *>(view->parent());
    if(p)
    {
        p->update();
    }
    else
    {
        view->update();
    }
}

void ControlView::ControlCommand::redo()
{
    if(newP.empty())
    {
        return;
    }
    qDebug("ControlCommand::redo();");
    qDebug(" insert control points from %d to %d.", newP.first().tick, newP.last().tick);
    control->replace(newP, oldP.first().tick, oldP.last().tick);
    QWidget *p = dynamic_cast<QWidget *>(view->parent());
    if(p)
    {
        p->update();
    }
    else
    {
        view->update();
    }
}
