#include "Manager.h"
#include "Worker.h"

#include <QThread>

using namespace stand::utility::task;

Manager::Manager(QList<Task *> tasks, int numThreads, bool background, QObject *parent) :
    QObject(parent)
{
    _running = false;
    set(tasks, numThreads, background);
}

Manager::~Manager()
{
    _destroy();
}

void Manager::_destroy()
{
    for(int i = 0; i < _threads.size(); i++)
    {
        delete _threads[i];
    }
    qDeleteAll(_workers);
    _workers.clear();
    _threads.clear();
    _running = false;
    _finishCount = 0;
}

bool Manager::set(QList<Task *> tasks, int numThreads, bool background)
{
    _mutex.lock();
    if(_running)
    {
        _mutex.unlock();
        qDebug("Manager::set(); // Manager cannot be set while running.");
        return false;
    }
    _destroy();
    _background = background;
    numThreads = std::max(numThreads, 1);

    for(int i = 0; i < numThreads; i++)
    {
        QList<Task *> t;
        _threads.push_back(new QThread);
        for(int j = i * tasks.size() / numThreads; j < (i + 1) * tasks.size() / numThreads; j++)
        {
            t.push_back(tasks[j]);
        }
        _workers.push_back(new Worker(t));
        _workers[i]->moveToThread(_threads[i]);
    }

    _mutex.unlock();
    return true;
}

void Manager::start()
{
    _mutex.lock();
    if(_running)
    {
        _mutex.unlock();
        return;
    }
    _running = true;

    for(int i = 0; i < _workers.size(); i++)
    {
        connect(_workers[i], SIGNAL(finished(bool)), this, SLOT(_workerFinished()), Qt::DirectConnection);
        connect(this, SIGNAL(canceled()), _workers[i], SLOT(cancel()), Qt::DirectConnection);
        QMetaObject::invokeMethod(_workers[i], "run");
        _threads[i]->start();
    }
    _mutex.unlock();
}

void Manager::cancel()
{
    if(_running)
    {
        emit canceled();
        emit finished();
    }
}

void Manager::_workerFinished()
{
    _mutex.lock();
    if(!_running)
    {
        _mutex.unlock();
        return;
    }
    _finishCount++;
    _mutex.unlock();
    if(_finishCount == _threads.size())
    {
        _running = false;
        emit finished();
    }
}
