#include "EchoesTest.h"

#include "SequenceModel.h"
#include "TrackItem.h"
#include "helper/WaveAnalysisTask.h"
#include "ControlItem.h"

#include "Alignment.h"
#include "Analyzer.h"
#include "Data.h"
#include "Dictionary.h"
#include "Echoes.h"
#include "EventItem.h"

#include "Utility.h"

#include <QTest>

void EchoesTest::analysisTest_data()
{
    QTest::addColumn<QString>("stfxPath");
    QTest::addColumn<QString>("wavePath");
    QTest::addColumn<QString>("logPath");

    QTest::newRow("RealData") << "test/data/test.stfx" << "test/data/test.wav" << "test/data/analysisTest1.log";
}

void EchoesTest::analysisTest()
{
    QFETCH(QString, stfxPath);
    QFETCH(QString, wavePath);
    QFETCH(QString, logPath);

    const double framePeriod = 20.0;

    stand::model::SequenceModel sequence(this);
    QVERIFY(sequence.read(stfxPath));
    stand::gui::helper::WaveAnalysisTask analysis(&sequence, sequence.track(0), wavePath, framePeriod, this);
    analysis.run();

    stand::echoes::Dictionary dict;
    stand::echoes::Analyzer analyzer;

    QVERIFY(dict.analyze(&analyzer, &sequence, 0, 0, 1));
    QCOMPARE(dict.size(), sequence.track(0)->events()->childCount());

    FILE *fp = fopen(logPath.toUtf8().data(), "w");
    if(!fp)
    {
        qDebug("Log File cannnot be open");
        return;
    }

    for(int i = 0; i < dict.size(); i++)
    {
        for(int j = 0; j < dict.at(i).second.length(); j++)
        {
            if(dict.at(i).second.f0()[j] != 0.0)
            {
                fprintf(fp, "%d\t%f\n", j - dict.at(i).second.changeIndex(), dict.at(i).second.f0()[j]);
            }
        }
        fprintf(fp, "\n\n\n");
    }
    fclose(fp);
}

void EchoesTest::ioTest_data()
{
    QTest::addColumn<QString>("stfxPath");
    QTest::addColumn<QString>("wavePath");
    QTest::addColumn<QString>("dictPath");

    QTest::newRow("RealData") << "test/data/test.stfx" << "test/data/test.wav" << "test/data/ioTest1.dict";
}

void EchoesTest::ioTest()
{
    QFETCH(QString, stfxPath);
    QFETCH(QString, wavePath);
    QFETCH(QString, dictPath);

    const double framePeriod = 10.0;

    stand::model::SequenceModel sequence(this);
    QVERIFY(sequence.read(stfxPath));
    stand::gui::helper::WaveAnalysisTask analysis(&sequence, sequence.track(0), wavePath, framePeriod, this);
    analysis.run();

    stand::echoes::Dictionary dict, dict2;
    stand::echoes::Analyzer analyzer;

    QVERIFY(dict.analyze(&analyzer, &sequence, 0, 0, 1));

    QCOMPARE(dict.size(), sequence.track(0)->events()->childCount());

    QVERIFY2(dict.write(dictPath), "Error while writing dictionary");
    QVERIFY2(dict2.read(dictPath), "Error while reading dictionary");
    QVERIFY2(dict.size() == dict2.size(), "Error; size not equal");

    bool r = true;
    for(int i = 0; i < dict.size(); i++)
    {
        r &= dict.at(i) == dict2.at(i);
    }
    QVERIFY2(r, "Error; comparison failed");
}

void EchoesTest::execTest_data()
{
    QTest::addColumn<QString>("stfxPath");
    QTest::addColumn<QString>("dictPath");
    QTest::addColumn<QString>("outPath");

    QTest::newRow("correctData")  << "test/data/test.stfx" << "test/data/ioTest1.dict" << "test/data/execTestOutput.stfx";
}

void EchoesTest::execTest()
{
    using namespace stand::model;
    QFETCH(QString, stfxPath);
    QFETCH(QString, dictPath);
    QFETCH(QString, outPath);

    stand::model::SequenceModel sequence;
    QVERIFY(sequence.read(stfxPath));
    stand::echoes::Dictionary dictionary;
    QVERIFY(dictionary.read(dictPath));

    stand::echoes::Computer::Contours contours;
    QList<stand::model::TreeItem *> controls = sequence.track(0)->controls();
    contours.f0 = dynamic_cast<stand::model::ControlItem *>(controls[0]->child(0));
    contours.volume = dynamic_cast<stand::model::ControlItem *>(controls[1]->child(0));
    contours.dictionary = &dictionary;
    contours.beginTick = 0;
    contours.step = 16;

    for(int i = 0; i < sequence.track(0)->events()->childCount(); i++)
    {
        EventItem *item = dynamic_cast<EventItem *>(sequence.track(0)->events()->child(i));
        contours.events.append(item);
    }
    contours.endTick = contours.events.last()->tick() + contours.events.last()->length();
    stand::echoes::Echoes echoes;
    QVERIFY(echoes.compute(contours));

    sequence.write(outPath);
}

