﻿#include "Sequence.h"
#include "importer/Importer.h"

#include "Track.h"
#include "Tempo.h"
#include "Measure.h"

#include <QStringList>

using namespace stand::sequence;

QHash<QString, importer::Interpreter* (Sequence::*)(const QString &)> Sequence::Factories;

Sequence::Sequence(QObject *parent) :
    QObject(parent)
{
    _tempo = new Tempo();
    _measure = new Measure();
    _tracks.clear();
    _tempo->clear(120.0);

    if(Factories.empty())
    {
        Factories["Mixer"] = &Sequence::_createMixer;
        Factories["Tempo"] = &Sequence::_createTempo;
        Factories["Track"] = &Sequence::_createTrack;
        Factories["Measure"] = &Sequence::_createMeasure;
        Factories["Sequence"] = &Sequence::_createSequence;
        Factories["PreMeasure"] = &Sequence::_setPreMeasure;

        Factories["PitchBend"] = &Sequence::_createControl;
        Factories["PBS"] = &Sequence::_createControl;
        Factories["Dynamics"] = &Sequence::_createControl;
        Factories["Residual"] = &Sequence::_createControl;
    //    Factories["Slope"] = &Sequence::_createControl;
    //    Factories["SlopeDepth"] = &Sequence::_createControl;
        Factories["Gender"] = &Sequence::_createControl;
    //    Factories["Portamento"] = &Sequence::_createControl;
    //    Factories["Tremolo"] = &Sequence::_createControl;
    //    Factories["Opening"] = &Sequence::_createControl;

        _defaults["PitchBend"] = 0;
        _defaults["PBS"] = 2;
        _defaults["Dynamics"] = 64;
        _defaults["Residual"] = 0;
    //    _defaults["Slope"] = 64;
    //    _defaults["SlopeDepth"] = 64;
        _defaults["Gender"] = 64;
    //    _defaults["Portamento"] = 0;
    //    _defaults["Tremolo"] = 0;
    //    _defaults["Opening"] = 128;
    }
    _preMeasure = DefaultPreMeasure;
}

Sequence::~Sequence()
{
    _destroy();
}

void Sequence::_destroy()
{
    _tempo->clear(120.0);
    _measure->clear();
    for(QList<Track *>::iterator it = _tracks.begin(); it != _tracks.end(); it++)
    {
        delete (*it);
    }
    _tracks.clear();
    _dictionary.clear();
}

Track *Sequence::track(int index)
{
    if(index < 0 || _tracks.empty() || index >= _tracks.size())
    {
        return NULL;
    }
    return _tracks.at(index);
}

int Sequence::tickAt(double ms) const
{
    return _tempo->tickAt(ms);
}

double Sequence::msAt(int tick) const
{
    return _tempo->msAt(tick);
}

bool Sequence::import(importer::Importer *importer)
{
    if(!importer || !importer->isOpen())
    {
        qDebug("Sequence::read(%d); // invalid args", importer);
        return false;
    }

    _destroy();
    _trackId = 0;
    _dictionary.insert("Sequence", this);

    QString name, oper, line;
    int c = 0;
    while(importer->pop(line))
    {
        c++;
        QStringList sl = line.split(" ", QString::SkipEmptyParts);
        if(sl.size() < 1)
        {
            qDebug(" Sequence::import();");
            qDebug("  Invalid operation in l.%d : %s", c, line.toUtf8().data());
        }
        name = sl.at(0);
        oper = sl.at(1);
        for(int i = 2; i < sl.size(); i++)
        {
            oper += " " + sl.at(i);
        }
        QHash<QString, Interpreter *>::iterator it = _dictionary.find(name);
        Interpreter *p = NULL;
        if(it != _dictionary.end())
        {
            p = it.value();
        }
        if(p)
        {
            p = p->interpret(oper);
            if(p)
            {
                _dictionary.insert(p->name(), p);
            }
        }
        else
        {
            qDebug(" Sequence::import();");
            qDebug("  Undefined Symbol; %s in line %d.", line.toUtf8().data(), c);
        }
    }
    if(!_measure)
    {
        _measure = new Measure();
        _measure->setName("measure");
        _dictionary.insert(_measure->name(), _measure);
    }
    qDebug("Sequence::import();");
    qDebug(" Imported: %d items.", _dictionary.size());
    return true;
}

importer::Interpreter *Sequence::interpret(const QString &line)
{
    Interpreter *p = NULL;
    QHash<QString, Interpreter* (Sequence::*)(const QString &)>::iterator it;
    QStringList sl = line.split(" ", QString::SkipEmptyParts);
    if(sl.size() != 1 && sl.size() != 2)
    {
        return NULL;
    }
    it = Factories.find(sl.at(0));
    if(it != Factories.end())
    {
        p = (this->*(it.value()))(sl.at(1));
    }
    return p;
}

importer::Interpreter *Sequence::_setPreMeasure(const QString &val)
{
    bool result;
    int v = val.toInt(&result);
    if(!result)
    {
        v = DefaultPreMeasure;
    }
    _preMeasure = v;
    return NULL;
}

importer::Interpreter *Sequence::_createMixer(const QString &name)
{
    // ToDo:: ミキサーを入れるならここ．
    return NULL;
}

importer::Interpreter *Sequence::_createSequence(const QString &name)
{
    this->setName(name);
    return this;
}

importer::Interpreter *Sequence::_createTempo(const QString &name)
{
    _tempo->clear(120.0);
    _tempo->setName(name);
    return _tempo;
}

importer::Interpreter *Sequence::_createTrack(const QString &name)
{
    Track *t = new Track(this);
    t->setName(name);
    this->_tracks.push_back(t);
    _trackId++;
    return t;
}

importer::Interpreter *Sequence::_createMeasure(const QString &name)
{
    _measure->clear();
    _measure->setName(name);
    return _measure;
}

importer::Interpreter *Sequence::_createControl(const QString &name)
{
    ControlCurve *c = new ControlCurve(_defaults[name]);
    c->setName(name);
    _tracks.at(_trackId - 1)->addControl(c);
    _dictionary[c->name().toLower() + QString::number(_trackId - 1)] = c;
    return NULL;
}

const QString Sequence::toString() const
{
    QString val;
    val = "Sequence " + name() + "\n";
    if(_tempo)
    {
        val += name() + " Tempo " + _tempo->name() + "\n";
        val += _tempo->toString();
    }
    if(_measure)
    {
        val += name() + " Measure " + _measure->name() + "\n";
        val += _measure->toString();
    }
    for(int i = 0; i < _tracks.size(); i++)
    {
        val += name() + " Track " + _tracks[i]->name() + "\n";
        val += _tracks[i]->toString();
    }

    return val;
}

