#include "UtauCorpusImporter.h"

#include "UtauLibrary.h"
#include "PrefixMap.h"
#include "CharacterSettings.h"

#include "CorpusModel.h"
#include "TreeItem.h"
#include "PhonemeItem.h"

#include "Utility.h"

#include <QFileInfoList>

using namespace stand::model::importer;
using namespace stand::model;
using namespace stand::io::utau;

bool UtauCorpusImporter::import(TreeItem *root, const QString &filepath) const
{
    qDebug("UtauCorpusImporter::import(%s);", filepath.toUtf8().data());

    PrefixMap prefix;
    CharacterSettings character;
    if(prefix.read(filepath + QDir::separator() + "prefix.map"))
    {
        qDebug(" Found prefix.map.");
    }

    if(character.read(filepath + QDir::separator() + "character.txt"))
    {
        qDebug(" Found character.txt");
        _setRoot(root, &character);
    }
    else
    {
        root->setData(CorpusModel::Name, QDir(filepath).path(), 0);
    }

    LibraryNode *node = new LibraryNode;

    node->read(filepath);

    QStringList alias = _getAllAlias(prefix.prefixList(), node);

    QHash<QString, TreeItem *> dictionary;

    for(int i = 0; i < alias.size(); i++)
    {
        qDebug(" current state = %d / %d", i, alias.size());
        if(dictionary.contains(alias[i]))
        {
            continue;
        }
        dictionary[alias[i]] = _createItem(root, node->phonemes(prefix.prefixList(), alias[i]), alias[i], QDir(filepath));
    }

    delete node;

    return true;
}

TreeItem *UtauCorpusImporter::_createItem(TreeItem *root, QList<LibraryNode::Item> phonemes, const QString &alias, QDir dir)
{
    if(phonemes.empty())
    {
        return NULL;
    }
    TreeItem *item = new TreeItem(alias, root);
    root->appendChild(item);
    for(int i = 0; i < phonemes.size(); i++)
    {
        TreeItem *child = new PhonemeItem(alias, item);
        item->appendChild(child);
        phonemes[i].note = stand::utility::NoteNumber(phonemes[i].prefix);
        _setItem(child, phonemes[i], dir);
    }
}

QStringList UtauCorpusImporter::_getAllAlias(const QStringList &prefix, LibraryNode *root)
{
    if(!root)
    {
        return QStringList();
    }
    QStringList l;
    if(root->library)
    {
        for(int i = 0; i < root->library->size(); i++)
        {
            const UtauPhoneme *p = root->library->at(i);
            QString val = p->pronounce;
            for(int j = 0; j < prefix.size(); j++)
            {
                val.remove(prefix[j]);
            }
            l.push_back(val);
        }
    }

    for(int i = 0; i < root->children.size(); i++)
    {
        QStringList tmp = _getAllAlias(prefix, root->children[i]);
        while(!tmp.empty())
        {
            l.push_back(tmp.takeFirst());
        }
    }
    return l;
}

void UtauCorpusImporter::_setRoot(TreeItem *root, CharacterSettings *settings)
{
    root->setData(CorpusModel::Name, settings->name(), 0);
    root->setData(CorpusModel::Address, settings->web(), 0);
    root->setData(CorpusModel::Author, settings->author(), 0);
    root->setData(CorpusModel::ImagePath, settings->imagePath(), 0);
    root->setData(CorpusModel::SamplePath, settings->sample(), 0);
}

void UtauCorpusImporter::_setItem(TreeItem *item, LibraryNode::Item phoneme, QDir rootdir)
{
    item->setData(PhonemeItem::Alias, phoneme.phoneme->pronounce, 0);
    QString path = QDir::toNativeSeparators(phoneme.directory.absolutePath() + phoneme.phoneme->filename);
    item->setData(PhonemeItem::FilePath, rootdir.relativeFilePath(path), 0);
    if(phoneme.phoneme->filename.contains(".vvd"))
    {
        item->setData(PhonemeItem::FileType, PhonemeItem::MelCepstrumWorld, 0);
        item->setData(PhonemeItem::Length, -phoneme.phoneme->rightBlank, 0);
    }
    else
    {
        item->setData(PhonemeItem::FileType, PhonemeItem::RawWave, 0);
        item->setData(PhonemeItem::Length, (int)stand::utility::msLengthForUtauPhoneme(path.toLocal8Bit().data(), phoneme.phoneme), 0);
    }

    item->setData(PhonemeItem::LeftBlank, phoneme.phoneme->leftBlank, 0);
    item->setData(PhonemeItem::FixedLength, phoneme.phoneme->fixedLength, 0);
    item->setData(PhonemeItem::Overlap, phoneme.phoneme->voiceOverlap, 0);
    item->setData(PhonemeItem::Preutterance, phoneme.phoneme->preUtterance, 0);
    item->setData(PhonemeItem::Note, phoneme.note, 0);
}

UtauCorpusImporter::LibraryNode::LibraryNode()
{
    library = NULL;
}

UtauCorpusImporter::LibraryNode::~LibraryNode()
{
    delete library;
    qDeleteAll(children);
}

void UtauCorpusImporter::LibraryNode::read(const QString &filepath)
{
    qDebug("LibraryNode::read(%s);", filepath.toUtf8().data());
    library = new UtauLibrary();
    library->readRecursive(filepath, QTextCodec::codecForName("Shift-JIS"), 0);

    QDir dir(filepath);
    QFileInfoList dirList = dir.entryInfoList(QDir::AllDirs | QDir::NoDot | QDir::NoDotDot);
    for(int i = 0; i < dirList.size(); i++)
    {
        LibraryNode *node = new LibraryNode();
        node->read(QDir::toNativeSeparators(dir.absoluteFilePath(dirList.at(i).fileName())));
        children.push_back(node);
    }
}

QList<UtauCorpusImporter::LibraryNode::Item> UtauCorpusImporter::LibraryNode::phonemes(const QStringList &prefix, const QString &key)
{
    // 頑張って読みだしてくる
    QList<Item> ret;

    for(int i = 0; i < prefix.size(); i++)
    {
        const UtauPhoneme *p = library->find(key + prefix[i]);
        if(p)
        {
            Item item;
            item.phoneme = p;
            item.prefix = prefix[i];
            item.directory = library->directory();
            ret.push_back(item);
        }
    }
    for(int i = 0; i < children.size(); i++)
    {
        QList<Item> tmp = children[i]->phonemes(prefix, key);
        for(int j = 0; j < tmp.size(); j++)
        {
            ret.push_back(tmp[j]);
        }
    }
    return ret;
}
