#include "TrackItem.h"

#include "ControlItem.h"
#include "EventItem.h"

#include "Track.h"
#include "ControlCurve.h"
#include "NoteEvent.h"
#include "SingerEvent.h"

#include "envelope/Envelope.h"

using namespace stand::model;

TrackItem::TrackItem(const QString &name, TreeItem *parent) :
    TreeItem(name, parent)
{
    appendChild(new TreeItem(tr("Events"), this));
    TreeItem *item;

    item = createControl("F0 Contour", 440.0);
    item->setData(ControlItem::ViewType, ControlItem::TypePitch, 0);
    item->setData(ControlItem::TimeType, ControlItem::TypeFrame, 0);
    item->setData(ControlItem::FramePeriod, 2.0, 0);

    item = createControl("Volume", -24.0);
    item->setData(ControlItem::TimeType, ControlItem::TypeFrame,0);
    item->setData(ControlItem::FramePeriod, 2.0, 0);
    item->setFlags(Qt::NoItemFlags);
}

TrackItem::~TrackItem()
{
}

int TrackItem::length() const
{
    const TreeItem *events = child(0);
    if(events->childCount() <= 0)
    {
        return 0;
    }
    const TreeItem *last = events->child(events->childCount() - 1);
    int tick = last->data(EventItem::TickData, 0).toInt();
    int length = last->data(EventItem::LengthData, 0).toInt();
    return tick + length;
}

QList<TreeItem *> TrackItem::controls() const
{
    QList<TreeItem *> ret;
    for(int i = 1; i < this->childCount(); i++)
    {
        TreeItem *item = this->child(i);
        const ControlItem *c;
        if(item && (c = dynamic_cast<ControlItem *>(item->child(0))))
        {
            ret.push_back(item);
        }
    }
    return ret;
}

ControlItem *TrackItem::controlAt(int id) const
{
    // ゼロ番目の子は必ず events なのでひとつ足す．
    TreeItem *item = this->child(id + 1);
    if(item)
    {
        ControlItem *c = dynamic_cast<ControlItem *>(item->child(0));
        return c;
    }
    return NULL;
}

void TrackItem::setTrack(stand::sequence::Track *track)
{
    if(!track)
    {
        return;
    }
    QVector<stand::sequence::NoteEvent *> notes = track->notes();

    for(int i = 0; i < notes.size(); i++)
    {
        if(notes[i]->length() == 1)
        {
            continue;
        }
        EventItem *item = new EventItem("", child(0));
        item->setData(EventItem::TickData, notes[i]->tick(),0);
        item->setData(EventItem::LengthData, notes[i]->length(),0);
        item->setData(EventItem::NoteData, notes[i]->note(),0);
        item->setData(EventItem::VelocityData, notes[i]->velocity(),0);
        item->setData(EventItem::LyricData, notes[i]->lyric(),0);
        item->setData(EventItem::PronounceData, notes[i]->pronounce(),0);
        child(0)->appendChild(item);
    }

    for(int i = 0; i < track->curves().size(); i++)
    {
        TreeItem *item = createControl(track->curves()[i]->name(), track->curves()[i]->defaultValue());
        ControlItem *c = (ControlItem *)(item->child(0));
        c->setControl(track->curves()[i]);
        double maxval = 1, minval = -1;
        if(item->name() == "PitchBend")
        {
            maxval = 8191;
            minval = -8192;
        }
        else if(item->name() == "PBS")
        {
            maxval = 24;
            minval = 0;
        }
        else
        {
            maxval = 127;
            minval = 0;
        }
        item->setData(ControlItem::MaximumValue, maxval, 0);
        item->setData(ControlItem::MinimumValue, minval, 0);
    }
}

void TrackItem::setUpDomElement(QDomDocument & /*doc*/, QDomElement &element)
{
    element.setTagName("track");
}

TreeItem *TrackItem::createControl(const QString &name, double value)
{
    TreeItem *item = new TreeItem(name, this);
    item->appendChild(new ControlItem("control data", value, item));
    item->setData(ControlItem::ViewType, ControlItem::TypeNormal, 0);
    item->setData(ControlItem::TimeType, ControlItem::TypeTick, 0);
    item->setData(ControlItem::FramePeriod, 2.0, 0);
    item->setData(ControlItem::MaximumValue, 1, 0);
    item->setData(ControlItem::MinimumValue, -1, 0);
    appendChild(item);
    return item;
}

TreeItem *TrackItem::createControl(const QString &name, const stand::utility::envelope::Envelope *e, const QVector<int> &ticks, int viewType)
{
    if(!e || e->tLen() == 0)
    {
        return NULL;
    }
    TreeItem *item = new TreeItem(name);
    item->setParent(this);
    item->appendChild(new ControlItem("control data", e, ticks, item));
    item->setData(ControlItem::ViewType, viewType, 0);
    item->setData(ControlItem::TimeType, ControlItem::TypeFrame, 0);
    item->setData(ControlItem::FramePeriod, e->framePeriod(), 0);
    item->setData(ControlItem::MaximumValue, 1, 0);
    item->setData(ControlItem::MinimumValue, -1, 0);
    appendChild(item);

    return item;
}
