#ifndef UTAULIBRARY_H
#define UTAULIBRARY_H

#include <QString>
#include <QHash>
#include <QVector>
#include <QDir>

#include "UtauPhoneme.h"

namespace stand
{
namespace io
{
namespace utau
{

//! @brief UTAU 音源読み書きクラス
class UtauLibrary
{
public:
    explicit UtauLibrary(const QString &filename = QString(), QTextCodec *codec = NULL, int maxDepth = 1);

    //! @brief 指定された oto.ini からUTAU 音源を読み込みます．
    //! @param[in] filename 読み込む音源の oto.ini パス．
    //! @param[in] codec oto.ini の文字コード．デフォルトは Shift-JIS です．
    //! @retval true: 成功， false: 失敗
    bool readFromOtoIni(const QString &filename, QTextCodec *codec = NULL);

    //! @brief UTAU音源を再帰的に読み込みます．全探索してしまわないように深さで枝を制限できます．
    //! @param[in] filename 読み込む音源の oto.ini パス．
    //! @param[in] codec oto.ini の文字コード．デフォルトは Shift-JIS です．
    //! @param[in] maxDepth 再起読み込みの制限値．
    //! @retval true: 成功， false: 失敗
    bool readRecursive(const QString &filename, QTextCodec *codec = NULL, int maxDepth = 1);

    //! @brief UTAU 音源を書き込みます．
    //! @param[in] filename 書き込む音源の oto.ini パス．
    //! @param[in] codec oto.ini の文字コード．デフォルトは Shift-JIS です．
    //! @retval true: 成功， false: 失敗
    bool write(const QString &filename, QTextCodec *codec = NULL);

    //! @brief 指定番の原音設定を返します．
    //! @param[in] i 得たい原音設定のインデックスです．
    //! @retval NULL: 失敗，それ以外: 成功．
    const UtauPhoneme *at(unsigned int i) const;

    //! @brief UTAU 音源の原音設定の数を返します．
    unsigned int size() const
    {
        return settingList.size();
    }

    bool empty() const
    {
        return settingMap.empty();
    }

    //! @brief UtauLibrary クラスを初期化します．
    void clear()
    {
        settingList.clear();
        settingMap.clear();
    }

    QDir directory() const
    {
        return _directory;
    }

    //! @brief 原音設定の一要素分を追加します．
    /// <param name = s> 追加したい原音設定 </parm>
    void push_back(const UtauPhoneme &s)
    {
        settingMap[s.pronounce] = settingList.count();
        settingList.push_back(s);
    }

    //! @brief 指定発音の原音設定を返します．
    //! @param[in] i 得たい原音設定の発音．
    //! @retval NULL: 失敗，それ以外: 成功．
    const UtauPhoneme *find(const QString &pronounce) const;

    void changeDirectory(const QDir &dst);

private:
    bool _readFromOtoIni(const QString &filename, QTextCodec *codec = NULL);
    bool _readOneLine(QString &line, QString &dir);
    bool _readRecursive(const QString &filename, QTextCodec *codec = NULL, int maxDepth = 1);

    QVector<UtauPhoneme> settingList;
    QHash<QString, int> settingMap;

    QDir _directory;
};

}
}
}

#endif // UTAULIBRARY_H
