#ifndef AUDIOMIXER_H
#define AUDIOMIXER_H

#include <QObject>
#include <QAudioOutput>
#include <QVector>

#include "StreamDevice.h"
#include "AudioTrack.h"

namespace stand
{
namespace io
{
namespace audio
{

/*!
 *  @brief AudioMixer クラスは AudioTrack クラスとその派生クラスを
 *         リアルタイムに再生するためのクラスです．
 *         ミキサークラスは内部バッファを用いてストリーミング再生を行います．
 *         バッファを登録したトラックから適時補いながら再生するため，
 *         処理が間に合わない場合再生を停止することがあります．
 */
class AudioMixer : public QObject
{
    Q_OBJECT
public:
    /*!
     *  @brief コンストラクタでは再生に必要な情報を与えます．
     *  @param[in] format 再生したいオーディオフォーマット
     */
    explicit AudioMixer(const QAudioFormat &format, QObject *parent = 0);
    ~AudioMixer();

    /*!
     *  @brief トラックを追加します．
     *  @param[in] t トラック
     *  @param[in] volume 音量
     *  @param[in] pan ステレオ再生時のパンポット
     */
    void addTrack(AudioTrack *t, int volume, int pan);

    /*!
     *  @brief トラックをミキサーから削除します．削除されたトラックは再生が停止されます．
     *  @param[in] t 削除したいトラック
     */
    void removeTrack(AudioTrack *t);

    /*!
     *  @brief 全てのトラックをミキサーから削除します．全てのトラックは再生が停止されます．
     */
    void clear();

    /*!
     *  @brief 再生フォーマットを変更します．
     *  @param[in] format オーディオフォーマット
     */
    void setFormat(const QAudioFormat &format);

    struct Track
    {
        AudioTrack *t;
        int volume;
        int pan;
    };

signals:
    /*!
     *  @brief 再生時バッファが空になった場合に送信されます．
     */
    void errorEmpty();


public slots:
    //! @brief 追加されているトラックを再生します．
    void start();
    //! @brief 再生を停止します．
    void stop();
    //! @brief 内部に保持しているトラックから内部バッファを作成します．
    //! @param[in] length 作成したいバッファサイズ
    void mixTracks(qint64 length);

private:
    void _destroy();
    qint64 _pos;
    QAudioFormat _format;
    QVector<Track> _tracks;
    QAudioOutput *_output;
    StreamDevice *_device;
};

}
}
}

#endif // AUDIOMIXER_H
