﻿// In the original BSD license, both occurrences of the phrase "COPYRIGHT HOLDERS AND CONTRIBUTORS"
// in the disclaimer read "REGENTS AND CONTRIBUTORS".
//
// Here is the license template:
//
// Copyright (c) 2010, Masanori Usami
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without modification, are permitted provided
// that the following conditions are met:
//
//  * Redistributions of source code must retain the above copyright notice, this list of conditions and the following disclaimer.
//  * Redistributions in binary form must reproduce the above copyright notice, this list of conditions and the following disclaimer
//    in the documentation and/or other materials provided with the distribution.
//  * Neither the name of the <ORGANIZATION> nor the names of its contributors may be used to endorse or promote products derived
//    from this software without specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
// BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
// IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, 
// OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA,
// OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF 
// THE POSSIBILITY OF SUCH DAMAGE.

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Collections;

namespace EAAddinDSM
{
    /// <summary>
    /// パーティショニング等の処理を行うクラス
    /// </summary>
    public class DSMHelper
    {
        /// <summary>
        /// 入力されたDSMの配列をパーティショニングして返す
        /// </summary>
        /// <param name="dsm">パーティショニング前のDSM配列</param>
        /// <returns>パーティショニング後のDSM配列</returns>
        public ArrayList partitioning(int[,] dsm)
        {
            int[,] reach = reachabilityMatrix(dsm);
            ArrayList[] row = partitionRow(reach);
            ArrayList[] column = partitionColumn(reach);
            return getNewSequence(row, column);
        }

        /// <summary>
        /// 与えられたDSM配列からReachability Matrixを作る
        /// </summary>
        /// <param name="dsm">DSM配列</param>
        /// <returns>Reachability Matrix</returns>
        public int[,] reachabilityMatrix(int[,] dsm)
        {
            // 対角には１を設定
            for(int i = 0; i < dsm.GetLength(0); i++)
            {
                dsm[i, i] = 1;
            }
            // Reachability Matrixの計算
            int[,] reach = (int[,])dsm.Clone();
            for (int i = 0; i < dsm.GetLength(0); i++)
            {
                int[,] temp = (int[,])reach.Clone();
                for (int j = 0; j < dsm.GetLength(0); j++)
                {
                    for (int m = 0; m < dsm.GetLength(0); m++)
                    {
                        for (int k = 0; k < dsm.GetLength(0); k++)
                        {
                            reach[j, m] = temp[j, k] * dsm[k, m] + reach[j, m];
                            if (reach[j, m] > 1)
                            {
                                reach[j, m] = 1;
                            }
                        }
                    }
                }
            }
            return reach;
        }

        /// <summary>
        /// Reachability Matrixの行成分をArrayListにして返す
        /// </summary>
        /// <param name="reach">Reachability Matrix（2次元配列）</param>
        /// <returns>Reachability Matrixの行成分の配列</returns>
        private ArrayList[] partitionRow(int[,] reach)
        {
            ArrayList[] row = new ArrayList[reach.GetLength(0)];
            for (int i = 0; i < reach.GetLength(0); i++)
            {
                row[i] = new ArrayList();
                for (int j = 0; j < reach.GetLength(0); j++)
                {
                    if (reach[i, j] != 0)
                    {
                        row[i].Add(j);
                    }
                }
            }
            return row;
        }

        /// <summary>
        /// Reachability Matrixの列成分をArrayListにして返す
        /// </summary>
        /// <param name="reach">Reachability Matrix（2次元配列）</param>
        /// <returns>Reachability Matrixの列成分の配列</returns>
        private ArrayList[] partitionColumn(int[,] reach)
        {
            ArrayList[] column = new ArrayList[reach.GetLength(0)];
            for (int i = 0; i < reach.GetLength(0); i++)
            {
                column[i] = new ArrayList();
                for (int j = 0; j < reach.GetLength(0); j++)
                {
                    if (reach[j, i] != 0)
                    {
                        column[i].Add(j);
                    }
                }
            }
            return column;

        }

        /// <summary>
        /// 与えられたReachability Matrixから新しい要素の並びを計算する
        /// </summary>
        private ArrayList getNewSequence(ArrayList[] row, ArrayList[] column)
        {
            int[] taken = new int[row.GetLength(0)];
            for(int i = 0; i < taken.GetLength(0); i++)
            {
                taken[i] = 0;
            }

            ArrayList newSeq = new ArrayList();
            for(; newSeq.Count < row.GetLength(0); )
            {
                for (int i = 0; i < row.GetLength(0); i++)
                {
                    if (checkRow(taken, row[i], column[i]))
                    {
                        foreach (int val in row[i])
                        {
                            if (taken[val] == 0)
                            {
                                newSeq.Add(val);
                                taken[val] = 2;
                            }
                        }
                    }
                }
                for (int i = 0; i < taken.GetLength(0); i++)
                {
                    if (taken[i] == 2)
                    {
                        taken[i] = 1;
                    }
                }
            }
            return newSeq;
        }

        /// <summary>
        /// 処理対象の要素（Row）かどうかのチェック
        ///  未処理の要素に参照されていれば処理対象外
        ///  相互参照の要素と処理済の要素にのみ参照されていれば処理対象
        /// </summary>
        /// <param name="taken">処理済要素を示す配列</param>
        /// <param name="row">チェック対象の行</param>
        /// <param name="column"></param>
        /// <returns>true:処理対象 false:処理対象外</returns>
        private bool checkRow(int[] taken, ArrayList row, ArrayList column)
        {
            bool signal = false;
            foreach(int val in row)
            {
                if (taken[val] != 1)
                {
                    signal = column.Contains(val);
                }
                else if (taken[val] == 1)
                {
                    signal = true;
                }
                if (!signal)
                {
                    break;
                }
            }
            return signal;
        }
    }
}
