/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.v2.cache;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.logging.Logger;
public class MemoryCache<K, V>
extends LinkedHashMap<K, V>
{
  private static final long serialVersionUID = 5029019894952017777L;
  private static final Logger LOGGER = Logger.getLogger(MemoryCache.class
      .getName());
  private CacheCostHandler<K, V> costHandler;
  private CacheSerializationHandler<K, V> serializationHandler;

  public MemoryCache(CacheCostHandler<K,V> costHandler, CacheSerializationHandler<K, V> serializationHandler)
  {
    setCostHandler(costHandler);
    setSerializationHandler(serializationHandler);
  }

  @Override
  protected boolean removeEldestEntry(Map.Entry<K, V> entry)
  {
    K key = entry.getKey();
    V value = entry.getValue();
    boolean wantPurge = getCostHandler().wantPurge(this, key, value);
    LOGGER.finest("wantPurge=" + wantPurge);
    if (wantPurge) {
      getCostHandler().modifyCostsOnRemove(this, key, value);
      marshal(key, value);
    }
 
    return wantPurge;
  }
  
  public void flush(K key)
  {
    V value = remove(key);
    if (value != null) {
      marshal(key, value);
    }
  }
  
  protected void marshal(K key, V value) {
    getSerializationHandler().marshal(key, value);
  }
  
  @SuppressWarnings("unchecked")
  @Override
  public V remove(Object key)
  {
    K k = (K) key;
    LOGGER.finest("remove key=" + key);
    V value = get(key);
    if (value != null) {
      getCostHandler().modifyCostsOnRemove(this, k, value);
      marshal(k, value);
    }
    return super.remove(key);
  }

  @Override
  public V put(K key, V value)
  {
    LOGGER.finest("modifying costs on append: key=" + key + " value=" + value);
    getCostHandler().modifyCostsOnAppend(this, key, value);
    return super.put(key, value);
  }
  
  @Override
  public V get(Object key)
  {
    V value = super.get(key);
    if (value != null) return value;
    value = unmarshal(key);
    return value;
  }

  @SuppressWarnings("unchecked")
  protected V unmarshal(Object key) {
    V value;
    K k = (K) key;
    value = getSerializationHandler().unmarshal(k);
    if (value != null) {
      put(k, value);
    }
    return value;
  }

  public CacheCostHandler<K, V> getCacheCostHandler() {
    return getCostHandler();
  }

  public CacheSerializationHandler<K, V> getSerializationHandler() {
    return serializationHandler;
  }

  public void setCostHandler(CacheCostHandler<K, V> costHandler) {
    if (costHandler == null) {
      this.costHandler = new EmptyCacheCostHandler<K, V>();
    } else {
      this.costHandler = costHandler;
    }
  }

  public CacheCostHandler<K, V> getCostHandler() {
    return costHandler;
  }

  public void setSerializationHandler(CacheSerializationHandler<K, V> serializationHandler) {
    if (serializationHandler == null) {
      this.serializationHandler = new EmptyCacheSerializationHandler<K, V>();
    } else {
      this.serializationHandler = serializationHandler;
    }
  }
}
