/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.util.concurrent;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.concurrent.Future;
import java.util.logging.Level;
import java.util.logging.Logger;

import dvi.util.DviUtils;


public class CachedComputer<K, V> implements Computer<K, V>
{
  private final class LinkedHashMapExtension extends
      LinkedHashMap<K, CacheEntry<K, V>> {
    private static final long serialVersionUID = 1L;

    @Override
    protected boolean removeEldestEntry(Map.Entry<K,CacheEntry<K, V>> eldest)
    {
      boolean doRemove = CachedComputer.this.removeEldestEntry(eldest);
      LOGGER.finer("doRemove = " + doRemove);
      return doRemove;
    }
  }

  private static final Logger LOGGER = Logger.getLogger(CachedComputer.class.getName());
  private final Map<K, CacheEntry<K, V>> map;
  private final Computer<K, V> computer;
  
  public CachedComputer(final Computer<K, V> computer)
  {
    this.computer = computer;
    map = Collections.synchronizedMap(createCache());
  }
  
  protected Map<K, CacheEntry<K, V>> createCache()
  {
    return new LinkedHashMapExtension();
  }
  
  protected boolean removeEldestEntry(Map.Entry<K, CacheEntry<K, V>> entry)
  {
    return false;
  }
  
  protected synchronized Future<V> startComputation(final Computation<K, V> computable)
  {
    final Future<V> future = computer.compute(new Computation<K, V>() {
      public V call() throws Exception {
        try {
          return computable.call();
        } catch (Throwable e) {
          // We would like to associate any exception with the return value of null.
          DviUtils.logStackTrace(LOGGER, Level.SEVERE, e);
          return null;
        }
      }

      public K getCacheKey() {
        return computable.getCacheKey();
      }
      
    });
    return future;
  }

  /* (non-Javadoc)
   * @see Computer#compute(Computation, java.util.concurrent.ExecutorService)
   */
  public synchronized Future<V> compute(Computation<K, V> computable)
  {
    K key = computable.getCacheKey();
    if (key == null) {
      LOGGER.finer("Key is null: " + computable);
      final Future<V> future = startComputation(computable);
      return future;
    }
    CacheEntry<K, V> cacheEntry = map.get(key);
    if (cacheEntry == null) {
      LOGGER.finer("Cache miss: " + key);
      final Future<V> future = startComputation(computable);
      cacheEntry = new CacheEntry<K, V>(this, computable, future);
      getCache().put(key, cacheEntry);
      LOGGER.finer("Wrote cache: key=" + key + " value=" + future);
    } else {
      LOGGER.finest("Cache hit: " + key);
    }
    return cacheEntry.getFuture();
  }

  public synchronized Future<V> getCachedResult(Computation<K, V> computable)
  {
    K key = computable.getCacheKey();
    if (key == null) {
      return null;
    }
    CacheEntry<K, V> cacheEntry = map.get(key);
    if (cacheEntry == null) {
      return null;
    } else {
      return cacheEntry.getFuture();
    }
  }
  
  public Map<K, CacheEntry<K, V>> getCache()
  {
    return map;
  }
}
