/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.render;

import java.util.Stack;
import java.util.logging.Logger;

import dvi.DviException;
import dvi.DviFontSpec;
import dvi.DviFontTable;
import dvi.DviObject;
import dvi.DviRegister;
import dvi.api.DevicePainter;
import dvi.api.DviContextSupport;
import dvi.api.DviExecutorContext;
import dvi.api.Geometer;
import dvi.api.GeometerContext;
import dvi.api.SimpleMetrics;
import dvi.cmd.DviBop;
import dvi.cmd.DviPostPost;
import dvi.cmd.DviPostamble;
import dvi.cmd.DviPreamble;


public class BasicGeometer
extends DviObject
implements Geometer
{
  private static final Logger LOGGER = Logger.getLogger(BasicGeometer.class.getName());
  
  public BasicGeometer(DviContextSupport dcs)
  {
    super(dcs);
  }
  
  protected DevicePainter dp;
  public void setPainter(DevicePainter dp) { this.dp = dp; }

  private static final class GeometerContextImpl
  implements GeometerContext
  {
    private DviFontSpec fs = null;
    private SimpleMetrics sm = null;
    private final Stack<DviRegister> stack = new Stack<DviRegister>();
    private DviRegister reg;
    private DviExecutorContext exe_ctx;

    public DviFontSpec currentFontSpec() { return fs; }
    public DviRegister getRegister() { return reg; }
    public DviExecutorContext getExecuterContext() { return exe_ctx; }
  }

  private GeometerContextImpl ctx = null;
  private DviFontTable ft = null;

  public void begin(DviExecutorContext ctx)
  throws DviException
  {
    this.ctx = new GeometerContextImpl();
    this.ctx.reg = new DviRegister();
    this.ctx.exe_ctx = ctx;
    ft = ctx.getData().getFontTable();

    dp.begin(this.ctx);
  }

  public void end()
  throws DviException
  {
    if (ctx.fs != null) {
      dp.endFont();
    }
    dp.end();
    ctx = null;
    ft = null;
  }


  public void doSet(int code)
  throws DviException
  {
    dp.drawChar(code);

    int tfmw = 0;
    if (ctx.sm != null) {
      tfmw = ctx.sm.getTfmWidth(code);
    } else {
    }
    if (ctx.fs != null) {
      ctx.reg.addH(ctx.fs.tfmToDvi(tfmw));
    }
  }

  public void doSetRule(int width, int height)
  throws DviException
  {
    dp.drawRule(width, height);
    ctx.reg.addH(width);
  }

  public void doPut(int code) throws DviException {
    dp.drawChar(code);
  }

  public void doPutRule(int w, int h) throws DviException {
    dp.drawRule(w, h);
  }

  public void doNop() throws DviException {
  }

  public void doSelectFont(int fn) throws DviException {
    if (ctx.fs != null)
      dp.endFont();

    ctx.fs = ft.get(fn);
    ctx.sm = getDviContext().findDviSimpleMetrics(ctx.fs);
    if (ctx.sm == null) {
      LOGGER.fine("No metric for font number " + fn + " font spec: " + ctx.fs);
    }
    dp.beginFont(ctx.fs);
  }
  public void doDefineFont(int fn, DviFontSpec fs) throws DviException {
    /* ignored. */
  }

  public void doPush() throws DviException {
    ctx.stack.push((DviRegister) ctx.reg.clone());
  }
  public void doPop() throws DviException {
    if (ctx.stack.empty())
      throw new DviException("stack underflow.");

    ctx.reg.copy(ctx.stack.pop());
  }

  public void doPre(DviPreamble preamble) throws DviException {
  }
  public void doBop(DviBop bop) throws DviException {
    ctx.reg.reset();
    ctx.stack.clear();
    dp.beginPage(bop);
  }
  public void doEop() throws DviException {
    dp.endPage();
  }
  public void doPost(DviPostamble postamble) throws DviException {
  }
  public void doPostPost(DviPostPost postPost) throws DviException {
    ctx.exe_ctx.setTerminate(true);
  }

  public void doRight(int by) throws DviException {
    ctx.reg.addH(by);
  }
  public void doW(int by) throws DviException {
    ctx.reg.setW(by);
    ctx.reg.addH(by);
  }
  public void doW0() throws DviException {
    ctx.reg.addH(ctx.reg.getW());
  }
  public void doX(int by) throws DviException {
    ctx.reg.setX(by);
    ctx.reg.addH(by);
  }
  public void doX0() throws DviException {
    ctx.reg.addH(ctx.reg.getX());
  }

  public void doDown(int by) throws DviException {
    ctx.reg.addV(by);
  }
  public void doY(int by) throws DviException {
    ctx.reg.setY(by);
    ctx.reg.addV(by);
  }
  public void doY0() throws DviException {
    ctx.reg.addV(ctx.reg.getY());
  }
  public void doZ(int by) throws DviException {
    ctx.reg.setZ(by);
    ctx.reg.addV(by);
  }
  
  public void doZ0() throws DviException {
    ctx.reg.addV(ctx.reg.getZ());
  }

  public void doSpecial(byte [] xxx) throws DviException {
    dp.drawSpecial(xxx);
  }
}
