/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.cmd;

/**
 * An immutable class to represent the BOP (Begin Of Page) command.
 * This class is used internally to parse the DVI data.
 * @author Takeyuki NAGAO (nagaotakeyuki@gmail.com)
 */

public class DviBop
implements java.io.Serializable
{
  private static final long serialVersionUID = 1326126283946043078L;
  
  private final int count[];
  private final int backPointer;

  public DviBop(int [] count, int backPointer) {
    this.count = count.clone(); // a defensive copy
    this.backPointer = backPointer;
    checkValues();
  }

  public void checkValues() {
    if (count.length != 10)
      throw new IllegalArgumentException
        ("count must be of type int [10].");
  }

  public String countAsString() {
    int i, j;

    StringBuilder sb = new StringBuilder();
    sb.append(String.valueOf(count[0]));
    for (i=count.length-1; i>0; i--) 
      if (count[i] != 0) break;
    for (j=1; j<=i; j++)
      sb.append(":" + count[j]);
    
    return sb.toString();
  }

  public int [] count() { return count.clone(); }
  public int backPointer() { return backPointer; }

  public String toString() {
    return getClass().getName() + "[count=" + countAsString() +
           ",backPointer=" + backPointer + "]";
  }

  public boolean equals(Object o) {
    if (o instanceof DviBop) {
      DviBop a = (DviBop) o;
      if (!(a.backPointer == backPointer
          && a.count.length == count.length
          && count.length == 10))
        return false;

      for (int i=0; i<10; i++)
        if (a.count[i] != count[i])
          return false;

      return true;
    }
    return false;
  }

  public int hashCode() {
    int hash = backPointer;
    for (int a : count)
      hash = a + 33 * hash;
    return hash;
  }
}
