/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.util.progress;

import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

import jp.sourceforge.dvibrowser.dvicore.api.DviContextSupport;


public class ProgressReporter
extends AbstractProgressModel
implements ProgressListener
{
  private final List<ProgressRecorder> recorders = new ArrayList<ProgressRecorder>();
  
  public ProgressReporter(DviContextSupport dcs)
  {
    super(dcs);
  }
  
  public AbstractProgressModel [] getProgressRecorders()
  {
    return recorders.toArray(new ProgressRecorder[recorders.size()]);
  }

  public void addProgressRecorder(ProgressRecorder recorder)
  {
    if (recorder == null) return;
    if (recorders.contains(recorder)) return;
    recorder.addProgressListener(this);
    recorders.add(recorder);
  }

  public void removeProgressRecorder(AbstractProgressModel recorder)
  {
    if (recorder == null) return;
    if (!recorders.contains(recorder)) return;
    recorders.remove(recorder);
    recorder.removeProgressListener(this);
  }
  
  private volatile ProgressItem item;
  public ProgressItem getMostRecentItem()
  {
    return item;
  }
  
  protected void update()
  {
    ArrayList<ProgressItem> list = new ArrayList<ProgressItem>();
    for (ProgressRecorder recorder : recorders) {
      ProgressItem[] items = recorder.getItems();
      if (items.length != 0) {
        ProgressItem item = items[items.length - 1];
        list.add(item);
      }
    }
    Collections.sort(list, new Comparator<ProgressItem>() {
      public int compare(ProgressItem o1, ProgressItem o2)
      {
        long s1 = o1.getSerialNumber();
        long s2 = o2.getSerialNumber();
        if (s1 < s2) {
          return -1;
        } else if (s1 > s2) {
          return 1;
        } else {
          return 0;
        }
      }
    });
    if (list.size() > 0) {
      item = list.get(list.size() - 1);
    }
  }

  public void progressClose(ProgressEvent e)
  {
    update();
    fireProgressCloseEvent(e.getManagedItem());
  }

  public void progressOpen(ProgressEvent e)
  {
    update();
    fireProgressOpenEvent(e.getManagedItem());
  }

  public void progressUpdate(ProgressEvent e)
  {
    update();
    fireProgressUpdateEvent(e.getManagedItem());
  }
}
