/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.ctx;
import java.io.File;
import java.net.URL;
import java.security.AccessControlException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.logging.Logger;

import jp.sourceforge.dvibrowser.dvicore.DviObject;
import jp.sourceforge.dvibrowser.dvicore.api.DviContextSupport;
import jp.sourceforge.dvibrowser.dvicore.util.concurrent.Computation;
import jp.sourceforge.dvibrowser.dvicore.util.progress.ProgressItem;


public class FileLocationResolver
extends DviObject
implements Computation<String, Collection<URL>> {
  private static final Logger LOGGER = Logger.getLogger(FileLocationResolver.class
      .getName());
  private final String systemResourcePath;
  private final String filename;
  
  public FileLocationResolver(DviContextSupport dcs, String systemResourcePath, String filename)
  {
    super(dcs);
    this.filename = filename;
    this.systemResourcePath = systemResourcePath;
  }

  public Collection<URL> call() throws Exception
  {
    ProgressItem progress = getDviContext().getProgressRecorder().open("preparing " + filename);
    List<URL> list = new ArrayList<URL>();
    try {
      try {
        if ("true".equals(System.getProperty("dvi.ctx.DefaultDviContext.usePackageShareDir"))) {
          File f = new File("share", filename);
          if (f.exists() && f.canRead()) {
            LOGGER.fine("Using resource from share: " + f);
            list.add(f.toURL());
            return list;
          }
        }
      } catch (AccessControlException ex) {
          LOGGER.warning(ex.toString());
      }
      
      LOGGER.finer("running kpsewhich: " + filename);
      try {
        URL url = null;
        KpseWhich kpseWhich = new KpseWhich(this);
        url = kpseWhich.findURL(filename, true);
        LOGGER.finer("kpsewhich result: " + filename + " => " + url);
        if (url != null)
          list.add(url);
      } catch (RuntimeException ex) {
          LOGGER.warning(ex.toString());
      }
      
      try {
        URL u = ClassLoader.getSystemResource(systemResourcePath + "/"
            + filename);
        LOGGER.fine("system resource by classloader: " + filename + " => " + u);
        if (u != null)
          list.add(u);
      } catch (RuntimeException ex) {
          LOGGER.warning(ex.toString());
      }
      
      try {
          URL u = getClass().getResource(systemResourcePath + "/"
              + filename);
          LOGGER.fine("resource by classloader: " + filename + " => " + u);
          if (u != null)
            list.add(u);
        } catch (RuntimeException ex) {
            LOGGER.warning(ex.toString());
        }

      
      if (list.size() == 0) {
          LOGGER.warning("Failed to resolve file: " + filename);
      }
    } finally {
      progress.close();
    }
    
    return list;
  }
  
  public String getFilename()
  {
    return filename;
  }
  
  public String getCacheKey()
  {
    return filename;
  }

  public String getSystemResourcePath()
  {
    return systemResourcePath;
  }
}
