/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.util.concurrent;
import java.util.concurrent.Future;
import java.util.logging.Logger;

public class ComputerProgressMonitor<K, V> implements Computer<K,V> {
  private static final Logger LOGGER = Logger.getLogger(ComputerProgressMonitor.class.getName());
  private final Computer<K, V> computer;
  
  public ComputerProgressMonitor(Computer<K, V> computer)
  {
    this.computer = computer;
  }
  
  public Future<V> compute(final Computation<K, V> computation)
  {
    final Computation<K, V> wrapper = new Computation<K, V>() {
      public V call() throws Exception
      {
        fireComputationBeginEvent(computation);
        try {
          V v = computation.call();
          return v;
        } catch (Exception e) {
          throw e;
        } finally {
          fireComputationEndEvent(computation);
        }
      }

      public K getCacheKey()
      {
        return computation.getCacheKey();
      }
      
    };
    return computer.compute(wrapper);
  }

  protected void fireComputationEndEvent(Computation<K, V> computation)
  {
    LOGGER.finer("computation ended: " + computation);
  }

  protected void fireComputationBeginEvent(Computation<K, V> computation)
  {
    LOGGER.finer("computation started: " + computation);
  }

  public Computer<K, V> getComputer()
  {
    return computer;
  }
}
