/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package dvi.gs;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Scanner;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import dvi.DviRect;
import dvi.util.CommandShellHandler;
import dvi.util.DviUtils;
public class GhostscriptBBOXParser implements CommandShellHandler {
  private static final Logger LOGGER = Logger
      .getLogger(GhostscriptBBOXParser.class.getName());

  private final ArrayList<DviRect> list = new ArrayList<DviRect>();
  
  private static final Pattern patPostscriptBoundingBox = Pattern.compile
  ("%%BoundingBox:\\s*([+-]?\\d\\d*)\\s\\s*([+-]?\\d\\d*)\\s\\s*([+-]?\\d\\d*)\\s\\s*([+-]?\\d\\d*)\\s*");
  
  public static DviRect parsePostscriptBoundingBox(String line)
  {
    Matcher mat = patPostscriptBoundingBox.matcher(line);
    if (mat.matches()) {
      int llx = Integer.parseInt(mat.group(1));
      int lly = Integer.parseInt(mat.group(2));
      int urx = Integer.parseInt(mat.group(3));
      int ury = Integer.parseInt(mat.group(4));
      return DviRect.getDviRect(llx, lly, urx, ury);
    }
    return null;
  }
  
  public GhostscriptBBOXParser()
  {
  }
  
  public int getTotalPages() {
    return list.size();
  }
  
  public DviRect getBoundingBoxOfPage(int page)
  {
    return list.get(page);
  }

  public void handleStderr(InputStream in) throws IOException {
    try {
      Scanner s = new Scanner(in);
      while (s.hasNext()) {
        String line = s.nextLine();
        DviRect bbox = parsePostscriptBoundingBox(line);
        if (bbox != null) {
          list.add(bbox);
        }
      }
    } finally {
      DviUtils.silentClose(in);
    }
  }

  public void handleStdin(OutputStream out) throws IOException {
    out.close();
  }

  public void handleStdout(InputStream in) throws IOException {
    in.close();
  }
}
