/*
 * Copyright (c) 2009, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvicore.special;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.util.logging.Level;
import java.util.logging.Logger;

import jp.sourceforge.dvibrowser.dvicore.DviException;
import jp.sourceforge.dvibrowser.dvicore.DviObject;
import jp.sourceforge.dvibrowser.dvicore.DviResolution;
import jp.sourceforge.dvibrowser.dvicore.DviSize;
import jp.sourceforge.dvibrowser.dvicore.api.DviContextSupport;
import jp.sourceforge.dvibrowser.dvicore.api.DviDocument;
import jp.sourceforge.dvibrowser.dvicore.api.DviPage;
import jp.sourceforge.dvibrowser.dvicore.api.HasURL;
import jp.sourceforge.dvibrowser.dvicore.ctx.DviToolkit;
import jp.sourceforge.dvibrowser.dvicore.gs.GhostscriptUtils;
import jp.sourceforge.dvibrowser.dvicore.gui.swing.ViewSpec;
import jp.sourceforge.dvibrowser.dvicore.image.split.ImageFileConfig;
import jp.sourceforge.dvibrowser.dvicore.image.split.SplitImage;
import jp.sourceforge.dvibrowser.dvicore.image.split.ZipSplitImageReader;
import jp.sourceforge.dvibrowser.dvicore.image.split.ZipSplitImageWriter;
import jp.sourceforge.dvibrowser.dvicore.util.DviUtils;
import jp.sourceforge.dvibrowser.dvicore.util.ZipBuilder;
import jp.sourceforge.dvibrowser.dvicore.util.concurrent.Computation;


public class EPS2SplitImagePreparator
extends DviObject
implements Computation<String, SplitImage> {
  private static final Logger LOGGER = Logger
      .getLogger(EPS2SplitImagePreparator.class.getName());

  private final DviPage page;
  private final ViewSpec viewSpec;
  private final String postScriptData;
  private final String hash;

  public EPS2SplitImagePreparator(DviContextSupport dcs, DviPage page, ViewSpec viewSpec)
  throws DviException
  {
    super(dcs);
    this.page = page;
    this.viewSpec = viewSpec;
    DviToolkit utils = getDviContext().getDviToolkit();
    this.postScriptData = utils.getEmbeddedPostScript(page, viewSpec);
    this.hash = DviUtils.md5Hex(postScriptData);
  }

  public SplitImage call() throws Exception
  {
    if (postScriptData == null) return null;
    
    File dir = getDviContext().getTemporaryDirectory();
    String key = getCacheKey();
    String keyHash = DviUtils.md5Hex(key);
    String basename = String.format("gs--%s--%08d", keyHash, page.getPageNumber());
    String imgExt = ".zip";
    // TODO: outsource the resolution configuration.
    DviResolution res = new DviResolution(400, 4);
    DviSize unit = new DviSize(256, 256);
    File imgFile = new File(dir, basename + imgExt);
    imgFile.deleteOnExit();
    if (!imgFile.exists()) {
      LOGGER.fine("Now creating postscript image for page " + page);
      LOGGER.fine("Using key=" + key);
      LOGGER.fine("Using keyHash=" + keyHash);
      LOGGER.finer("PostScriptData= " + postScriptData);
      try {
        File tmpFile = File.createTempFile("gs-temp", null);
        tmpFile.deleteOnExit();
        try {
          final FileOutputStream fos = new FileOutputStream(tmpFile);
          final ZipBuilder zb = new ZipBuilder(fos);
          // TODO: outsource the encoding configuration.
          final InputStream is = new ByteArrayInputStream(postScriptData.getBytes("UTF-8")); 
          try {
            final ZipSplitImageWriter imageWriter = new ZipSplitImageWriter
              (basename, ImageFileConfig.PNG, res, zb);
            // Remark that Ghostscript counts pages from 1 not 0.
            GhostscriptUtils.renderAndSplit(getDviContext(), is, unit, imageWriter, res, "pnmraw", 1, 0, null);
          } finally {
            DviUtils.silentClose(fos);
            DviUtils.silentClose(is);
          }
          // TODO: lock before renaming to avoid race condition against other thread.
          if (!tmpFile.renameTo(imgFile)) {
            throw new DviException("Failed to create cache file: " + imgFile);
          }
        } finally {
          tmpFile.delete();
        }
      } catch (FileNotFoundException e) {
        DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
        throw new DviException(e);
      } catch (IOException e) {
        DviUtils.logStackTrace(LOGGER, Level.WARNING, e);
        throw new DviException(e);
      }
    }
    ZipSplitImageReader imageReader = new ZipSplitImageReader(imgFile);
    return imageReader.getSplitImage();
  }
  
  public String getCacheKey()
  {
    // TODO: use viewSpec.
    String pageKey = null;
    try {
      DviDocument doc = page.getDocument();
      if (doc instanceof HasURL) {
        URL url = ((HasURL) doc).getURL();
        pageKey = url.toExternalForm() + "#page" + page.getPageNumber();
      }
    } catch (DviException e) {
      LOGGER.warning(e.toString());
      pageKey = page.getCacheKey();
    }
    return getClass().getName() + "--" + hash + "-" + pageKey;
  }

  public ViewSpec getViewSpec()
  {
    return viewSpec;
  }

  public DviPage getPage()
  {
    return page;
  }
}
