/*
 * Copyright (c) 2012, Takeyuki Nagao
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or
 * without modification, are permitted provided that the
 * following conditions are met:
 * 
 *  * Redistributions of source code must retain the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer.
 *  * Redistributions in binary form must reproduce the above
 *    copyright notice, this list of conditions and the
 *    following disclaimer in the documentation and/or other
 *    materials provided with the distribution.
 *    
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING,
 * BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
 * USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY
 * OF SUCH DAMAGE.
 */

package jp.sourceforge.dvibrowser.dvi2epub.cmd;

import java.io.PrintWriter;

import jp.sourceforge.dvibrowser.dvi2epub.opt.BooleanValueOption;
import jp.sourceforge.dvibrowser.dvi2epub.opt.DefaultOptionMapper;

public abstract class AnnotatedCommand extends AbstractCommand {
	private CommandLineParser parser;
	private boolean wantHelp;
	private String[] args;

	public AnnotatedCommand() throws CommandException {
		parser = new AnnotatedCommandLineParser(this);
	}

	@BooleanValueOption(shortName = "h", longName = "help", description = "Show this information")
	public void wantHelp(boolean want) {
		this.wantHelp = want;
	}

	public boolean wantHelp() {
		return wantHelp;
	}

	@Override
	public int execute(String[] args) throws CommandException {
		try {
			parser.parse(args);
			int ret = doProcessCommandLine();
			return ret;
		} finally {
			PrintWriter out = getWriter();
			if (out != null) {
				out.flush();
			}
		}
	}

	protected int doProcessCommandLine()
			throws CommandException {
		if (wantHelp()) {
			showUsage();
			return Command.EXIT_SUCCESS;
		} else {
			return processCommandLine();
		}
	}

	protected abstract int processCommandLine() throws CommandException;

	@Override
	public void showUsage(PrintWriter pw) throws CommandException {
		parser.printHelp(pw);
		pw.flush();
	}

	@Override
	public String getApplicationName() throws CommandException {
		String name = System.getProperty("application.name", "java " + getClass().getName());
		return name;
	}

	public String[] getArgs() {
		return args;
	}

	public void setArgs(String[] args) {
		this.args = args;
	}
	
	protected OptionMapper createOptionMapper() {
		return new DefaultOptionMapper();
	}
	
	protected OptionPrinter createOptionPrinter(PrintWriter writer) {
		return new OptionPrinter(writer);
	}
}
