﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using System.Threading;
using System.IO;

namespace ds8617
{
    public partial class Form1 : Form
    {
        Bitmap bitmap;

        class Ds8617Property
        {
            public float tmdv = 1;
            public float vdiv = 1;
            public float afrq = 1;

            public string cmod;
            public float emval1, emval2;
            public float vcur1, vcur2;
            public float tcur1, tcur2;
        }

        Ds8617Property ds8617property = new Ds8617Property();

        public Form1()
        {
            InitializeComponent();

            bitmap = new Bitmap(640, 480);
            pictureBox1.Image = bitmap;
        }

        /// <summary>
        /// 数値をリードアウト表示用の文字列に変換する
        /// </summary>
        /// <param name="f"></param>
        /// <returns></returns>
        private string ReadoutFloat(double f, int count)
        {
            // k, m, u などを使い
            // 0.xxx とならないようにする

            string unit = "";

            if (f >= 1000000)
            {
                unit = "M";
                f /= 1000000;
            }
            else if (f >= 1000)
            {
                unit = "k";
                f /= 1000;
            }
            else if (f < 0.000001)
            {
                unit = "n";
                f *= 1000000000;
            }
            else if (f < 0.001)
            {
                unit = "μ";
                f *= 1000000;
            }
            else if (f < 1)
            {
                unit = "m";
                f *= 1000;
            }

            string s = f.ToString("f4");
            if (s.Length > count && s[count] == '.')
                s = s.Substring(0, count);
            else
                s = s.Substring(0, count + 1);

            // 適当
            if (count == 3)
            {
                if (s.Length > 2 && s.Substring(1) == ".00")
                    s = s.Substring(0, 1);
                if (s.Length > 3 && s.Substring(2) == ".0")
                    s = s.Substring(0, 2);
            }

            s += unit;

            return s;
        }

        private void CaptureScreen(object sender, EventArgs e)
        {
            messageBox.Clear();

            Gpib gpib = new Gpib(axIfxGpibAdv1);

            {
                string[] commands = { "DATE?", "TIME?", "*IDN?" };
                query(commands);
            }

            updateProperty();
            captureScreen();

            gpib.Dispose();
        }

        private void updateProperty()
        {
            string[] commands = {
                    "VDIV?", "TMDV?", "AFRQ?",
                    "CMOD?", "EMVAL?", "VCUR?", "TCUR?",
                    "VCPL?", "VVAR?", "BNDW?", "SWMD?", "VPOS?" };
            string[] res = query(commands);
            if (res != null)
            {
                ds8617property.vdiv = (float)Gpib.AnalyzeResponseFloat(res[0]);
                ds8617property.tmdv = (float)Gpib.AnalyzeResponseFloat(res[1]);
                ds8617property.afrq = (float)Gpib.AnalyzeResponseFloat(res[2]);

                ds8617property.cmod = res[3];

                {
                    string[] split = res[4].Split(',');
                    if (split.Length == 2)
                    {
                        ds8617property.emval1 = (float)Gpib.AnalyzeResponseFloat(split[0]);
                        ds8617property.emval2 = (float)Gpib.AnalyzeResponseFloat(split[1]);
                    }
                }

                {
                    string[] split = res[5].Split(',');
                    if (split.Length == 2)
                    {
                        ds8617property.vcur1 = (float)Gpib.AnalyzeResponseFloat(split[0]);
                        ds8617property.vcur2 = (float)Gpib.AnalyzeResponseFloat(split[1]);
                    }
                }

                {
                    string[] split = res[6].Split(',');
                    if (split.Length == 2)
                    {
                        ds8617property.tcur1 = (float)Gpib.AnalyzeResponseFloat(split[0]);
                        ds8617property.tcur2 = (float)Gpib.AnalyzeResponseFloat(split[1]);
                    }
                }
            }
        }

        private string[] query(string[] commands)
        {
            string[] resString = new string[commands.Length];
            int result;

            int p = 0;
            foreach (string command in commands)
            {
                messageBox.Text += command + " : ";
                result = axIfxGpibAdv1.MasterSend("DS8617", command);
                if (result < 0)
                {
                    messageBox.Text += "SEND ERROR:" + result + "\r\n";
                    return null;
                }

                int recvlen = 1024;
                result = axIfxGpibAdv1.MasterReceive("DS8617", "", ref recvlen);
                if (result < 0)
                {
                    messageBox.Text += "RECV ERROR:" + result + ":" + recvlen + "\r\n";
                    return null;
                }

                if (recvlen > 0)
                {
                    object vtdata;
                    vtdata = "";
                    result = axIfxGpibAdv1.GetData(ref vtdata, recvlen);
                    if (result < 0)
                    {
                        messageBox.Text += "DATA ERROR:" + result + ":" + recvlen + ":" + (string)vtdata + "\r\n";
                        return null;
                    }

                    messageBox.Text += (string)vtdata + "\r\n";
                    resString[p++] = (string)vtdata;
                }
                else
                {
                    resString[p++] = "";
                }
            }

            return resString;
        }

        private void captureScreen()
        {
            string[] commands = { "MEMR CH1,1,0,1000" };

            int result;

            Graphics g = Graphics.FromImage(pictureBox1.Image);
            InitScreen(g);

            writeScreenProperty(g);

            foreach (string command in commands)
            {
                messageBox.Text += command + " : ";
                result = axIfxGpibAdv1.MasterSend("DS8617", command);
                if (result < 0)
                {
                    messageBox.Text += "SEND ERROR:" + result + "\r\n";
                    return;
                }

                int recvlen = 1024;
                result = axIfxGpibAdv1.MasterReceive("DS8617", "", ref recvlen);
                if (result < 0)
                {
                    messageBox.Text += "RECV ERROR:" + result + ":" + recvlen + "\r\n";
                    return;
                }

                if (recvlen > 0)
                {
                    object vtdata;
                    vtdata = new byte[recvlen];
                    result = axIfxGpibAdv1.GetData(ref vtdata, recvlen);
                    if (result < 0)
                    {
                        messageBox.Text += "DATA ERROR:" + result + ":" + recvlen + ":" + (string)vtdata + "\r\n";
                        return;
                    }

                    int prev = -1;
                    int prevx = -1;
                    byte[] data = (byte[])vtdata;
                    for (int i = 0; i < recvlen && i < 10; ++i)
                    {
                        int s = data[i];
                        messageBox.Text += s.ToString("D") + "\r\n";
                    }
                    for (int i = 0; i < recvlen / 2; ++i)
                    {
                        int s = data[i * 2] * 0x100 + data[i * 2 + 1];
                        if (s >= 0x8000)
                        {
                            s = s - 0x10000;
                        }
                        s += 0x8000;

                        if (i == 0)
                            messageBox.Text += "FirstData" + s.ToString("D") + "\r\n";

                        int cur = 480 - (int)((float)s / 0x10000 * 480);
                        int x = (int)((float)i / 1000 * 640);
                        if (prevx >= 0)
                        {
                            g.DrawLine(Pens.LightGreen, prevx, prev, x, cur);
                        }
                        prev = cur;
                        prevx = x;
                    }
                }
            }
            pictureBox1.Refresh();

        }

        private void InitScreen(Graphics g)
        {
            g.Clear(Color.FromArgb(0, 20, 0));

            for (int x = 1; x < 10; ++x)
            {
                int xd = (int)((float)640 / 10 * x);
                g.DrawLine(Pens.Black, xd, 0, xd, 480);
            }

            for (int y = 1; y < 8; ++y)
            {
                int yd = (int)((float)480 / 8 * y);
                g.DrawLine(Pens.Black, 0, yd, 640, yd);
            }
        }

        private void writeScreenProperty(Graphics g)
        {
            Font readoutFont = new Font("Courier", 12, FontStyle.Regular);

            g.DrawString("A " + ReadoutFloat(ds8617property.tmdv, 3) + "s CH1",
                readoutFont, Brushes.LightGreen, 10, 0);

            g.DrawString("f=" + ReadoutFloat(ds8617property.afrq, 4) + "Hz",
                readoutFont, Brushes.LightGreen, 530, 444);

            g.DrawString("CH1 " + ReadoutFloat(ds8617property.vdiv, 3) + "V~",
                readoutFont, Brushes.LightGreen, 10, 460);

            if (ds8617property.cmod == "V")
            {
                int y1 = 480 - (int)((ds8617property.vcur1 + 4) / 8 * 480);
                int y2 = 480 - (int)((ds8617property.vcur2 + 4) / 8 * 480);

                {
                    Pen cursorLine = new Pen(Color.LightGreen);
                    cursorLine.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
                    float[] dashPattern = { 4, 4 };
                    cursorLine.DashPattern = dashPattern;

                    g.DrawLine(cursorLine, 640 / 10, y1, 640 / 10 * 9, y1);
                }

                {
                    Pen cursorLine = new Pen(Color.LightGreen);
                    cursorLine.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
                    float[] dashPattern = { 2, 2 };
                    cursorLine.DashPattern = dashPattern;

                    g.DrawLine(cursorLine, 640 / 10, y2, 640 / 10 * 9, y2);
                }

                g.DrawString("CH1 ΔV= " + ReadoutFloat(ds8617property.emval1, 3) + "V",
                    readoutFont, Brushes.LightGreen, 10, 444);
            }
        }

        private void SaveScreenPng(object sender, EventArgs e)
        {
            try
            {
                DateTime time = new DateTime(DateTime.Now.Ticks);
                string timestring = time.ToString("yyyyMMddhhmmss");

                FileStream fs = new FileStream("ds8617_" + timestring + ".png", FileMode.CreateNew);

                bitmap.Save(fs, System.Drawing.Imaging.ImageFormat.Png);

                fs.Close();
            }
            catch (Exception)
            {
                messageBox.Text += "Write Error\r\n";
            }
        }

    }
}
