﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using System;

namespace Sasa.QualityTools.DrFx.Core.Tests.Unit
{
    /// <summary>
    /// The unit test class for <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessGenerator"/>.
    /// </summary>
    [TestFixture]
    public class FxCopProcessGeneratorTest
    {
        /// <summary>
        /// Tests generated FxCopCmd arguments reflect the specified target files
        /// by <see cref=" Sasa.QualityTools.DrFx.Core.FxCopControllerConfiguration"/>.
        /// </summary>
        /// <param name="files">The analysis target files that was separated by camma.</param>
        /// <returns>The generated command string.</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("asm1.dll", Result = @"/c /o:""codeanalysisreport.xml"" /f:""asm1.dll"" /r:"".\Rules""")]
        [TestCase("asm1.dll,lib\\asm2.dll,C:\\asm3.dll", Result = @"/c /o:""codeanalysisreport.xml"" /f:""asm1.dll"" /f:""lib\asm2.dll"" /f:""C:\asm3.dll"" /r:"".\Rules""")]
        public string TestGeneratedArgumentsReflectsTargetFiles(string files)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string file in files.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.TargetFiles.Add(file);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }

        /// <summary>
        /// Tests generated FxCopCmd arguments reflect the specified output file
        /// by <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/>.
        /// </summary>
        [Test]
        public void TestGeneratedArgumentsReflectsOutputFile()
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            configuration.OutputFile = "result\\report.xml";
            Assert.AreEqual(@"/c /o:""result\report.xml"" /r:"".\Rules""", FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration));
        }

        /// <summary>
        /// Tests generated FxCopCmd arguments reflect the specified rule files
        /// by <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/>.
        /// </summary>
        /// <param name="files">The analysis rule files that was separated by camma.</param>
        /// <returns>The generated command string.</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("asm1.dll", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /r:""asm1.dll""")]
        [TestCase("asm1.dll,lib\\asm2.dll,C:\\asm3.dll", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /r:""asm1.dll"" /r:""lib\asm2.dll"" /r:""C:\asm3.dll""")]
        public string TestGeneratdArgumentsReflectsRuleFiles(string files)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string file in files.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.RuleFiles.Add(file);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }

        /// <summary>
        /// Tests generated FxCopCmd arguments reflect the specified dependency directories
        /// by <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/>.
        /// </summary>
        /// <param name="files">The dependency directories that was separated by camma.</param>
        /// <returns>The generated command string.</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("dir1", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /d:""dir1""")]
        [TestCase("dir1,lib\\dir2,C:\\dir3", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /d:""dir1"" /d:""lib\dir2"" /d:""C:\dir3""")]
        public string TestGeneratedArgumentsReflectsDependencyDirectories(string directories)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string directory in directories.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.DependencyDirectories.Add(directory);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }
    }
}
