﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using System;
using System.Collections;
using System.Collections.Generic;

namespace Sasa.QualityTools.DrFx.Console
{
    /// <summary>
    /// ロングオプション定義のコレクションを表します。
    /// </summary>
    internal class LongOptionCollection : IEnumerable<string>
    {
        /// <summary>
        /// ロングオプションとオプション定義のマッピング。
        /// </summary>
        private Dictionary<string, Option> options;



        /// <summary>
        /// 新しいインスタンスを初期化します。
        /// </summary>
        internal LongOptionCollection()
        {
            this.options = new Dictionary<string, Option>();
        }



        /// <summary>
        /// コレクションに新しいオプション定義を追加します。
        /// </summary>
        /// <param name="definition">ロングオプションの定義 (プレフィックス "--" は不要)。</param>
        /// <param name="symbol">オプションを表すシンボル。</param>
        /// <param name="hasArgument">オプションが引数を持つかどうか。</param>
        /// <returns>オプション定義追加後のオプションコレクション。</returns>
        internal LongOptionCollection Add(string definition, char symbol, bool hasArgument)
        {
            this.options.Add("--" + definition, new Option(symbol, hasArgument));
            return this;
        }

        /// <summary>
        /// 指定されたロングオプションを表す文字列に対応するオプション定義を取得します。
        /// </summary>
        /// <param name="longOption">ロングオプションを表す文字列 (プレフィックス "--" が必要)。</param>
        /// <returns>指定されたロングオプションに対応するオプション定義。</returns>
        internal Option GetOption(string longOption)
        {
            if (this.options.ContainsKey(longOption) == false)
            {
                throw new UndefinedOptionException("指定されたオプションは定義されていないため、認識できません。", longOption);
            }
            return this.options[longOption];
        }

        /// <summary>
        /// ロングオプション文字列を反復処理するための列挙子を取得します。
        /// </summary>
        /// <returns>反復処理のための列挙子。</returns>
        public IEnumerator<string> GetEnumerator()
        {
            return this.options.Keys.GetEnumerator();
        }

        /// <summary>
        /// ロングオプション文字列を反復処理するための列挙子を取得します。
        /// </summary>
        /// <returns>反復処理のための列挙子。</returns>
        IEnumerator IEnumerable.GetEnumerator()
        {
            return this.options.Keys.GetEnumerator();
        }
    }
}
