﻿/*
 * DrFx - FxCop Report Translator and Visualizer.
 * Copyright (C) 2010 Sasa Yuan
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */



using NUnit.Framework;
using System;

namespace Sasa.QualityTools.DrFx.Core.Tests.Unit
{
    /// <summary>
    /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessGenerator"/> のテストクラスです。.
    /// </summary>
    [TestFixture]
    public class FxCopProcessGeneratorTest
    {
        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// 分析対象アセンブリが FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <param name="files">分析対象アセンブリファイル名 (カンマ区切り)。</param>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("asm1.dll", Result = @"/c /o:""codeanalysisreport.xml"" /f:""asm1.dll"" /r:"".\Rules""")]
        [TestCase("asm1.dll,lib\\asm2.dll,C:\\asm3.dll", Result = @"/c /o:""codeanalysisreport.xml"" /f:""asm1.dll"" /f:""lib\asm2.dll"" /f:""C:\asm3.dll"" /r:"".\Rules""")]
        public string TestGeneratedArgumentsReflectsTargetFiles(string files)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string file in files.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.TargetFiles.Add(file);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }

        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// レポート出力先が FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [Test]
        public void TestGeneratedArgumentsReflectsOutputFile()
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            configuration.OutputFile = "result\\report.xml";
            Assert.AreEqual(@"/c /o:""result\report.xml"" /r:"".\Rules""", FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration));
        }

        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// カスタムルールアセンブリが FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <param name="files">カスタムルールアセンブリファイル名 (カンマ区切り)。</param>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("asm1.dll", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /r:""asm1.dll""")]
        [TestCase("asm1.dll,lib\\asm2.dll,C:\\asm3.dll", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /r:""asm1.dll"" /r:""lib\asm2.dll"" /r:""C:\asm3.dll""")]
        public string TestGeneratdArgumentsReflectsRuleFiles(string files)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string file in files.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.RuleFiles.Add(file);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }

        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// 依存先アセンブリ検索ディレクトリが FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <param name="directories">依存先アセンブリ検索ディレクトリ名 (カンマ区切り)。</param>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [TestCase("", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase("dir1", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /d:""dir1""")]
        [TestCase("dir1,lib\\dir2,C:\\dir3", Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /d:""dir1"" /d:""lib\dir2"" /d:""C:\dir3""")]
        public string TestGeneratedArgumentsReflectsDependencyDirectories(string directories)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            foreach (string directory in directories.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries))
            {
                configuration.DependencyDirectories.Add(directory);
            }
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }

        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// カスタムディクショナリファイルが FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [Test]
        public void TestGeneratedArgumentsReflectsDictionaryFile()
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            configuration.DictionaryFile = "resources\\dictionary.xml";
            Assert.AreEqual(@"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /dic:""resources\dictionary.xml""", FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration));
        }

        /// <summary>
        /// <see cref=" Sasa.QualityTools.DrFx.Core.FxCopProcessConfiguration"/> に指定された
        /// GAC 検索の有効／無効が FxCopCmd の引数に反映されることをテストします。
        /// </summary>
        /// <param name="enableSearchGac"> GAC 検索が有効かどうか。</param>
        /// <returns>生成された FxCopCmd 実行コマンド文字列。</returns>
        [TestCase(false, Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules""")]
        [TestCase(true, Result = @"/c /o:""codeanalysisreport.xml"" /r:"".\Rules"" /gac")]
        public string TestGeneratedArgumentsReflectsEnableSearchGac(bool enableSearchGac)
        {
            FxCopProcessConfiguration configuration = new FxCopProcessConfiguration();
            configuration.FxCopInstallDirectory = "";
            configuration.EnableSearchGac = enableSearchGac;
            return FxCopProcessGenerator.GenerateFxCopCmdArguments(configuration);
        }
    }
}
